/* -*- C++ -*-
 *            kPPP: A pppd front end for the KDE project
 *
 * $Id: ruleset.h 1157648 2010-07-31 19:48:06Z tpearson $
 *
 *            Copyright (C) 1997 Bernd Johannes Wuebben
 *                   wuebben@math.cornell.edu
 *
 * This file was contributed by Mario Weilguni <mweilguni@sime.com>
 * Thanks Mario !
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this program; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#ifndef __RULESET__H__
#define __RULESET__H__

#include <tqmemarray.h>
#include <tqdatetime.h>
#include <tqstring.h>

// this structure is used to save
// accounting rules
typedef struct RULE {
  int type;
  double costs;
  double len;
  double after;
  TQTime from, until;
  struct {
    TQDate from, until;
  } date;
  struct {
    int from, until;
  } weekday;
};

// this class is used for loading and parsing of rules
class RuleSet {
public:
  /// create an empty rule
  RuleSet();

  /// gcc needs a destructor (otherwise compiler error)
  ~RuleSet() {};

  /// returns the name of the ruleset
  TQString name() const;

  /** Load a ruleset from a file. If an error occurs,
   *  returns the linenumber the error was in,
   *  otherwise 0. If the file could not be opened,
   *  returns -1
   */
  int load(const TQString &filename);

  /// returns the currency symbol
  TQString currencySymbol() const;

  /** returns a string representation of the
   *  of a doubleingpoint number using the
   *  currency-settings
   */
  TQString currencyString(double val) const;

  /// sets the start time -- must be called when the connection has bee established
  void setStartTime(TQDateTime dt);

  /// returns the "per-connection" costs
  double perConnectionCosts() const;

  /** returns the minimum number of costs (some
   *  phony companies have this
   */
  double minimumCosts() const;

  /// returns the currently valid rule settings
  void getActiveRule(TQDateTime dt, double connect_time, double &costs, double &len);

  /// checks if a rulefile is ok (no parse errors...)
  static int checkRuleFile(const TQString &);

protected:
  /** converts an english name of a day to integer,
   * beginning with monday=0 .. sunday=6
   */
  int dayNameToInt(const char *s);

  /// returns the date of easter-sunday for a year
  static TQDate get_easter(int year);

  /// add a rule to this ruleset
  void addRule(RULE r);

  /// parses on entry of the "on(...)" fields
  bool parseEntry(RULE &ret, TQString s, int year);

  /// parses the "on(...)" fields
  bool parseEntries(TQString s, int year,
		    TQTime t1, TQTime t2,
		    double costs, double len, double after);

  /// parses the "between(...)" time fields
  bool parseTime(TQTime &t1, TQTime &t2, TQString s);

  /// parses the "use(...)" fields
  bool parseRate(double &costs, double &len, double &after, TQString s);

  /// parses a whole line
  bool parseLine(const TQString &line);

  /// returns midnight time (00:00:00.000)
  TQTime midnight() const;

  /// returns the last valid time BEFORE midnight
  TQTime beforeMidnight() const;

protected:
  TQString _name;
  TQString _currency_symbol;
  TQDateTime starttime;
  int _currency_digits;
  double default_costs;
  double _minimum_costs;
  double  default_len;
  double pcf;
  bool have_flat_init_costs;
  double flat_init_duration;
  double flat_init_costs;

  TQMemArray<RULE> rules;
};

#endif
