/*
 * qcaprovider.h - QCA Plugin API
 * Copyright (C) 2003  Justin Karneges
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef QCAPROVIDER_H
#define QCAPROVIDER_H

#include<tqglobal.h>
#include<tqstring.h>
#include<tqdatetime.h>
#include<tqobject.h>
#include<tqhostaddress.h>
#include"qca.h"

#define QCA_PLUGIN_VERSION 1

class QCAProvider
{
public:
	QCAProvider() {}
	virtual ~QCAProvider() {}

	virtual void init()=0;
	virtual int qcaVersion() const=0;
	virtual int capabilities() const=0;
	virtual void *context(int cap)=0;
};

class QCA_HashContext
{
public:
	virtual ~QCA_HashContext() {}

	virtual QCA_HashContext *clone()=0;
	virtual void reset()=0;
	virtual void update(const char *in, unsigned int len)=0;
	virtual void final(TQByteArray *out)=0;
};

class QCA_CipherContext
{
public:
	virtual ~QCA_CipherContext() {}

	virtual QCA_CipherContext *clone()=0;
	virtual int keySize()=0;
	virtual int blockSize()=0;
	virtual bool generateKey(char *out, int keysize=-1)=0;
	virtual bool generateIV(char *out)=0;

	virtual bool setup(int dir, int mode, const char *key, int keysize, const char *iv, bool pad)=0;
	virtual bool update(const char *in, unsigned int len)=0;
	virtual bool final(TQByteArray *out)=0;
};

class QCA_RSAKeyContext
{
public:
	virtual ~QCA_RSAKeyContext() {}

	virtual QCA_RSAKeyContext *clone() const=0;
	virtual bool isNull() const=0;
	virtual bool havePublic() const=0;
	virtual bool havePrivate() const=0;
	virtual bool createFromDER(const char *in, unsigned int len)=0;
	virtual bool createFromPEM(const char *in, unsigned int len)=0;
	virtual bool createFromNative(void *in)=0;
	virtual bool generate(unsigned int bits)=0;
	virtual bool toDER(TQByteArray *out, bool publicOnly)=0;
	virtual bool toPEM(TQByteArray *out, bool publicOnly)=0;

	virtual bool encrypt(const TQByteArray &in, TQByteArray *out, bool oaep)=0;
	virtual bool decrypt(const TQByteArray &in, TQByteArray *out, bool oaep)=0;
};

struct QCA_CertProperty
{
	TQString var;
	TQString val;
};

class QCA_CertContext
{
public:
	virtual ~QCA_CertContext() {}

	virtual QCA_CertContext *clone() const=0;
	virtual bool isNull() const=0;
	virtual bool createFromDER(const char *in, unsigned int len)=0;
	virtual bool createFromPEM(const char *in, unsigned int len)=0;
	virtual bool toDER(TQByteArray *out)=0;
	virtual bool toPEM(TQByteArray *out)=0;

	virtual TQString serialNumber() const=0;
	virtual TQString subjectString() const=0;
	virtual TQString issuerString() const=0;
	virtual TQValueList<QCA_CertProperty> subject() const=0;
	virtual TQValueList<QCA_CertProperty> issuer() const=0;
	virtual TQDateTime notBefore() const=0;
	virtual TQDateTime notAfter() const=0;
	virtual bool matchesAddress(const TQString &realHost) const=0;
};

class QCA_TLSContext
{
public:
	enum Result { Success, Error, Continue };
	virtual ~QCA_TLSContext() {}

	virtual void reset()=0;
	virtual bool startClient(const TQPtrList<QCA_CertContext> &store, const QCA_CertContext &cert, const QCA_RSAKeyContext &key)=0;
	virtual bool startServer(const TQPtrList<QCA_CertContext> &store, const QCA_CertContext &cert, const QCA_RSAKeyContext &key)=0;

	virtual int handshake(const TQByteArray &in, TQByteArray *out)=0;
	virtual int shutdown(const TQByteArray &in, TQByteArray *out)=0;
	virtual bool encode(const TQByteArray &plain, TQByteArray *to_net, int *encoded)=0;
	virtual bool decode(const TQByteArray &from_net, TQByteArray *plain, TQByteArray *to_net)=0;
	virtual bool eof() const=0;
	virtual TQByteArray unprocessed()=0;

	virtual QCA_CertContext *peerCertificate() const=0;
	virtual int validityResult() const=0;
};

struct QCA_SASLHostPort
{
	TQHostAddress addr;
	Q_UINT16 port;
};

struct QCA_SASLNeedParams
{
	bool user, authzid, pass, realm;
};

class QCA_SASLContext
{
public:
	enum Result { Success, Error, NeedParams, AuthCheck, Continue };
	virtual ~QCA_SASLContext() {}

	// common
	virtual void reset()=0;
	virtual void setCoreProps(const TQString &service, const TQString &host, QCA_SASLHostPort *local, QCA_SASLHostPort *remote)=0;
	virtual void setSecurityProps(bool noPlain, bool noActive, bool noDict, bool noAnon, bool reqForward, bool reqCreds, bool reqMutual, int ssfMin, int ssfMax, const TQString &_ext_authid, int _ext_ssf)=0;
	virtual int security() const=0;
	virtual int errorCond() const=0;

	// init / first step
	virtual bool clientStart(const TQStringList &mechlist)=0;
	virtual int clientFirstStep(bool allowClientSendFirst)=0;
	virtual bool serverStart(const TQString &realm, TQStringList *mechlist, const TQString &name)=0;
	virtual int serverFirstStep(const TQString &mech, const TQByteArray *in)=0;

	// get / set params
	virtual QCA_SASLNeedParams clientParamsNeeded() const=0;
	virtual void setClientParams(const TQString *user, const TQString *authzid, const TQString *pass, const TQString *realm)=0;
	virtual TQString username() const=0;
	virtual TQString authzid() const=0;

	// continue steps
	virtual int nextStep(const TQByteArray &in)=0;
	virtual int tryAgain()=0;

	// results
	virtual TQString mech() const=0;
	virtual const TQByteArray *clientInit() const=0;
	virtual TQByteArray result() const=0;

	// security layer
	virtual bool encode(const TQByteArray &in, TQByteArray *out)=0;
	virtual bool decode(const TQByteArray &in, TQByteArray *out)=0;
};

#endif
