/*
    kopetelistviewitem.h - Kopete's modular QListViewItems

    Copyright (c) 2005      by Engin AYDOGAN          <engin@bzzzt.biz>
    Copyright (c) 2004      by Richard Smith          <kde@metafoo.co.uk>

    Kopete    (c) 2002-2004 by the Kopete developers  <kopete-devel@kde.org>

    *************************************************************************
    *                                                                       *
    * This program is free software; you can redistribute it and/or modify  *
    * it under the terms of the GNU General Public License as published by  *
    * the Free Software Foundation; either version 2 of the License, or     *
    * (at your option) any later version.                                   *
    *                                                                       *
    *************************************************************************
*/

#ifndef KOPETE_LISTVIEWITEM_H
#define KOPETE_LISTVIEWITEM_H

#include <klistview.h>
#include <kopetecontact.h>

#include <utility>
#include <tqimage.h>

class TQPixmap;

namespace Kopete {
namespace UI {
namespace ListView {

class Component;

class ComponentBase
{
public:
	ComponentBase();
	virtual ~ComponentBase() = 0;

	uint components();
	Component *component( uint n );
	Component *componentAt( const TQPoint &pt );

	/** Repaint this item */
	virtual void repaint() = 0;
	/** Relayout this item */
	virtual void relayout() = 0;

	/**
	 * Get the tool tip string and rectangle for a tip request at position
	 * relativePos relative to this item. Queries the appropriate child component.
	 *
	 * @return a pair where the first element is the tooltip, and the second is
	 *         the rectangle within the item for which the tip should be displayed.
	 */
	virtual std::pair<TQString,TQRect> toolTip( const TQPoint &relativePos );

protected:
	/** A child item has been added to this item */
	virtual void componentAdded( Component *component );
	/** A child item has been removed from this item */
	virtual void componentRemoved( Component *component );
	/** A child item has been resized */
	virtual void componentResized( Component *component );
	/** Remove all children */
	virtual void clear();

	/** @internal animate items */
	void updateAnimationPosition( int p, int s );
private:
	class Private;
	Private *d;

	// calls componentAdded and componentRemoved
	friend class Component;
};

/**
 * @author Richard Smith <kde@metafoo.co.uk>
 */
class ToolTipSource
{
public:
	/**
	 * Get the tooltip string and rect for a component
	 *
	 * @param component The component to get a tip for
	 * @param pt   The point (relative to the list item) the mouse is at
	 * @param rect The tip will be removed when the mouse leaves this rect.
	 *             Will initially be set to \p component's rect().
	 */
	virtual TQString operator() ( ComponentBase *component, const TQPoint &pt, TQRect &rect ) = 0;
};

/**
 * This class represents a rectangular subsection of a ListItem.
 *
 * @author Richard Smith <kde@metafoo.co.uk>
 */
class Component : public ComponentBase
{
protected:
	Component( ComponentBase *parent );
public:
	virtual ~Component() = 0;

	/**
	 * Set the size and position of this item relative to the list view item. Should
	 * only be called by the containing item.
	 * @param rect the new rectangle this component will paint in, relative to the painter
	 *             passed to the paint() function by the parent item.
	 */
	virtual void layout( const TQRect &rect );

	/**
	 * Paint this item, inside the rectangle returned by rect().
	 * The default implementation calls paint on all children.
	 */
	virtual void paint( TQPainter *painter, const TQColorGroup &cg );

	void repaint();
	void relayout();

	/**
	 * @return the rect this component was allocated last time it was laid out
	 */
	TQRect rect();
	/**
	 * Prevents this component to be drawn
	 */
	void hide();

	/**
	 * Makes this component to be drawn
	 */
	void show();

	bool isShown();
	bool isHidden();

	/**
	 * Returns the smallest this component can become horizontally while still
	 * being useful.
	 */
	int minWidth();
	/**
	 * Returns the smallest this component can become vertically while still
	 * being useful.
	 */
	int minHeight();

	/**
	 * Returns the width this component desires for a given @a height. By default
	 * this function returns minWidth().
	 */
	virtual int widthForHeight( int height );
	/**
	 * Returns the height this component desires for a given @a width. By default
	 * this function returns minHeight().
	 */
	virtual int heightForWidth( int width );

	/**
	 * Set a tool tip source for this item. The tool tip source object is
	 * still owned by the caller, and must live for at least as long as
	 * this component.
	 */
	void setToolTipSource( ToolTipSource *source = 0 );

	/**
	 * Get the tool tip string and rectangle for a tip request at position
	 * relativePos relative to this item. If a tooltip source is set, it will
	 * be used. Otherwise calls the base class.
	 *
	 * @return a pair where the first element is the tooltip, and the second is
	 *         the rectangle within the item for which the tip should be displayed.
	 */
	std::pair<TQString,TQRect> toolTip( const TQPoint &relativePos );

	/**
	 * RTTI: Runtime Type Information
	 * Exactly the same as Qt's approach to identify types of
	 * QCanvasItems.
	 */

	enum RttiValues {
		Rtti_Component, Rtti_BoxComponent, Rtti_TextComponent,
		Rtti_ImageComponent, Rtti_DisplayNameComponent,
		Rtti_HSpacerComponent, Rtti_VSpacerComponent
	};

	static int RTTI;
	virtual int rtti() const { return RTTI; }
protected:
	/**
	 * Change the minimum width, in pixels, this component requires in order
	 * to be at all useful. Note: do not call this from your layout() function.
	 * @param width the minimum width
	 * @return true if the size has actually changed, false if it's been set to
	 *         the existing values. if it returns true, you do not need to relayout,
	 *         since the parent component will do that for you.
	 */
	bool setMinWidth( int width );
	/**
	 * Change the minimum height, in pixels, this component requires in order
	 * to be at all useful. Note: do not call this from your layout() function.
	 * @param height the minimum height
	 * @return true if the size has actually changed, false if it's been set to
	 *         the existing values. If it returns true, you do not need to relayout,
	 *         since the parent component will do that for you.
	 */
	bool setMinHeight( int height );

	void componentAdded( Component *component );
	void componentRemoved( Component *component );

private:
	// calls the three functions below
	friend void ComponentBase::updateAnimationPosition( int p, int s );

	// used for animation
	void setRect( const TQRect &rect );
	TQRect startRect();
	TQRect targetRect();

	class Private;
	Private *d;
};

class BoxComponent : public Component
{
public:
	enum Direction { Horizontal, Vertical };
	BoxComponent( ComponentBase *parent, Direction dir = Horizontal );
	~BoxComponent();

	virtual void layout( const TQRect &rect );

	virtual int widthForHeight( int height );
	virtual int heightForWidth( int width );

	static int RTTI;
	virtual int rtti() const { return RTTI; }

protected:
	void componentAdded( Component *component );
	void componentRemoved( Component *component );
	void componentResized( Component *component );

private:
	void calcMinSize();

	class Private;
	Private *d;
};

/**
 * ContactBoxComponent
 */
class ContactBoxComponent : public BoxComponent
{
public:
	ContactBoxComponent(ComponentBase *parent, Direction dir = Horizontal);
	~ContactBoxComponent();

	virtual void	reloadTheme();

	virtual void	layout(const TQRect &rect);

	virtual int		widthForHeight(int height);
	virtual int		heightForWidth(int width);

	virtual void	paint(TQPainter *painter, const TQColorGroup &cg);

private:
	class	Private;
	Private	*d;
};

/**
 * GroupBoxComponent
 */
class GroupBoxComponent : public BoxComponent
{
public:
	GroupBoxComponent(ComponentBase *parent, Direction dir = Horizontal);
	~GroupBoxComponent();

	virtual void	reloadTheme();

	virtual void	layout(const TQRect &rect);

	virtual int		widthForHeight(int height);
	virtual int		heightForWidth(int width);

	virtual void	paint(TQPainter *painter, const TQColorGroup &cg);

private:
	class	Private;
	Private	*d;
};

class TextComponent : public Component
{
public:
	TextComponent( ComponentBase *parent, const TQFont &font = TQFont(), const TQString &text = TQString::null );
	~TextComponent();

	TQString text();
	void setText( const TQString &text );

	TQFont font();
	void setFont( const TQFont &font );

	TQColor color();
	void setColor( const TQColor &color );
	void setDefaultColor();

	int widthForHeight( int );

	void paint( TQPainter *painter, const TQColorGroup &cg );

	static int RTTI;
	virtual int rtti() const { return RTTI; }

private:
	void calcMinSize();

	class Private;
	Private *d;
};

class ImageComponent : public Component
{
public:
	ImageComponent( ComponentBase *parent );
	ImageComponent( ComponentBase *parent, int minW, int minH );
	~ImageComponent();

	virtual void setPixmap( const TQPixmap &img, bool adjustSize = true);
	TQPixmap pixmap( void );

	virtual void paint( TQPainter *painter, const TQColorGroup &cg );

	virtual void scale( int w, int h, TQImage::ScaleMode );
	static int RTTI;
	virtual int rtti() const { return RTTI; }
protected:
	class Private;
	Private *d;
};

/**
 * FaceComponent
 */
class FaceComponent : public ImageComponent
{
public:
	FaceComponent(ComponentBase *parent): ImageComponent(parent) {}
	FaceComponent(ComponentBase *parent, int minW, int minH): ImageComponent(parent, minH, minW) {}

	void	setPixmap(const TQPixmap &img, bool adjustSize = true);
	void	paint(TQPainter *painter, const TQColorGroup &cg);
};

/**
 * ContactComponent
 */
class ContactComponent : public ImageComponent
{
public:
	ContactComponent( ComponentBase *parent, Kopete::Contact *contact, int iconSize);
	~ContactComponent();
	void updatePixmap();
	Kopete::Contact *contact();
	std::pair<TQString,TQRect> toolTip( const TQPoint &relativePos );
protected:
	class Private;
	Private *d;
};

/**
 * SpacerComponent
 */
class SpacerComponent : public Component
{
public:
	SpacerComponent( ComponentBase *parent, int w, int h );
};

/**
 * DisplayNameComponent
 */

class DisplayNameComponent : public BoxComponent
{
public:
	/**
	 * Constructor
	 */
	DisplayNameComponent( ComponentBase *parent );

	/**
	 * Dtor
	 */
	~DisplayNameComponent();
	void layout( const TQRect& rect );

	TQString text();
	void setText( const TQString& text );
	void setFont( const TQFont& font );
	void setColor( const TQColor& color );
	void setDefaultColor();
	static int RTTI;
	virtual int rtti() const { return RTTI; }
	/**
	 * reparse again for emoticon (call this when emoticon theme change)
	 */
	void redraw();

private:
	class Private;
	Private *d;
};

class HSpacerComponent : public Component
{
public:
	HSpacerComponent( ComponentBase *parent );
	int widthForHeight( int );

	static int RTTI;
	virtual int rtti() const { return RTTI; }
};

class VSpacerComponent : public Component
{
public:
	VSpacerComponent( ComponentBase *parent );
	int heightForWidth( int );

	static int RTTI;
	virtual int rtti() const { return RTTI; }
};

/**
 * List-view item composed of Component items. Supports height-for-width, tooltips and
 * some animation effects.
 *
 * @author Richard Smith <kde@metafoo.co.uk>
 */
class Item : public TQObject, public KListViewItem, public ComponentBase
{
	Q_OBJECT
public:
	Item( TQListView *parent, TQObject *owner = 0, const char *name = 0 );
	Item( TQListViewItem *parent, TQObject *owner = 0, const char *name = 0  );
	~Item();

	void repaint();
	void relayout();

	void setup();
	virtual void paintCell( TQPainter *p, const TQColorGroup &cg, int column, int width, int align );
	//TODO: startRename(...)

	float opacity();
	void setOpacity( float alpha );

	bool targetVisibility();
	void setTargetVisibility( bool vis );

	/**
	 * Turn on and off certain visual effects for all Items.
	 * @param animation whether changes to items should be animated.
	 * @param fading whether requests to setTargetVisibility should cause fading of items.
	 * @param folding whether requests to setTargetVisibility should cause folding of items.
	 */
	static void setEffects( bool animation, bool fading, bool folding );

	int width( const TQFontMetrics & fm, const TQListView * lv, int c ) const;

	/**
	 * Show or hide this item in a clean way depending on whether it matches
	 * the current quick search
	 * @param match If true, show or hide the item as normal. If false, hide the item immediately.
	 */
	virtual void setSearchMatch( bool match );

protected:
	void componentAdded( Component *component );
	void componentRemoved( Component *component );
	void componentResized( Component *component );

	void setHeight( int );

private:
	void initLVI();
	void recalcHeight();
	void scheduleLayout();

private slots:
	void slotColumnResized();
	void slotLayoutItems();
	void slotLayoutAnimateItems();
	void slotUpdateVisibility();

private:
	class Private;
	Private *d;
};

} // END namespace ListView
} // END namespace UI
} // END namespace Kopete

#endif

// vim: set noet ts=4 sts=4 sw=4:
