/* This file is part of the KDE libraries
   Copyright (C) 1996 Martynas Kunigelis

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License version 2 as published by the Free Software Foundation.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
/*****************************************************************************
*                                                                            *
*  KGameProgress -- progress indicator widget for KDE by Martynas Kunigelis  *
*                                                                            *
*****************************************************************************/

#ifndef _KPROGRES_H
#define _KPROGRES_H "$Id: kgameprogress.h 1157643 2010-07-31 19:42:31Z tpearson $"

#include <tqframe.h>
#include <tqrangecontrol.h>
#include <kdemacros.h>
/**
 * @short A progress indicator widget.
 *
 * KGameProgress is derived from TQFrame and TQRangeControl, so
 * you can use all the methods from those classes. The only difference
 * is that setValue() is now made a slot, so you can connect
 * stuff to it.
 *
 * None of the constructors take line step and page step as arguments,
 * so by default they're set to 1 and 10 respectively.
 *
 * The Blocked style ignores the textEnabled() setting and displays
 * no text, since it looks truly ugly (and for other reasons). Signal
 * percentageChanged() is emitted whenever the value changes so you
 * can set up a different widget to display the current percentage complete
 * and connect the signal to it.
 *
 * @author Martynas Kunigelis
 * @version $Id: kgameprogress.h 1157643 2010-07-31 19:42:31Z tpearson $
 */
class KDE_EXPORT KGameProgress : public TQFrame, public QRangeControl
{
  Q_OBJECT
  Q_ENUMS( BarStyle )
  Q_PROPERTY( int value READ value WRITE setValue)
  Q_PROPERTY( BarStyle barStyle READ barStyle WRITE setBarStyle )
  Q_PROPERTY( TQColor barColor READ barColor WRITE setBarColor )
  Q_PROPERTY( TQPixmap barPixmap READ barPixmap WRITE setBarPixmap )
  Q_PROPERTY( Orientation orientation READ orientation WRITE setOrientation )
  Q_PROPERTY( bool textEnabled READ textEnabled WRITE setTextEnabled )

public:
  /**
   * Possible values for bar style.
   *
   * @p Solid means one continuous progress bar, @p Blocked means a
   * progress bar made up of several blocks.
   */
  enum BarStyle { Solid, Blocked };

  /**
   * Construct a horizontal progress bar.
   */
  KGameProgress(TQWidget *parent=0, const char *name=0);

  /**
   * Construct a progress bar with orientation @p orient.
   */
  KGameProgress(Orientation orient, TQWidget *parent=0, const char *name=0);

  /**
   * Construct a progress bar with minimum, maximum and initial values.
   */
  KGameProgress(int minValue, int maxValue, int value, Orientation,
                TQWidget *parent=0, const char *name=0);

  /**
   * Destruct the progress bar.
   */
  ~KGameProgress();

  /**
   * Set the progress bar style.
   *
   * Allowed values are @p Solid and @p Blocked.
   */
  void setBarStyle(BarStyle style);

  /**
   * Set the color of the progress bar.
   */
  void setBarColor(const TQColor &);

  /**
   * Set a pixmap to be shown in the progress bar.
   */
  void setBarPixmap(const TQPixmap &);

  /**
   * Set the orientation of the progress bar.
   *
   * Allowed values are @p Horizontal and @p Vertical.
   */
  void setOrientation(Orientation);

  /**
   * If this is set to @p true, the progress text will be displayed.
   *
   */
  void setTextEnabled(bool);

  /**
   * Retrieve the bar style.
   *
   * @see setBarStyle()
   */
  BarStyle barStyle() const;

  /**
   * Retrieve the bar color.
   * @see setBarColor()
   */
  const TQColor &barColor() const;

  /**
   * Retrieve the bar pixmap.
   *
   * @see setBarPixmap()
   */
  const TQPixmap *barPixmap() const;

  /**
   * Retrive the current status
   *
   * @see setValue()
   */
  int value() const { return TQRangeControl::value(); }
  /**
   * Retrive the orientation of the progress bar.
   *
   * @see setOrientation()
   */
  Orientation orientation() const;

  /**
   * Returns @p true if progress text will be displayed,
   * @p false otherwise.
   *
   * @see setFormat()
   */
  bool textEnabled() const;

  /**
   */
  virtual TQSize sizeHint() const;

  /**
   */
  virtual TQSize minimumSizeHint() const;

  /**
   */
  virtual TQSizePolicy sizePolicy() const;

  /**
   * Retrieve the current format for printing status text.
   * @see setFormat()
   */
  TQString format() const;

public slots:

  /**
   * Set the format of the text to use to display status.
   *
   * The default format is "%p%" (which looks like "42%".)
   *
   * @param format %p is replaced by percentage done, %v is replaced by actual
   * value, %m is replaced by the maximum value.
   */
  void setFormat(const TQString & format);

  /**
   * Set the current value of the progress bar to @p value.
   *
   * This must be a number in the range 0..100.
   */
  void setValue(int value);

  /**
   * Advance the progress bar by @p prog.
   *
   * This method is
   * provided for convenience and is equivalent with
   * setValue(value()+prog).
   */
  void advance(int prog);

signals:
  /**
   * Emitted when the state of the progress bar changes.
   */
  void percentageChanged(int);

protected:
  /**
   */
  void valueChange();
  /**
   */
  void rangeChange();
  /**
   */
  void styleChange( TQStyle& );
  /**
   */
  void paletteChange( const TQPalette & );
  /**
   */
  void drawContents( TQPainter * );

private slots:
  void paletteChange();

private:
  TQPixmap  *bar_pixmap;
  bool      use_supplied_bar_color;
  TQColor    bar_color;
  TQColor    bar_text_color;
  TQColor    text_color;
  TQRect     fr;
  BarStyle  bar_style;
  Orientation orient;
  bool      text_enabled;
  TQString   format_;
  void      initialize();
  int       recalcValue(int);
  void      drawText(TQPainter *);
  void      adjustStyle();

  class KGameProgressPrivate;
  KGameProgressPrivate *d;
};


#endif
