/* Yo Emacs, this is -*- C++ -*- */
/*
 *   ksame 0.4 - simple Game
 *   Copyright (C) 1997,1998  Marcus Kreutzberger
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 *
 */

#include "KSameWidget.h"

#include <tqwidget.h>
#include <tqvbox.h>

#include <kapplication.h>
#include <kiconloader.h>
#include <kdialogbase.h>
#include <kscoredialog.h>
#include <kstatusbar.h>
#include <knuminput.h>
#include <klocale.h>
#include <knotifyclient.h>
#include <knotifydialog.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include <kconfig.h>
#include <kstdgameaction.h>
#include <kaction.h>
#include <kdebug.h>

#include "StoneWidget.h"

static int default_colors=3;

#define Board 	KScoreDialog::Custom1

KSameWidget::KSameWidget(TQWidget *parent, const char* name, WFlags fl) :
             KMainWindow(parent,name,fl)
{
	KStdGameAction::gameNew(this, TQT_SLOT(m_new()), actionCollection(), "game_new");
	restart = new KAction(i18n("&Restart This Board"), CTRL+Key_R, this,
						  TQT_SLOT(m_restart()), actionCollection(), "game_restart");
	KStdGameAction::highscores(this, TQT_SLOT(m_showhs()), actionCollection(), "game_highscores");
	KStdGameAction::quit(this, TQT_SLOT(close()), actionCollection(), "game_quit");
	undo = KStdGameAction::undo(this, TQT_SLOT(m_undo()), actionCollection(), "edit_undo");

	random = new KToggleAction(i18n("&Random Board"), 0, 0, 0, actionCollection(), "random_board");
	showNumberRemaining = new KToggleAction(i18n("&Show Number Remaining"), 0, this, TQT_SLOT(showNumberRemainingToggled()), actionCollection(), "showNumberRemaining");

	KStdAction::configureNotifications(this, TQT_SLOT(configureNotifications()),
               actionCollection());

	status=statusBar();
	status->insertItem(i18n("Colors: XX"),1,1);
	status->insertItem(i18n("Board: XXXXXX"),2,1);
	status->insertItem(i18n("Marked: XXXXXX"),3,1);
	status->insertItem(i18n("Score: XXXXXX"),4,1);

	stone = new StoneWidget(this,15,10);

	connect( stone, TQT_SIGNAL(s_gameover()), this, TQT_SLOT(gameover()));
	connect( stone, TQT_SIGNAL(s_colors(int)), this, TQT_SLOT(setColors(int)));
	connect( stone, TQT_SIGNAL(s_board(int)), this, TQT_SLOT(setBoard(int)));
	connect( stone, TQT_SIGNAL(s_marked(int)), this, TQT_SLOT(setMarked(int)));
	connect( stone, TQT_SIGNAL(s_score(int)), this, TQT_SLOT(setScore(int)));
	connect( stone, TQT_SIGNAL(s_remove(int,int)), this, TQT_SLOT(stonesRemoved(int,int)));

	connect(stone, TQT_SIGNAL(s_sizechanged()), this, TQT_SLOT(sizeChanged()));

	sizeChanged();
	setCentralWidget(stone);

	// Once the main view can scale then use defualt setupGUI() and remove the
	//    noMerge="1" from the uirc file
	// StatusBar | ToolBar
	setupGUI(KMainWindow::Save | Keys | Create );

	random->setChecked(true);
	setScore(0);

	if (!kapp->isRestored())
		newGame(kapp->random(),default_colors);
	
	KConfig *cfg = kapp->config();
	if (cfg->readBoolEntry("showRemaining"))
	{
		showNumberRemaining->setChecked(true);
		showNumberRemainingToggled();
	}
}

void KSameWidget::readProperties(KConfig *conf) {
	Q_ASSERT(conf);
	stone->readProperties(conf);
}

void KSameWidget::saveProperties(KConfig *conf) {
	Q_ASSERT(conf);
	stone->saveProperties(conf);
	conf->sync();
}

void KSameWidget::sizeChanged() {
	stone->setFixedSize(stone->sizeHint());
}

void KSameWidget::showNumberRemainingToggled()
{
	if(showNumberRemaining->isChecked()){
		TQStringList list;
		for(int i=1;i<=stone->colors();i++)
			list.append(TQString("%1").arg(stone->count(i)));
		TQString count = TQString(" (%1)").arg(list.join(","));
		status->changeItem(i18n("%1 Colors%2").arg(stone->colors()).arg(count),1);
	}
	else status->changeItem(i18n("%1 Colors").arg(stone->colors()),1);
	
	KConfig *cfg = kapp->config();
	cfg->writeEntry("showRemaining", showNumberRemaining->isChecked());
	cfg->sync();
}

void KSameWidget::newGame(unsigned int board,int colors) {
	while (board>=1000000) board-=1000000;
	// kdDebug() << "newgame board " << board << " colors " << colors << endl;
	stone->newGame(board,colors);
	setScore(0);
}

bool KSameWidget::confirmAbort() {
	return stone->isGameover() ||
		stone->isOriginalBoard() ||
		(KMessageBox::questionYesNo(this, i18n("Do you want to resign?"),
									i18n("New Game"),i18n("Resign"),KStdGuiItem::cancel()) == KMessageBox::Yes);
}

void KSameWidget::m_new() {
	if (random->isChecked()) {
		if (confirmAbort())
			newGame(kapp->random(),default_colors);
	} else {
		KDialogBase dlg(this, "boardchooser", true,
		                i18n("Select Board"),
		                KDialogBase::Ok | KDialogBase::Cancel,
		                KDialogBase::Ok);

		TQVBox *page = dlg.makeVBoxMainWidget();

		KIntNumInput bno(0, page);
		bno.setRange(0, 1000000, 1);
		bno.setLabel(i18n("Select a board:"));
		bno.setFocus();
		bno.setFixedSize(bno.sizeHint());
		bno.setValue(stone->board());

		if (dlg.exec()) newGame(bno.value(),default_colors);
	}
}

void KSameWidget::m_restart() {
	if (confirmAbort())
		newGame(stone->board(),default_colors);
}

void KSameWidget::m_undo() {
	Q_ASSERT(stone);
	stone->undo();
}

void KSameWidget::m_showhs() {
	Q_ASSERT(stone);
	stone->unmark();
	KScoreDialog d(KScoreDialog::Name | KScoreDialog::Score, this);
	d.addField(Board, i18n("Board"), "Board");
	d.exec();
}

void KSameWidget::setColors(int colors) {
	status->changeItem(i18n("%1 Colors").arg(colors),1);
}

void KSameWidget::setBoard(int board) {
	status->changeItem(i18n("Board: %1").arg(board, 6), 2);
}

void KSameWidget::setMarked(int m) {
	status->changeItem(i18n("Marked: %1").arg(m, 6),3);
	m_markedStones=m;
}

void KSameWidget::stonesRemoved(int,int) {
	KNotifyClient::event(winId(),"stones removed",
	   i18n("One stone removed.","%n stones removed.",m_markedStones));
}

void KSameWidget::setScore(int score) {
	if(showNumberRemaining->isChecked()){
		TQStringList list;
		for(int i=1;i<=stone->colors();i++)
			list.append(TQString("%1").arg(stone->count(i)));
		TQString count = TQString(" (%1)").arg(list.join(","));
		status->changeItem(i18n("%1 Colors%2").arg(stone->colors()).arg(count),1);
	}
	status->changeItem(i18n("Score: %1").arg(score, 6),4);
	undo->setEnabled(stone->undoPossible());
	restart->setEnabled(!stone->isOriginalBoard());
}

void KSameWidget::gameover() {
  if (stone->hasBonus()) {
	  KNotifyClient::event(winId(), "game won",
		  i18n("You even removed the last stone, great job! "
			   "This gave you a score of %1 in total.").arg(stone->score()));
  } else {
	  KNotifyClient::event(winId(), "game over",
		  i18n("There are no more removeable stones. "
			   "You got a score of %1 in total.").arg(stone->score()));
  }
  stone->unmark();
  KScoreDialog d(KScoreDialog::Name | KScoreDialog::Score, this);
  d.addField(Board, i18n("Board"), "Board");

  KScoreDialog::FieldInfo scoreInfo;
  scoreInfo[Board].setNum(stone->board());

  if (d.addScore(stone->score(), scoreInfo))
    d.exec();
}

void KSameWidget::configureNotifications() {
    KNotifyDialog::configure(this);
}

#include "KSameWidget.moc"
