/***************************************************************************
    begin                : Sun Mar 6 2003
    copyright            : (C) 2003 by Richard Lrkng
    email                : nouseforaname@home.se

    copyright            : (C) 2003 - 2004 by Scott Wheeler
    email                : wheeler@kde.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <klocale.h>
#include <klineedit.h>
#include <kiconloader.h>
#include <kcombobox.h>
#include <kdebug.h>
#include <kaction.h>

#include <tqlayout.h>
#include <tqlabel.h>
#include <tqcheckbox.h>
#include <tqpushbutton.h>
#include <tqtoolbutton.h>

#include "searchwidget.h"
#include "collectionlist.h"
#include "actioncollection.h"

using namespace ActionCollection;

////////////////////////////////////////////////////////////////////////////////
// SearchLine public methods
////////////////////////////////////////////////////////////////////////////////

SearchLine::SearchLine(TQWidget *parent, bool simple, const char *name) :
    TQHBox(parent, name),
    m_simple(simple),
    m_searchFieldsBox(0)
{
    setSpacing(5);

    if(!m_simple) {
	m_searchFieldsBox = new KComboBox(this, "searchFields");
	connect(m_searchFieldsBox, TQT_SIGNAL(activated(int)),
		this, TQT_SIGNAL(signalQueryChanged()));
    }

    m_lineEdit = new KLineEdit(this, "searchLineEdit");
    m_lineEdit->installEventFilter(this);
    connect(m_lineEdit, TQT_SIGNAL(textChanged(const TQString &)),
            this, TQT_SIGNAL(signalQueryChanged()));
    connect(m_lineEdit, TQT_SIGNAL(returnPressed()),
            this, TQT_SLOT(slotActivate()));

    if(!m_simple) {
	m_caseSensitive = new KComboBox(this);
	m_caseSensitive->insertItem(i18n("Normal Matching"), 0);
	m_caseSensitive->insertItem(i18n("Case Sensitive"), 1);
	m_caseSensitive->insertItem(i18n("Pattern Matching"), 2);
	connect(m_caseSensitive, TQT_SIGNAL(activated(int)),
		this, TQT_SIGNAL(signalQueryChanged()));
    }
    else
	m_caseSensitive = 0;

    updateColumns();
}

PlaylistSearch::Component SearchLine::searchComponent() const
{
    TQString query = m_lineEdit->text();
    bool caseSensitive = m_caseSensitive && m_caseSensitive->currentItem() == CaseSensitive;

    Playlist *playlist = CollectionList::instance();

    TQValueList<int> searchedColumns;

    if(!m_searchFieldsBox || m_searchFieldsBox->currentItem() == 0) {
	TQValueListConstIterator<int> it = m_columnList.begin();
	for(; it != m_columnList.end(); ++it) {
	    if(playlist->isColumnVisible(*it))
		searchedColumns.append(*it);
	}
    }
    else
	searchedColumns.append(m_columnList[m_searchFieldsBox->currentItem() - 1]);

    if(m_caseSensitive && m_caseSensitive->currentItem() == Pattern)
	return PlaylistSearch::Component(TQRegExp(query), searchedColumns);
    else
	return PlaylistSearch::Component(query, caseSensitive, searchedColumns);
}

void SearchLine::setSearchComponent(const PlaylistSearch::Component &component)
{
    if(component == searchComponent())
	return;

    if(m_simple || !component.isPatternSearch()) {
	m_lineEdit->setText(component.query());
	if(m_caseSensitive)
	    m_caseSensitive->setCurrentItem(component.isCaseSensitive() ? CaseSensitive : Default);
    }
    else {
	m_lineEdit->setText(component.pattern().pattern());
	if(m_caseSensitive)
	    m_caseSensitive->setCurrentItem(Pattern);
    }

    if(!m_simple) {
	if(component.columns().isEmpty() || component.columns().size() > 1)
	    m_searchFieldsBox->setCurrentItem(0);
	else
	    m_searchFieldsBox->setCurrentItem(component.columns().front() + 1);
    }
}

void SearchLine::clear()
{
    // We don't want to emit the signal if it's already empty.
    if(!m_lineEdit->text().isEmpty())
	m_lineEdit->clear();
}

void SearchLine::setFocus()
{
    m_lineEdit->setFocus();
}

bool SearchLine::eventFilter(TQObject *watched, TQEvent *e)
{
    if(watched != m_lineEdit || e->type() != TQEvent::KeyPress)
	return TQHBox::eventFilter(watched, e);

    TQKeyEvent *key = static_cast<TQKeyEvent *>(e);
    if(key->key() == Qt::Key_Down)
	emit signalDownPressed();

    return TQHBox::eventFilter(watched, e);
}

void SearchLine::slotActivate()
{
    action("stop")->activate();
    action("playFirst")->activate();
}

void SearchLine::updateColumns()
{
    TQString currentText;

    if(m_searchFieldsBox) {
	currentText = m_searchFieldsBox->currentText();
	m_searchFieldsBox->clear();
    }

    TQStringList columnHeaders;

    columnHeaders.append(TQString("<%1>").arg(i18n("All Visible")));

    Playlist *playlist = CollectionList::instance();

    int selection = -1;
    m_columnList.clear();

    for(int i = 0; i < playlist->columns(); i++) {
	m_columnList.append(i);
	TQString text = playlist->columnText(i);
	columnHeaders.append(text);
	if(currentText == text)
	    selection = m_columnList.size() - 1;
    }

    if(m_searchFieldsBox) {
	m_searchFieldsBox->insertStringList(columnHeaders);
	m_searchFieldsBox->setCurrentItem(selection + 1);
    }
}

////////////////////////////////////////////////////////////////////////////////
// SearchWidget public methods
////////////////////////////////////////////////////////////////////////////////

SearchWidget::SearchWidget(TQWidget *parent, const char *name) : KToolBar(parent, name)
{
    setupLayout();
    updateColumns();
}

SearchWidget::~SearchWidget()
{

}

void SearchWidget::setSearch(const PlaylistSearch &search)
{
    PlaylistSearch::ComponentList components = search.components();

    if(components.isEmpty()) {
	clear();
	return;
    }

    m_searchLine->setSearchComponent(*components.begin());
}

TQString SearchWidget::searchText() const
{
    return m_searchLine->searchComponent().query();
}

void SearchWidget::setSearchText(const TQString &text)
{
    m_searchLine->setSearchComponent(PlaylistSearch::Component(text));
}

PlaylistSearch SearchWidget::search(const PlaylistList &playlists) const
{
    PlaylistSearch::ComponentList components;
    components.append(m_searchLine->searchComponent());
    return PlaylistSearch(playlists, components);
}



////////////////////////////////////////////////////////////////////////////////
// SearchWidget public slots
////////////////////////////////////////////////////////////////////////////////

void SearchWidget::clear()
{
    m_searchLine->clear();
}

void SearchWidget::setEnabled(bool enable)
{
    emit signalShown(enable);
    setShown(enable);
}

void SearchWidget::setFocus()
{
    m_searchLine->setFocus();
}

////////////////////////////////////////////////////////////////////////////////
// SearchWidget private methods
////////////////////////////////////////////////////////////////////////////////

void SearchWidget::updateColumns()
{
    m_searchLine->updateColumns();
}

void SearchWidget::setupLayout()
{
    boxLayout()->setSpacing(5);

    TQToolButton *clearSearchButton = new TQToolButton(this);
    clearSearchButton->setTextLabel(i18n("Clear Search"), true);
    clearSearchButton->setIconSet(SmallIconSet("locationbar_erase"));

    TQLabel *label = new TQLabel(i18n("Search:"), this, "kde toolbar widget");

    m_searchLine = new SearchLine(this, true, "kde toolbar widget");

    label->setBuddy(m_searchLine);

    connect(m_searchLine, TQT_SIGNAL(signalQueryChanged()), this, TQT_SIGNAL(signalQueryChanged()));
    connect(m_searchLine, TQT_SIGNAL(signalDownPressed()), this, TQT_SIGNAL(signalDownPressed()));
    connect(clearSearchButton, TQT_SIGNAL(pressed()), m_searchLine, TQT_SLOT(clear()));
    setStretchableWidget(m_searchLine);

    // I've decided that I think this is ugly, for now.

    /*
      TQToolButton *b = new TQToolButton(this);
      b->setTextLabel(i18n("Advanced Search"), true);
      b->setIconSet(SmallIconSet("wizard"));

      connect(b, TQT_SIGNAL(clicked()), this, TQT_SIGNAL(signalAdvancedSearchClicked()));
    */
}

#include "searchwidget.moc"
