// -*- c-basic-offset: 4 -*-

/*
    Rosegarden
    A sequencer and musical notation editor.
 
    This program is Copyright 2000-2008
        Guillaume Laurent   <glaurent@telegraph-road.org>,
        Chris Cannam        <cannam@all-day-breakfast.com>,
        Richard Bown        <bownie@bownie.com>
 
    The moral right of the authors to claim authorship of this work
    has been asserted.
 
    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/

#include <vector>
#include "misc/Debug.h"

#include <tqpainter.h>

#include "QCanvasSimpleSprite.h"

namespace Rosegarden {

QCanvasSimpleSprite::QCanvasSimpleSprite(TQPixmap *pixmap, TQCanvas *canvas)
        : TQCanvasSprite(0, canvas),
        m_pixmapArray(0)
{
    m_pixmapArray = makePixmapArray(pixmap);
    setSequence(m_pixmapArray);
}

QCanvasSimpleSprite::QCanvasSimpleSprite(TQCanvasPixmap *pixmap, TQCanvas *canvas)
        : TQCanvasSprite(0, canvas),
        m_pixmapArray(0)
{
    m_pixmapArray = makePixmapArray(pixmap);
    setSequence(m_pixmapArray);
}

QCanvasSimpleSprite::QCanvasSimpleSprite(const TQString &pixmapfile,
        TQCanvas *canvas)
        : TQCanvasSprite(0, canvas),
        m_pixmapArray(0)
{
    m_pixmapArray = makePixmapArray(pixmapfile);
    setSequence(m_pixmapArray);
}

QCanvasSimpleSprite::QCanvasSimpleSprite(TQCanvas *canvas)
        : TQCanvasSprite(0, canvas),
        m_pixmapArray(0)
{
    TQCanvasPixmapArray* tmpArray = makePixmapArray(new TQPixmap());
    setSequence(tmpArray);
}


QCanvasSimpleSprite::~QCanvasSimpleSprite()
{
    PixmapArrayGC::registerForDeletion(m_pixmapArray);
    m_pixmapArray = 0;

    // We can't delete m_pixmapArray or we get a core dump.
    //
    // The reason I think is that after the QCanvasSprite is deleted,
    // it is removed from the TQCanvas, which therefore needs the
    // pixmaps to know how to update itself (the crash is in
    // TQCanvas::removeChunks(), usually).
    //
    // So instead we have to do this GCish
    // thingy. PixmapArrayGC::deleteAll() is called by
    // NotationView::redoLayout
}

TQCanvasPixmapArray*
QCanvasSimpleSprite::makePixmapArray(TQPixmap *pixmap)
{
    TQPtrList<TQPixmap> pixlist;
    pixlist.setAutoDelete(true); // the TQCanvasPixmapArray creates its
    // own copies of the pixmaps, so we
    // can delete the one we're passed
    pixlist.append(pixmap);

    TQPtrList<TQPoint> spotlist;
    spotlist.setAutoDelete(true);
    spotlist.append(new TQPoint(0, 0));

    return new TQCanvasPixmapArray(pixlist, spotlist);
}

TQCanvasPixmapArray*
QCanvasSimpleSprite::makePixmapArray(TQCanvasPixmap *pixmap)
{
    TQPtrList<TQPixmap> pixlist;
    pixlist.setAutoDelete(true); // the TQCanvasPixmapArray creates its
    // own copies of the pixmaps, so we
    // can delete the one we're passed
    pixlist.append(pixmap);

    TQPtrList<TQPoint> spotlist;
    spotlist.setAutoDelete(true);
    spotlist.append(new TQPoint(pixmap->offsetX(), pixmap->offsetY()));

    return new TQCanvasPixmapArray(pixlist, spotlist);
}

TQCanvasPixmapArray*
QCanvasSimpleSprite::makePixmapArray(const TQString &pixmapfile)
{
    return new TQCanvasPixmapArray(pixmapfile);
}

//////////////////////////////////////////////////////////////////////

TQCanvasNotationSprite::TQCanvasNotationSprite(NotationElement& n,
        TQPixmap* pixmap,
        TQCanvas* canvas)
        : QCanvasSimpleSprite(pixmap, canvas),
        m_notationElement(n)
{}

TQCanvasNotationSprite::TQCanvasNotationSprite(NotationElement& n,
        TQCanvasPixmap* pixmap,
        TQCanvas* canvas)
        : QCanvasSimpleSprite(pixmap, canvas),
        m_notationElement(n)

{}

TQCanvasNotationSprite::~TQCanvasNotationSprite()
{}


TQCanvasNonElementSprite::TQCanvasNonElementSprite(TQPixmap *pixmap,
        TQCanvas *canvas) :
        QCanvasSimpleSprite(pixmap, canvas)
{}

TQCanvasNonElementSprite::TQCanvasNonElementSprite(TQCanvasPixmap *pixmap,
        TQCanvas *canvas) :
        QCanvasSimpleSprite(pixmap, canvas)
{}

TQCanvasNonElementSprite::~TQCanvasNonElementSprite()
{}

TQCanvasTimeSigSprite::TQCanvasTimeSigSprite(double layoutX,
        TQPixmap *pixmap,
        TQCanvas *canvas) :
        TQCanvasNonElementSprite(pixmap, canvas),
        m_layoutX(layoutX)
{}

TQCanvasTimeSigSprite::TQCanvasTimeSigSprite(double layoutX,
        TQCanvasPixmap *pixmap,
        TQCanvas *canvas) :
        TQCanvasNonElementSprite(pixmap, canvas),
        m_layoutX(layoutX)
{}

TQCanvasTimeSigSprite::~TQCanvasTimeSigSprite()
{}


QCanvasStaffNameSprite::QCanvasStaffNameSprite(TQPixmap *pixmap,
        TQCanvas *canvas) :
        TQCanvasNonElementSprite(pixmap, canvas)
{}

QCanvasStaffNameSprite::QCanvasStaffNameSprite(TQCanvasPixmap *pixmap,
        TQCanvas *canvas) :
        TQCanvasNonElementSprite(pixmap, canvas)
{}

QCanvasStaffNameSprite::~QCanvasStaffNameSprite()
{}


//////////////////////////////////////////////////////////////////////

void PixmapArrayGC::registerForDeletion(TQCanvasPixmapArray* array)
{
    m_pixmapArrays.push_back(array);
}

void PixmapArrayGC::deleteAll()
{
    RG_DEBUG << "PixmapArrayGC::deleteAll() : "
    << m_pixmapArrays.size() << " pixmap arrays to delete\n";

    static unsigned long total = 0;

    for (unsigned int i = 0; i < m_pixmapArrays.size(); ++i) {
        TQCanvasPixmapArray *array = m_pixmapArrays[i];
        TQPixmap *pixmap = array->image(0);
        if (pixmap) {
            total += pixmap->width() * pixmap->height();
            //	    NOTATION_DEBUG << "PixmapArrayGC::deleteAll(): " << pixmap->width() << "x" << pixmap->height() << " (" << (pixmap->width()*pixmap->height()) << " px, " << total << " total)" << endl;
        }
        delete m_pixmapArrays[i];
    }

    m_pixmapArrays.clear();
}

std::vector<TQCanvasPixmapArray*> PixmapArrayGC::m_pixmapArrays;

}

//////////////////////////////////////////////////////////////////////
