
/* -*- c-basic-offset: 4 indent-tabs-mode: nil -*- vi:set ts=8 sts=4 sw=4: */

/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.

    This program is Copyright 2000-2008
        Guillaume Laurent   <glaurent@telegraph-road.org>,
        Chris Cannam        <cannam@all-day-breakfast.com>,
        Richard Bown        <richard.bown@ferventsoftware.com>

    The moral rights of Guillaume Laurent, Chris Cannam, and Richard
    Bown to claim authorship of this work have been asserted.

    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/

#ifndef _RG_NOTATIONCANVASVIEW_H_
#define _RG_NOTATIONCANVASVIEW_H_

#include "gui/general/RosegardenCanvasView.h"
#include <tqrect.h>
#include <vector>


class TQWidget;
class TQString;
class TQPoint;
class TQPaintEvent;
class TQPainter;
class TQMouseEvent;
class TQCanvasLineGroupable;
class TQCanvasItemGroup;
class TQCanvasItem;
class TQCanvas;


namespace Rosegarden
{

class NotationStaff;
class NotationElement;
class LinedStaffManager;


/**
 * Central widget for the NotationView window
 *
 * This class only takes care of the event handling
 * (see the various signals).
 *
 * It does not deal with any canvas element. All elements are added by
 * the NotationView.
 *
 *@see NotationView
 */

class NotationCanvasView : public RosegardenCanvasView
{
    Q_OBJECT
  TQ_OBJECT

public:
    NotationCanvasView(const LinedStaffManager &staffmgr,
                       TQCanvas *viewing, TQWidget *parent=0,
                       const char *name=0, WFlags f=0);

    ~NotationCanvasView();

    void setHeightTracking(bool t);

signals:

    /**
     * Emitted when the user clicks on a staff (e.g. mouse button press)
     * \a pitch is set to the MIDI pitch on which the click occurred
     * \a staffNo is set to the staff on which the click occurred
     * \a point is set to the coordinates of the click event
     * \a el points to the NotationElement which was clicked on, if any
     */
    void itemPressed(int pitch, int staffNo,
                     TQMouseEvent*,
                     NotationElement* el);

    /**
     * Emitted when the user clicks on a TQCanvasItem which is active
     *
     * @see TQCanvasItem#setActive
     */
    void activeItemPressed(TQMouseEvent*,
                           TQCanvasItem* item);

    /**
     * Emitted when the user clicks on a TQCanvasItem which is neither
     * active nor a notation element
     */
    void nonNotationItemPressed(TQMouseEvent *,
                                TQCanvasItem *);

    /**
     * Emitted when the user clicks on a TQCanvasItem which is a
     * plain TQCanvasText
     */
    void textItemPressed(TQMouseEvent *,
                         TQCanvasItem *);

    /**
     * Emitted when the mouse cursor moves to a different height
     * on the staff
     *
     * \a noteName contains the MIDI name of the corresponding note
     */
    void hoveredOverNoteChanged(const TQString &noteName);

    /**
     * Emitted when the mouse cursor moves to a note which is at a
     * different time
     *
     * \a time is set to the absolute time of the note the cursor is
     * hovering on
     */
    void hoveredOverAbsoluteTimeChanged(unsigned int time);

    /**
     * Emitted when the mouse cursor moves (used by the selection tool)
     */
    void mouseMoved(TQMouseEvent*);

    /**
     * Emitted when the mouse button is released
     */
    void mouseReleased(TQMouseEvent*);

    /**
     * Emitted when a region is about to be drawn by the canvas view.
     * Indicates that any on-demand rendering for that region should
     * be carried out.
     */
    void renderRequired(double cx0, double cx1);

public slots:
     void slotRenderComplete();

    void slotExternalWheelEvent(TQWheelEvent* e);

protected:

    virtual void viewportPaintEvent(TQPaintEvent *e);
    virtual void drawContents(TQPainter *p, int cx, int cy, int cw, int ch);

    const LinedStaffManager &m_linedStaffManager;

    /**
     * Callback for a mouse button press event in the canvas
     */
    virtual void contentsMousePressEvent(TQMouseEvent*);

    /**
     * Callback for a mouse button release event in the canvas
     */
    virtual void contentsMouseReleaseEvent(TQMouseEvent*);

    /**
     * Callback for a mouse move event in the canvas
     */
    virtual void contentsMouseMoveEvent(TQMouseEvent*);

    /**
     * Callback for a mouse double click event in the canvas
     */
    virtual void contentsMouseDoubleClickEvent(TQMouseEvent*);

    void processActiveItems(TQMouseEvent*, TQCanvasItemList);

    void handleMousePress(int height, int staffNo,
                          TQMouseEvent*,
                          NotationElement* pressedNotationElement = 0);

    bool posIsTooFarFromStaff(const TQPoint &pos);

    int getLegerLineCount(int height, bool &offset);

    void setHeightMarkerHeight(TQMouseEvent *e);

    NotationElement *getElementAtXCoord(TQMouseEvent *e);

    //--------------- Data members ---------------------------------

    int m_lastYPosNearStaff;

    unsigned int m_staffLineThreshold; 

    NotationStaff *m_currentStaff;
    int m_currentHeight;

    TQCanvasItemGroup *m_heightMarker;
    TQCanvasLineGroupable *m_vert1;
    TQCanvasLineGroupable *m_vert2;
    std::vector<TQCanvasLineGroupable *> m_legerLines;
    bool m_legerLineOffset;

    bool m_heightTracking;

    TQRect m_lastRender;
};



}

#endif
