/* -*- c-basic-offset: 4 indent-tabs-mode: nil -*- vi:set ts=8 sts=4 sw=4: */

/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.

    This program is Copyright 2000-2008
        Guillaume Laurent   <glaurent@telegraph-road.org>,
        Chris Cannam        <cannam@all-day-breakfast.com>,
        Richard Bown        <richard.bown@ferventsoftware.com>

    The moral rights of Guillaume Laurent, Chris Cannam, and Richard
    Bown to claim authorship of this work have been asserted.

    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/

#ifndef _RG_ROSEGARDENTRANSPORTDIALOG_H_
#define _RG_ROSEGARDENTRANSPORTDIALOG_H_

#include <map>
#include <kdockwidget.h>
#include <tqcolor.h>
#include <tqpixmap.h>
#include "base/Composition.h" // for tempoT
#include "RosegardenTransport.h" // generated by uic

class RosegardenTransport;
class TQWidget;
class TQTimer;
class TQPushButton;
class TQCloseEvent;
class TQAccel;


namespace Rosegarden
{

class TimeSignature;
class RealTime;
class MappedEvent;


class TransportDialog : public TQWidget
{
Q_OBJECT
  TQ_OBJECT
public:
    TransportDialog(TQWidget *parent=0,
                              const char *name=0,
                              WFlags flags = /*TQt::WStyle_StaysOnTop |*/
                                             TQt::WStyle_NormalBorder);
    ~TransportDialog();

    enum TimeDisplayMode { RealMode, SMPTEMode, BarMode, BarMetronomeMode, FrameMode };

    std::string getCurrentModeAsString();
    TimeDisplayMode getCurrentMode() { return m_currentMode; }
    void setNewMode(const std::string& newModeAsString);
    void setNewMode(const TimeDisplayMode& newMode);
    bool isShowingTimeToEnd();
    bool isExpanded();

    void displayRealTime(const RealTime &rt);
    void displaySMPTETime(const RealTime &rt);
    void displayFrameTime(const RealTime &rt);
    void displayBarTime(int bar, int beat, int unit);

    void setTempo(const tempoT &tempo);
    void setTimeSignature(const TimeSignature &timeSig);

    void setSMPTEResolution(int framesPerSecond, int bitsPerFrame);
    void getSMPTEResolution(int &framesPerSecond, int &bitsPerFrame);

    // Called indirectly from the sequencer and from the GUI to
    // show incoming and outgoing MIDI events on the Transport
    //
    void setMidiInLabel(const MappedEvent *mE);
    void setMidiOutLabel(const MappedEvent *mE);

    // Return the accelerator object
    //
    TQAccel* getAccelerators() { return m_accelerators; }

    // RosegardenTransport member accessors
    TQPushButton* MetronomeButton()   { return m_transport->MetronomeButton; }
    TQPushButton* SoloButton()        { return m_transport->SoloButton; }
    TQPushButton* LoopButton()        { return m_transport->LoopButton; }
    TQPushButton* PlayButton()        { return m_transport->PlayButton; }
    TQPushButton* StopButton()        { return m_transport->StopButton; }
    TQPushButton* FfwdButton()        { return m_transport->FfwdButton; }
    TQPushButton* RewindButton()      { return m_transport->RewindButton; }
    TQPushButton* RecordButton()      { return m_transport->RecordButton; }
    TQPushButton* RewindEndButton()   { return m_transport->RewindEndButton; }
    TQPushButton* FfwdEndButton()     { return m_transport->FfwdEndButton; }
    TQPushButton* TimeDisplayButton() { return m_transport->TimeDisplayButton; }
    TQPushButton* ToEndButton()       { return m_transport->ToEndButton; }

    virtual void show();
    virtual void hide();

protected:
    virtual void closeEvent(TQCloseEvent * e);
    void computeSampleRate();
    void cycleThroughModes();
    void displayTime();

public slots:

    // These two slots are activated by TQTimers
    //
    void slotClearMidiInLabel();
    void slotClearMidiOutLabel();

    // These just change the little labels that say what
    // mode we're in, nothing else
    //
    void slotChangeTimeDisplay();
    void slotChangeToEnd();

    void slotLoopButtonClicked();

    void slotPanelOpenButtonClicked();
    void slotPanelCloseButtonClicked();

    void slotEditTempo();
    void slotEditTimeSignature();
    void slotEditTime();

    void slotSetBackground(TQColor);
    void slotResetBackground();

    void slotSetStartLoopingPointAtMarkerPos();
    void slotSetStopLoopingPointAtMarkerPos();

signals:
    void closed();

    // Set and unset the loop at the RosegardenGUIApp
    //
    void setLoop();
    void unsetLoop();
    void setLoopStartTime();
    void setLoopStopTime();

    void editTempo(TQWidget *);
    void editTimeSignature(TQWidget *);
    void editTransportTime(TQWidget *);
    void scrollTempo(int);
    void panic();

private:
    void loadPixmaps();
    void resetFonts();
    void resetFont(TQWidget *);
    void initModeMap();

    //--------------- Data members ---------------------------------

    RosegardenTransport* m_transport;

    std::map<int, TQPixmap> m_lcdList;
    TQPixmap m_lcdNegative;

    int m_lastTenHours;
    int m_lastUnitHours;
    int m_lastTenMinutes;
    int m_lastUnitMinutes;
    int m_lastTenSeconds;
    int m_lastUnitSeconds;
    int m_lastTenths;
    int m_lastHundreths;
    int m_lastThousandths;
    int m_lastTenThousandths;

    bool m_lastNegative;
    TimeDisplayMode m_lastMode;
    TimeDisplayMode m_currentMode;

    int m_tenHours;
    int m_unitHours;
    int m_tenMinutes;
    int m_unitMinutes;
    int m_tenSeconds;
    int m_unitSeconds;
    int m_tenths;
    int m_hundreths;
    int m_thousandths;
    int m_tenThousandths;

    tempoT m_tempo;
    int m_numerator;
    int m_denominator;

    int m_framesPerSecond;
    int m_bitsPerFrame;

    TQTimer *m_midiInTimer;
    TQTimer *m_midiOutTimer;
    TQTimer *m_clearMetronomeTimer;

    TQPixmap m_panelOpen;
    TQPixmap m_panelClosed;

    void updateTimeDisplay();

    TQAccel *m_accelerators;
    bool    m_isExpanded;

    bool m_haveOriginalBackground;
    bool m_isBackgroundSet;
    TQColor m_originalBackground;

    int m_sampleRate;

    std::map<std::string, TimeDisplayMode> m_modeMap;
};

 



}

#endif
