
/* -*- c-basic-offset: 4 indent-tabs-mode: nil -*- vi:set ts=8 sts=4 sw=4: */

/*
    Rosegarden
    A MIDI and audio sequencer and musical notation editor.

    This program is Copyright 2000-2008
        Guillaume Laurent   <glaurent@telegraph-road.org>,
        Chris Cannam        <cannam@all-day-breakfast.com>,
        Richard Bown        <richard.bown@ferventsoftware.com>

    The moral rights of Guillaume Laurent, Chris Cannam, and Richard
    Bown to claim authorship of this work have been asserted.

    Other copyrights also apply to some parts of this work.  Please
    see the AUTHORS file and individual file headers for details.

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of the
    License, or (at your option) any later version.  See the file
    COPYING included with this distribution for more information.
*/

#ifndef _RG_AUDIOPLUGINDIALOG_H_
#define _RG_AUDIOPLUGINDIALOG_H_

#include "base/Instrument.h"
#include "base/MidiProgram.h"
#include <kdialogbase.h>
#include <tqstring.h>
#include <tqstringlist.h>
#include <vector>


class TQWidget;
class TQPushButton;
class TQLabel;
class TQGridLayout;
class TQFrame;
class TQCloseEvent;
class TQCheckBox;
class TQAccel;
class KComboBox;


namespace Rosegarden
{

class PluginControl;
class PluginContainer;
class AudioPluginOSCGUIManager;
class AudioPluginManager;
class AudioPluginInstance;


class AudioPluginDialog : public KDialogBase
{
    Q_OBJECT
  TQ_OBJECT

public:
    AudioPluginDialog(TQWidget *parent,
                      AudioPluginManager *aPM,
#ifdef HAVE_LIBLO
                      AudioPluginOSCGUIManager *aGM,
#endif
                      PluginContainer *instrument,
                      int index);

    PluginContainer* getPluginContainer() const { return m_pluginContainer; }

    TQAccel* getAccelerators() { return m_accelerators; }

    bool isSynth() { return m_index == int(Instrument::SYNTH_PLUGIN_POSITION); }

    void updatePlugin(int number);
    void updatePluginPortControl(int port);
    void updatePluginProgramControl();
    void updatePluginProgramList();
    void guiExited() { m_guiShown = false; }

public slots:
    void slotCategorySelected(int);
    void slotPluginSelected(int index);
    void slotPluginPortChanged(float value);
    void slotPluginProgramChanged(const TQString &value);
    void slotBypassChanged(bool);
    void slotCopy();
    void slotPaste();
    void slotDefault();
    void slotShowGUI();

#ifdef HAVE_LIBLO
    virtual void slotDetails();
#endif

signals:
    void pluginSelected(InstrumentId, int pluginIndex, int plugin);
    void pluginPortChanged(InstrumentId, int pluginIndex, int portIndex);
    void pluginProgramChanged(InstrumentId, int pluginIndex);
    void changePluginConfiguration(InstrumentId, int pluginIndex,
                                   bool global, TQString key, TQString value);
    void showPluginGUI(InstrumentId, int pluginIndex);
    void stopPluginGUI(InstrumentId, int pluginIndex);

    // is the plugin being bypassed
    void bypassed(InstrumentId, int pluginIndex, bool bp);
    void destroyed(InstrumentId, int index);

    void windowActivated();

protected slots:
    virtual void slotClose();

protected:
    virtual void closeEvent(TQCloseEvent *e);
    virtual void windowActivationChange(bool);

    void makePluginParamsBox(TQWidget*, int portCount, int tooManyPorts);
    TQStringList getProgramsForInstance(AudioPluginInstance *inst, int &current);

    //--------------- Data members ---------------------------------

    AudioPluginManager  *m_pluginManager;
#ifdef HAVE_LIBLO
    AudioPluginOSCGUIManager *m_pluginGUIManager;
#endif
    PluginContainer     *m_pluginContainer;
    InstrumentId         m_containerId;

    TQFrame              *m_pluginParamsBox;
    TQWidget             *m_pluginCategoryBox;
    KComboBox           *m_pluginCategoryList;
    TQLabel              *m_pluginLabel;
    KComboBox           *m_pluginList;
    std::vector<int>     m_pluginsInList;
    TQLabel              *m_insOuts;
    TQLabel              *m_pluginId;
    TQCheckBox           *m_bypass;
    TQPushButton         *m_copyButton;
    TQPushButton         *m_pasteButton;
    TQPushButton         *m_defaultButton;
    TQPushButton         *m_guiButton;
    
    TQLabel              *m_programLabel;
    KComboBox           *m_programCombo;
    std::vector<PluginControl*> m_pluginWidgets;
    TQGridLayout         *m_gridLayout;

    int                  m_index;

    bool                 m_generating;
    bool                 m_guiShown;

    TQAccel              *m_accelerators;

    void                 populatePluginCategoryList();
    void                 populatePluginList();
};


} // end of namespace



#endif
