/***************************************************************************
 *   Copyright (C) 2005 Nicolas Hadacek <hadacek@kde.org>                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#include "log_view.h"

#include <tqpopupmenu.h>
#include <tqeventloop.h>
#include <tqapplication.h>
#include <kiconloader.h>

#include "global_config.h"
#include "common/gui/purl_gui.h"
#include "common/gui/misc_gui.h"

Log::Widget::Widget(TQWidget *parent, const char *name)
  : TQTextEdit(parent, name)
{
  setTextFormat(LogText);
  setMinimumWidth(300);
}

void Log::Widget::updateDebugLevel()
{
  setDebugLevel(GlobalConfig::debugLevel());
}

void Log::Widget::logExtra(const TQString &text)
{
  _text += text;
  if ( GlobalConfig::logOutputType()==GuiConsole ) fprintf(stdout, "%s", text.latin1());
}

void Log::Widget::doLog(LineType type, const TQString &text, Action action)
{
  doLog(text, type.data().color, type.data().bold, action);
}

void Log::Widget::doLog(DebugLevel level, const TQString &text, Action action)
{
  doLog(text, level.data().color, false, action);
}

void Log::Widget::doLog(const TQString &text, const TQString &color, bool bold, Action action)
{
  logExtra(text + "\n");
  TQString s = TQString("<font color=%1>").arg(color);
  if (bold) s += "<b>";
  s += escapeXml(text);
  if (bold) s += "</b>";
  s += "</font>";
  TQTextEdit::append(s);
  updateContents(); // #### fix bug in TQt (display is messed up)
  ensureVisible(0, contentsHeight());
  if ( action==Immediate)
    TQApplication::eventLoop()->processEvents(TQEventLoop::ExcludeUserInput);
}

void Log::Widget::appendToLastLine(const TQString &text)
{
  logExtra(text);
  uint p = paragraphs() - 1;
  insertAt(escapeXml(text), p, paragraphLength(p));
  updateContents(); // #### fix bug in TQt (display is messed up)
  ensureVisible(0, contentsHeight());
  // immediately visible...
  TQApplication::eventLoop()->processEvents(TQEventLoop::ExcludeUserInput);
}

TQPopupMenu *Log::Widget::createPopupMenu(const TQPoint &pos)
{
  updateDebugLevel();
  _popup = TQTextEdit::createPopupMenu(pos);
  KIconLoader loader;
  TQIconSet iset = loader.loadIconSet("filesave", KIcon::Small, 0);
  _popup->insertItem(iset, "Save As...", this, TQT_SLOT(saveAs()));
  iset = loader.loadIconSet("fileclose", KIcon::Small, 0);
  _popup->insertItem(iset, "Clear", this, TQT_SLOT(clear()));
  _popup->insertSeparator();
  FOR_EACH(DebugLevel, level) {
    _id[level.type()] = _popup->insertItem(level.label());
    _popup->setItemChecked(_id[level.type()], _debugLevel==level);
  }
  _popup->insertSeparator();
  int id = _popup->insertItem(i18n("Output in console"), this, TQT_SLOT(toggleConsoleOutput()));
  _popup->setItemChecked(id, GlobalConfig::logOutputType()==GuiConsole);
  connect(_popup, TQT_SIGNAL(activated(int)), TQT_SLOT(toggleVisible(int)));
  return _popup;
}

void Log::Widget::toggleVisible(int id)
{
  FOR_EACH(DebugLevel, level) {
    if ( _id[level.type()]==id ) {
      _debugLevel = level;
      GlobalConfig::writeDebugLevel(level);
      break;
    }
  }
}

void Log::Widget::toggleConsoleOutput()
{
  GlobalConfig::writeLogOutputType(GlobalConfig::logOutputType()==GuiOnly ? GuiConsole : GuiOnly);
}

void Log::Widget::sorry(const TQString &message, const TQString &details)
{
  logExtra(message + " [" + details + "]\n");
  MessageBox::detailedSorry(message, details, Log::Show);
}

bool Log::Widget::askContinue(const TQString &message)
{
  bool ok = MessageBox::askContinue(message);
  logExtra(message + " [" + (ok ? "continue" : "cancel") + "]\n");
  return ok;
}

void Log::Widget::clear()
{
  TQTextEdit::clear();
  _text = TQString();
}

void Log::Widget::saveAs()
{
  PURL::Url url = PURL::getSaveUrl(":save_log", "text/x-log", this, i18n("Save log to file"), PURL::AskOverwrite);
  if ( url.isEmpty() ) return;
  url.write(_text, *this);
}
