/***************************************************************************
 *   Copyright (C) 2005 Nicolas Hadacek <hadacek@kde.org>                  *
 *   Copyright (C) 2003-2004 Alain Gibaud <alain.gibaud@free.fr>           *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#ifndef EDITOR_MANAGER_H
#define EDITOR_MANAGER_H

class TQEvent;
#include <ktabbar.h>
#include <ktabwidget.h>
#include <klineedit.h>

#include "text_editor.h"
#include "common/gui/misc_gui.h"
#include "common/gui/dialog.h"

//-----------------------------------------------------------------------------
class SwitchToDialog : public Dialog
{
Q_OBJECT
  TQ_OBJECT
public:
  SwitchToDialog(const TQStringList &names, TQWidget *parent);
  TQString name() const { return _edit->text(); }

private:
  KLineEdit *_edit;
};

//-----------------------------------------------------------------------------
class EditorTabBar : public TabBar
{
Q_OBJECT
  TQ_OBJECT
public:
  EditorTabBar(TQWidget *parent) : TabBar(parent, "editor_tab_bar") {}
  void setReadOnly(uint index, bool readOnly) { _readOnly[tabAt(index)] = readOnly; }

private:
  TQMap<TQTab *, bool> _readOnly;
  virtual void paintLabel(TQPainter *p, const TQRect &br, TQTab *t, bool has_focus) const;
};

//-----------------------------------------------------------------------------
class EditorHistory
{
public:
  EditorHistory() : _current(0) {}
  bool hasBack() const { return _current!=0; }
  bool hasForward() const { return (_current+1)<_names.count(); }
  void add(const TQString &name);
  void closedLast();
  TQString goBack();
  TQString goForward();

private:
  uint _current;
  TQValueVector<TQString> _names;
};

//-----------------------------------------------------------------------------
class EditorManager : public TabWidget
{
Q_OBJECT
  TQ_OBJECT
public:
  EditorManager(TQWidget *parent);

  PURL::UrlList files() const;
  TQValueList<Editor *> &editors() { return _editors; }
  uint nbEditors() const { return _editors.count(); }
  Editor *createEditor(PURL::FileType type, const PURL::Url &url);
  void addEditor(Editor *e);
  Editor *currentEditor() const { return _current; }
  Editor *findEditor(const PURL::Url &file);
  Editor *findEditor(const TQString &tag);
  void showEditor(Editor *e);
  bool closeEditor(const PURL::Url &url);
  bool closeEditor(Editor *e, bool ask);
  bool openFile(const PURL::Url &url);
  Editor *openEditor(const PURL::Url &url);
  void connectEditor(Editor *editor);
  void disconnectEditor(Editor *editor);
  const EditorHistory &history() const { return _history; }
  enum EditorType { DeviceEditor = 0, RegisterEditor, Nb_EditorTypes };
  Editor *openEditor(EditorType type);

public slots:
  void updateTitles();
  void slotDropEvent(TQDropEvent *e);
  void saveAllFiles();
  bool closeCurrentEditor();
  bool closeAllEditors();
  bool closeAllOtherEditors();
  void switchHeaderImplementation();
  void switchToEditor();
  void goBack();
  void goForward();

signals:
  void modified(const PURL::Url &url);
  void guiChanged();
  void statusChanged(const TQString &);

private:
  void changeToEditor(Editor *e);
  void enableActions(bool enable);
  TQString title(const Editor &e) const;
  TQString name(const Editor &e) const;
  virtual void contextMenu(int i, const TQPoint &p);
  void saveBookmarks(const Editor &e);
  void restoreBookmarks(Editor &e);

private slots:
  void showEditor(TQWidget *w) { showEditor(static_cast<Editor *>(w)); }
  void closeRequest(int i);
  void modifiedSlot();

private:
  Editor *_current;
  TQValueList<Editor *> _editors;
  EditorHistory _history;
  static const char * const EDITOR_TAGS[Nb_EditorTypes];
};

#endif
