/***************************************************************************
 *   Copyright (C) 2006 Nicolas Hadacek <hadacek@kde.org>                  *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/
#include "dialog.h"

#include <tqheader.h>
#include <tqtimer.h>
#include <tqlabel.h>
#include <tqwidgetstack.h>
#include <ktextedit.h>

#include "misc_gui.h"

//-----------------------------------------------------------------------------
Dialog::Dialog(TQWidget *parent, const char *name, bool modal,
               const TQString &caption, int buttonMask, ButtonCode defaultButton,
               bool separator, const TQSize &defaultSize)
  : KDialogBase(parent, name, modal, caption, buttonMask, defaultButton, separator),
    _defaultSize(defaultSize)
{
  PBusyCursor::start();
  Q_ASSERT(name);
  TQWidget *main = new TQWidget(this);
  setMainWidget(main);

  TQTimer::singleShot(0, this, TQT_SLOT(updateSize()));
}

Dialog::Dialog(DialogType type, const TQString &caption, int buttonMask, ButtonCode defaultButton,
               TQWidget *parent, const char *name, bool modal, bool separator, const TQSize &defaultSize)
  : KDialogBase(type, caption, buttonMask, defaultButton, parent, name, modal, separator),
    _defaultSize(defaultSize)
{
  PBusyCursor::start();
  Q_ASSERT(name);
  TQTimer::singleShot(0, this, TQT_SLOT(updateSize()));
}

Dialog::~Dialog()
{
  GuiConfig gc;
  gc.writeEntry(TQString(name()) + "_size", size());
}

void Dialog::updateSize()
{
  GuiConfig gc;
  resize(gc.readSizeEntry(TQString(name()) + "_size", &_defaultSize));
  PBusyCursor::stop();
}

//-----------------------------------------------------------------------------
TreeListDialog::Item::Item(const TQString &label, TQWidget *page, const TQString &title, TQListView *listview)
  : KListViewItem(listview, label), _page(page), _title(title)
{}
TreeListDialog::Item::Item(const TQString &label, TQWidget *page, const TQString &title, TQListViewItem *item)
  : KListViewItem(item, label), _page(page), _title(title)
{}

TreeListDialog::TreeListDialog(TQWidget *parent, const char *name, bool modal,
                               const TQString &caption, int buttonMask, ButtonCode defaultButton,
                               bool separator)
  : Dialog(parent, name, modal, caption, buttonMask, defaultButton, separator)
{
  TQVBoxLayout *top = new TQVBoxLayout(mainWidget(), 0, 10);

  // list view
  TQValueList<int> widths;
  widths += 80;
  widths += 500;
  Splitter *splitter = new Splitter(widths,Qt::Horizontal, mainWidget(), name);
  top->addWidget(splitter);
  _listView = new KListView(splitter);
  connect(_listView, TQT_SIGNAL(currentChanged(TQListViewItem *)), TQT_SLOT(currentChanged(TQListViewItem *)));
  _listView->setAllColumnsShowFocus(true);
  _listView->setRootIsDecorated(true);
  _listView->setSorting(0);
  _listView->addColumn(TQString());
  _listView->header()->hide();
  _listView->setResizeMode(TQListView::LastColumn);

  // pages
  _frame = new TQFrame(splitter);
  TQVBoxLayout *vbox = new TQVBoxLayout(_frame, 10, 10);
  _titleBox = new TQHBoxLayout(vbox);
  _label = new TQLabel(_frame);
  _titleBox->addWidget(_label);
  _stack = new TQWidgetStack(_frame);
  connect(_stack, TQT_SIGNAL(aboutToShow(TQWidget *)), TQT_SIGNAL(aboutToShowPage(TQWidget *)));
  vbox->addWidget(_stack);
  vbox->addStretch(1);
}

TQWidget *TreeListDialog::addPage(const TQStringList &labels)
{
  Q_ASSERT( !labels.isEmpty() );

  TQWidget *page = 0;
  TQListViewItem *item = 0;
  TQListViewItemIterator it(_listView);
  for (; it.current(); ++it) {
    if ( it.current()->text(0)==labels[0] ) {
      item = it.current();
      break;
    }
  }
  if ( item==0 ) {
    page = new TQWidget(_stack);
    connect(page, TQT_SIGNAL(destroyed(TQObject *)), TQT_SLOT(pageDestroyed(TQObject *)));
    _stack->addWidget(page);
    item = new Item(labels[0], page, labels[0], _listView);
    item->setOpen(true);
    bool last = ( labels.count()==1 );
    item->setSelectable(last);
  }
  for (uint i=1; i<labels.count(); i++) {
    TQListViewItem *parent = item;
    item = 0;
    TQListViewItemIterator iti(parent);
    for (; it.current(); ++it) {
      if ( it.current()->text(0)==labels[i] ) {
        item = it.current();
        break;
      }
    }
    if ( item==0 ) {
      page = new TQWidget(_stack);
      connect(page, TQT_SIGNAL(destroyed(TQObject *)), TQT_SLOT(pageDestroyed(TQObject *)));
      _stack->addWidget(page);
      item = new Item(labels[i], page, labels[i], parent);
      item->setOpen(true);
      bool last = ( labels.count()==i+1 );
      item->setSelectable(last);
    }
  }

  return page;
}

void TreeListDialog::currentChanged(TQListViewItem *lvitem)
{
  if ( lvitem==0 ) return;
  Item *item = static_cast<Item *>(lvitem);
  _listView->ensureItemVisible(item);
  _label->setText(item->_title);
  _stack->raiseWidget(item->_page);
}

void TreeListDialog::showPage(TQWidget *page)
{
  TQListViewItemIterator it(_listView);
  for (; it.current(); ++it) {
    Item *item = static_cast<Item *>(it.current());
    if ( item->_page==page ) {
      _listView->setCurrentItem(item);
      currentChanged(item);
      break;
    }
  }
}

int TreeListDialog::pageIndex(TQWidget *page) const
{
  return _stack->id(page);
}

int TreeListDialog::activePageIndex() const
{
  const Item *item = static_cast<const Item *>(_listView->currentItem());
  if ( item==0 ) return -1;
  return pageIndex(item->_page);
}

void TreeListDialog::pageDestroyed(TQObject *object)
{
  TQListViewItemIterator it(_listView);
  for (; it.current(); ++it) {
    Item *item = static_cast<Item *>(it.current());
    if ( TQT_BASE_OBJECT(item->_page)!=TQT_BASE_OBJECT(object) ) continue;
    delete item;
    break;
  }
}

//-----------------------------------------------------------------------------
TextEditorDialog::TextEditorDialog(const TQString &text, const TQString &caption,
                                   bool wrapAtWidgetWidth, TQWidget *parent)
  : Dialog(parent, "text_editor_dialog", true, caption, Close, Close, false, TQSize(200, 100))
{
  KTextEdit *w = new KTextEdit(text, TQString(), this);
  w->setReadOnly(true);
  w->setWordWrap(wrapAtWidgetWidth ? TQTextEdit::WidgetWidth : TQTextEdit::NoWrap);
  setMainWidget(w);
}
