/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
*/

/*
  Copyright (C) 2006-2008 Eike Hein <hein@kde.org>
*/

#ifndef VIEWCONTAINER_H
#define VIEWCONTAINER_H

#include "konversationmainwindow.h"
#include "common.h"
#include "server.h"

#include <tqobject.h>
#include <tqguardedptr.h>


class TQSplitter;

class KTabWidget;
class KActionCollection;

class KonversationMainWindow;
class ViewTree;
class ChatWindow;
class Server;
class Images;
class UrlCatcher;
class DccTransferPanel;
class NicksOnline;
class QueueTuner;

namespace Konversation
{
    class InsertCharDialog;
    class ServerGroupSettings;
}

class ViewContainer : public TQObject
{
    Q_OBJECT
  TQ_OBJECT

    public:
        explicit ViewContainer(KonversationMainWindow* window);
        ~ViewContainer();

        TQSplitter* getWidget() { return m_viewTreeSplitter; }
        KonversationMainWindow* getWindow() { return m_window; }
        KActionCollection* actionCollection() { return m_window->actionCollection(); }

        TQGuardedPtr<ChatWindow> getFrontView() { return m_frontView; }
        Server* getFrontServer() { return m_frontServer; }

        void prepareShutdown();

        TQString currentViewTitle();
        TQString currentViewURL(bool passNetwork);

        void appendToFrontmost(const TQString& type,const TQString& message,ChatWindow* serverView,
                               bool parseURL = true);

        void showQueueTuner(bool);

        int getViewIndex(TQWidget* widget);

    public slots:
        void updateAppearance();
        void saveSplitterSizes();
        void setViewTreeShown(bool show);
        void syncTabBarToTree();

        void updateViews(const Konversation::ServerGroupSettings* serverGroup = 0);
        void updateViewIcons();
        void setViewNotification(ChatWindow* widget, const Konversation::TabNotifyType& type);
        void unsetViewNotification(ChatWindow* view);
        void toggleViewNotifications();
        void toggleAutoJoin();

        void switchView(TQWidget* newView);
        void showView(ChatWindow* view);

        void goToView(int page);
        void showNextView();
        void showPreviousView();
        void moveViewLeft();
        void moveViewRight();

        void closeView(TQWidget* view);
        void closeView(ChatWindow* view);
        void closeViewMiddleClick(TQWidget* view);
        void closeCurrentView();
        void cleanupAfterClose(ChatWindow* view);

        void changeViewCharset(int index);
        void updateViewEncoding(ChatWindow* view);

        void showViewContextMenu(TQWidget* tab, const TQPoint& pos);

        void clearView();
        void clearAllViews();

        void findText();
        void findNextText();
        void findPrevText();

        void insertCharacter();
        void insertChar(const TQChar& chr);
        void insertIRCColor();

        void clearViewLines();
        void insertRememberLine();
        void cancelRememberLine();
        void insertMarkerLine();
        void insertRememberLines(Server* server);

        void openLogFile();
        void openLogFile(const TQString& caption, const TQString& file);

        void addKonsolePanel();

        void addUrlCatcher();
        void closeUrlCatcher();

        void toggleDccPanel();
        void addDccPanel();
        void closeDccPanel();
        void deleteDccPanel();
        DccTransferPanel* getDccPanel();

        void addDccChat(const TQString& myNick,const TQString& nick,const TQStringList& arguments,bool listen);

        StatusPanel* addStatusView(Server* server);
        RawLog* addRawLog(Server* server);
        void disconnectFrontServer();
        void reconnectFrontServer();
        void showJoinChannelDialog();
        void connectionStateChanged(Server* server, Konversation::ConnectionState state);
        void channelJoined(Channel* channel);

        Channel* addChannel(Server* server, const TQString& name);
        void rejoinChannel();
        void openChannelSettings();
        void toggleChannelNicklists();

        Query* addQuery(Server* server,const NickInfoPtr & name, bool weinitiated=true);
        void updateQueryChrome(ChatWindow* view, const TQString& name);
        void closeQueries();

        ChannelListPanel* addChannelListPanel(Server* server);
        void openChannelList(const TQString& filter = TQString(), bool getList = false);

        void openNicksOnlinePanel();
        void closeNicksOnlinePanel();

        void showNextActiveView();

    signals:
        void viewChanged(ChatWindow* view);
        void removeView(ChatWindow* view);
        void setWindowCaption(const TQString& caption);
        void updateChannelAppearance();
        void contextMenuClosed();
        void resetStatusBar();
        void setStatusBarTempText(const TQString& text);
        void clearStatusBarTempText();
        void setStatusBarInfoLabel(const TQString& text);
        void clearStatusBarInfoLabel();
        void setStatusBarLagLabelShown(bool shown);
        void updateStatusBarLagLabel(Server* server, int msec);
        void resetStatusBarLagLabel();
        void setStatusBarLagLabelTooLongLag(Server* server, int msec);
        void updateStatusBarSSLLabel(Server* server);
        void removeStatusBarSSLLabel();
        void autoJoinToggled(const Konversation::ServerGroupSettings*);

        void frontServerChanging(Server *);

    private:
        void setupTabWidget();
        void setupViewTree();
        void removeViewTree();
        void updateTabWidgetAppearance();

        void addView(ChatWindow* view, const TQString& label, bool weinitiated=true);

        void updateViewActions(int index);
        void updateSwitchViewAction();
        void updateFrontView();

        void setFrontServer(Server *);

        void initializeSplitterSizes();
        bool m_saveSplitterSizesLock;

        KonversationMainWindow* m_window;

        TQSplitter* m_viewTreeSplitter;
        KTabWidget* m_tabWidget;
        ViewTree* m_viewTree;
        TQVBox *m_vbox;
        QueueTuner *m_queueTuner;

        Images* images;

        TQGuardedPtr<Server> m_frontServer;
        TQGuardedPtr<Server> m_contextServer;
        TQGuardedPtr<ChatWindow> m_frontView;
        TQGuardedPtr<ChatWindow> m_searchView;

        UrlCatcher* m_urlCatcherPanel;
        NicksOnline* m_nicksOnlinePanel;

        DccTransferPanel* m_dccPanel;
        bool m_dccPanelOpen;

        Konversation::InsertCharDialog* m_insertCharDialog;

        int m_popupViewIndex;
        int m_queryViewCount;

        TQValueList<ChatWindow*> m_activeViewOrderList;
};

#endif
