/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
*/

/*
  Copyright (C) 2006 Dario Abatianni <eisfuchs@tigress.com>
  Copyright (C) 2006 John Tapsell <johnflux@gmail.com>
*/

#include "alias_preferences.h"
#include "config/preferences.h"

#include <tqlabel.h>
#include <tqpushbutton.h>
#include <tqheader.h>

#include <kapplication.h>
#include <kdebug.h>
#include <kconfig.h>
#include <klocale.h>
#include <klineedit.h>
#include <klistview.h>


Alias_Config::Alias_Config(TQWidget* parent, const char* name)
 : Alias_ConfigUI(parent, name)
{
  // reset flag to defined state (used to block signals when just selecting a new item)
  m_newItemSelected = false;

  // populate listview
  loadSettings();

  // make items react to drag & drop
  aliasListView->setSorting(-1,false);
  aliasListView->header()->setMovingEnabled(false);

  connect(aliasListView, TQT_SIGNAL(selectionChanged(TQListViewItem*)), this, TQT_SLOT(entrySelected(TQListViewItem*)));
  connect(aliasListView, TQT_SIGNAL(clicked(TQListViewItem*)), this, TQT_SLOT(entrySelected(TQListViewItem*)) );
  connect(aliasListView, TQT_SIGNAL(moved()), this, TQT_SIGNAL(modified()));

  connect(aliasInput, TQT_SIGNAL(textChanged(const TQString&)), this, TQT_SLOT(nameChanged(const TQString&)));
  connect(replacementInput, TQT_SIGNAL(textChanged(const TQString&)), this, TQT_SLOT(actionChanged(const TQString&)));

  connect(newButton, TQT_SIGNAL(clicked()), this, TQT_SLOT(addEntry()));
  connect(removeButton, TQT_SIGNAL(clicked()), this, TQT_SLOT(removeEntry()));
}

Alias_Config::~Alias_Config()
{
}

void Alias_Config::loadSettings()
{
    setAliasListView(Preferences::aliasList());
}

void Alias_Config::saveSettings()
{
    TQStringList newList=currentAliasList();
    Preferences::setAliasList(newList);

    // saved list is now old list, to check for changes
    m_oldAliasList=newList;
}

void Alias_Config::restorePageToDefaults()
{
    aliasListView->clear();
    setAliasListView(Preferences::defaultAliasList());
}

bool Alias_Config::hasChanged()
{
    return (currentAliasList() != m_oldAliasList);
}

void Alias_Config::setAliasListView(const TQStringList& aliasList)
{
    aliasListView->clear();

    // Insert alias items backwards to get them sorted properly
    for(int index=aliasList.count(); index!=0; index--)
    {
        TQString item=aliasList[index-1];
        new KListViewItem(aliasListView,item.section(' ',0,0),item.section(' ',1));
    }

    aliasListView->setSelected(aliasListView->firstChild(), true);
    // remember alias list
    m_oldAliasList=aliasList;
}

TQStringList Alias_Config::currentAliasList()
{
    TQStringList newList;

    TQListViewItem* item=aliasListView->itemAtIndex(0);
    while(item)
        {
        newList.append(item->text(0)+' '+item->text(1));
        item=item->itemBelow();
        }
    return newList;
}

// what to do when the user selects an item
void Alias_Config::entrySelected(TQListViewItem* aliasEntry)
{
    // play it safe, assume disabling all widgets first
    bool enabled = false;

    // check if there really was an item selected
    if (aliasEntry)
    {
        // remember to enable the editing widgets
        enabled = true;
        // tell the editing widgets not to emit modified() on signals now
        m_newItemSelected = true;
        // update editing widget contents
        aliasInput->setText(aliasEntry->text(0));
        replacementInput->setText(aliasEntry->text(1));
        // re-enable modified() signal on text changes in edit widgets
        m_newItemSelected = false;
    }
    // enable or disable editing widgets
    removeButton->setEnabled(enabled);
    aliasLabel->setEnabled(enabled);
    aliasInput->setEnabled(enabled);
    replacementLabel->setEnabled(enabled);
    replacementInput->setEnabled(enabled);
}

// what to do when the user change the name of a quick button
void Alias_Config::nameChanged(const TQString& newName)
{
    // get possible first selected item
    TQListViewItem* item = aliasListView->selectedItem();

    // sanity check
    if (item)
    {
        // rename item
        item->setText(0,newName);
        // tell the config system that something has changed
        if (!m_newItemSelected) emit modified();
    }
}

// what to do when the user change the action definition of a quick button
void Alias_Config::actionChanged(const TQString& newAction)
{
    // get possible first selected item
    TQListViewItem* item = aliasListView->selectedItem();

    // sanity check
    if (item)
    {
        // rename item
        item->setText(1,newAction);
        // tell the config system that something has changed
        if(!m_newItemSelected) emit modified();
    }
}

// add button pressed
void Alias_Config::addEntry()
{
    // add new item at the bottom of list view
    KListViewItem* newItem = new KListViewItem(aliasListView,aliasListView->lastChild(),i18n("New"),TQString());
    // if successful ...
    if (newItem)
    {
        // select new item and make it the current one
        aliasListView->setSelected(newItem,true);
        aliasListView->setCurrentItem(newItem);
        // set input focus on item name edit
        aliasInput->setFocus();
        // select all text to make overwriting easier
        aliasInput->selectAll();
        // tell the config system that something has changed
        emit modified();
    }
}

// remove button pressed
void Alias_Config::removeEntry()
{
    // get possible first selected item
    TQListViewItem* item = aliasListView->selectedItem();

    // sanity check
    if (item)
    {
        // get item below the current one
        TQListViewItem* nextItem = item->itemBelow();
        // if there was none, get the one above
        if(!nextItem) nextItem = item->itemAbove();

        // remove the item from the list
        delete item;

        // check if we found the next item
        if (nextItem)
        {
            // select the item and make it the current ite,
            aliasListView->setSelected(nextItem,true);
            aliasListView->setCurrentItem(nextItem);
        }
        else
        {
            // no next item found, this means the list is empty
            entrySelected(0);
        }
        // tell the config system that somethig has changed
        emit modified();
    }
}

#include "alias_preferences.moc"
