/***************************************************************************
 * scriptaction.cpp
 * This file is part of the KDE project
 * copyright (C) 2005 by Sebastian Sauer (mail@dipe.org)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * You should have received a copy of the GNU Library General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "scriptaction.h"
#include "manager.h"

#include <tqstylesheet.h>
#include <tqdir.h>
#include <tqfile.h>
#include <tqfileinfo.h>
#include <kurl.h>
#include <kstandarddirs.h>
#include <kmimetype.h>

using namespace Kross::Api;

namespace Kross { namespace Api {

    /// @internal
    class ScriptActionPrivate
    {
        public:
            /**
            * The packagepath is the directory that belongs to this
            * \a ScriptAction instance. If this \a ScriptAction points
            * to a scriptfile the packagepath will be the directory
            * the scriptfile is located in.
            */
            TQString packagepath;

            /**
            * List of logs this \a ScriptAction has. Initialization,
            * execution and finalization should be logged for
            * example. So, the logs are usuabled to provide some
            * more detailed visual information to the user what
            * our \a ScriptAction did so far.
            */
            TQStringList logs;

            /**
            * The versionnumber this \a ScriptAction has. We are using
            * the version to handle \a ScriptAction instances which
            * have the same unique \a ScriptAction::name() . If the name
            * is the same, we are able to use the version to determinate
            * which \a ScriptAction is newer / replaces the other.
            */
            int version;

            /**
            * The description used to provide a way to the user to describe
            * the \a ScriptAction with a longer string.
            */
            TQString description;

            /**
            * List of \a ScriptActionCollection instances this \a ScriptAction
            * is attached to.
            */
            TQValueList<ScriptActionCollection*> collections;

            /**
            * Constructor.
            */
            explicit ScriptActionPrivate() : version(0) {}
    };

}}

ScriptAction::ScriptAction(const TQString& file)
    : KAction(0, file.latin1())
    , Kross::Api::ScriptContainer(file)
    , d( new ScriptActionPrivate() ) // initialize d-pointer class
{
    KURL url(file);
    if(url.isLocalFile()) {
        setFile(file);
        setText(url.fileName());
        setIcon(KMimeType::iconForURL(url));
    }
    else {
        setText(file);
    }

    setDescription(file);
    setEnabled(false);
}

ScriptAction::ScriptAction(const TQString& scriptconfigfile, const TQDomElement& element)
    : KAction()
    , Kross::Api::ScriptContainer()
    , d( new ScriptActionPrivate() ) // initialize d-pointer class
{
    TQString name = element.attribute("name");
    TQString text = element.attribute("text");
    TQString description = element.attribute("description");
    TQString file = element.attribute("file");
    TQString icon = element.attribute("icon");

    TQString version = element.attribute("version");
    bool ok;
    int v = version.toInt(&ok);
    if(ok) d->version = v;

    if(file.isEmpty()) {
        if(text.isEmpty())
            text = name;
    }
    else {
        if(name.isEmpty())
            name = file;
        if(text.isEmpty())
            text = file;
    }

    //d->scriptcontainer = Manager::scriptManager()->getScriptContainer(name);

    TQString interpreter = element.attribute("interpreter");
    if(interpreter.isNull())
        setEnabled(false);
    else
        setInterpreterName( interpreter );

    if(file.isNull()) {
        setCode( element.text().stripWhiteSpace() );
        if(description.isNull())
            description = text;
        ScriptContainer::setName(name);
    }
    else {
        TQDir dir = TQFileInfo(scriptconfigfile).dir(true);
        d->packagepath = dir.absPath();
        TQFileInfo fi(dir, file);
        file = fi.absFilePath();
        setEnabled(fi.exists());
        setFile(file);
        if(icon.isNull())
            icon = KMimeType::iconForURL( KURL(file) );
        if(description.isEmpty())
            description = TQString("%1<br>%2").arg(text.isEmpty() ? name : text).arg(file);
        else
            description += TQString("<br>%1").arg(file);
        ScriptContainer::setName(file);
    }

    KAction::setName(name.latin1());
    KAction::setText(text);
    setDescription(description);
    KAction::setIcon(icon);

    // connect signal
    connect(this, TQT_SIGNAL(activated()), this, TQT_SLOT(activate()));
}

ScriptAction::~ScriptAction()
{
    detachAll();
    delete d;
}

int ScriptAction::version() const
{
    return d->version;
}

const TQString ScriptAction::getDescription() const
{
    return d->description;
}

void ScriptAction::setDescription(const TQString& description)
{
    d->description = description;
    setToolTip( description );
    setWhatsThis( description );
}

void ScriptAction::setInterpreterName(const TQString& name)
{
    setEnabled( Manager::scriptManager()->hasInterpreterInfo(name) );
    Kross::Api::ScriptContainer::setInterpreterName(name);
}

const TQString ScriptAction::getPackagePath() const
{
    return d->packagepath;
}

const TQStringList& ScriptAction::getLogs() const
{
    return d->logs;
}

void ScriptAction::attach(ScriptActionCollection* collection)
{
    d->collections.append( collection );
}

void ScriptAction::detach(ScriptActionCollection* collection)
{
    d->collections.remove( collection );
}

void ScriptAction::detachAll()
{
    for(TQValueList<ScriptActionCollection*>::Iterator it = d->collections.begin(); it != d->collections.end(); ++it)
        (*it)->detach( this );
}

void ScriptAction::activate()
{
    emit activated(this);
    Kross::Api::ScriptContainer::execute();
    if( Kross::Api::ScriptContainer::hadException() ) {
        TQString errormessage = Kross::Api::ScriptContainer::getException()->getError();
        TQString tracedetails = Kross::Api::ScriptContainer::getException()->getTrace();
        d->logs << TQString("<b>%1</b><br>%2")
                   .arg( TQStyleSheet::escape(errormessage) )
                   .arg( TQStyleSheet::escape(tracedetails) );
        emit failed(errormessage, tracedetails);
    }
    else {
        emit success();
    }
}

void ScriptAction::finalize()
{
    Kross::Api::ScriptContainer::finalize();
}

#include "scriptaction.moc"
