/* This file is part of the KDE project
   Copyright (C) 1998, 1999 Reginald Stadlbauer <reggie@kde.org>
   Copyright (C) 2002, 2003 Ariya Hidayat <ariya@kde.org>
   Copyright (C) 2004, 2005 Laurent Montel <montel@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include "KPrPgConfDia.h"
#include "KPrDocument.h"
#include "KPrPage.h"

#include <tqbuttongroup.h>
#include <tqhbuttongroup.h>
#include <tqcheckbox.h>
#include <tqcombobox.h>
#include <tqhbox.h>
#include <tqvbox.h>
#include <tqvgroupbox.h>
#include <tqheader.h>
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqlistview.h>
#include <tqpen.h>
#include <tqpushbutton.h>
#include <tqradiobutton.h>
#include <tqvaluelist.h>
#include <tqvbuttongroup.h>
#include <tqwhatsthis.h>

#include <kcolorbutton.h>
#include <kglobal.h>
#include <klocale.h>
#include <knuminput.h>
#include <tqslider.h>

KPrPgConfDia::KPrPgConfDia( TQWidget* parent, KPrDocument* doc )
    : KDialogBase( KDialogBase::Tabbed, i18n("Configure Slide Show"),
                   Ok|Cancel, Ok, parent, "pgConfDia", true ),
      m_doc( doc )
{
    setupPageGeneral();
    setupPageSlides();

    connect( this, TQT_SIGNAL( okClicked() ), this, TQT_SLOT( confDiaOk() ) );
    connect( this, TQT_SIGNAL( okClicked() ), this, TQT_SLOT( accept() ) );
}

void KPrPgConfDia::setupPageGeneral()
{
    TQFrame* generalPage = addPage( i18n("&General") );
    TQWhatsThis::add( generalPage, i18n("<p>This dialog allows you to configure how the slideshow "
				       "will be displayed, including whether the slides are "
				       "automatically sequenced or manually controlled, and also "
				       "allows you to configure a <em>drawing pen</em> that can "
				       "be used during the display of the presentation to add "
				       "additional information or to emphasise particular points.</p>") );
    TQVBoxLayout *generalLayout = new TQVBoxLayout( generalPage, 0, KDialog::spacingHint() );

    TQVButtonGroup *switchGroup = new TQVButtonGroup( i18n("&Transition Type"), generalPage );
    generalLayout->addWidget( switchGroup );
    TQWhatsThis::add( switchGroup, i18n("<li><p>If you select <b>Manual transition to next step or slide</b> "
					  "then each transition and effect on a slide, or transition from "
					  "one slide to the next, will require an action. Typically this "
					  "action will be a mouse click, or the space bar.</p></li>"
					  "<li><p>If you select <b>Automatic transition to next step or slide</b> "
					  "then the presentation will automatically sequence each transition "
					  "and effect on a slide, and will automatically transition to the "
					  "next slide when the current slide is fully displayed. The speed "
					  "of sequencing is controlled using the slider below. This also "
					  "enables the option to automatically loop back to the first "
					  "slide after the last slide has been shown.</p></li>") );
    m_manualButton = new TQRadioButton( i18n("&Manual transition to next step or slide"), switchGroup );
    m_manualButton->setChecked( m_doc->spManualSwitch() );
    m_autoButton = new TQRadioButton( i18n("&Automatic transition to next step or slide"), switchGroup );
    m_autoButton->setChecked( !m_doc->spManualSwitch() );

    infiniteLoop = new TQCheckBox( i18n( "&Infinite loop" ), generalPage );
    generalLayout->addWidget( infiniteLoop );
    TQWhatsThis::add( infiniteLoop, i18n("<p>If this checkbox is selected, then the slideshow "
					"will restart at the first slide after the last slide "
					"has been displayed. It is only available if the "
					"<b>Automatic transition to next step or slide</b> "
					"button is selected above.</p> <p>This option may be "
					"useful if you are running a promotional display.</p>") );

    infiniteLoop->setEnabled( !m_doc->spManualSwitch() );
    connect( m_autoButton, TQT_SIGNAL( toggled(bool) ), infiniteLoop, TQT_SLOT( setEnabled(bool) ) );
    connect( m_autoButton, TQT_SIGNAL( toggled(bool) ), infiniteLoop, TQT_SLOT( setChecked(bool) ) );

    endOfPresentationSlide = new TQCheckBox( i18n( "&Show 'End of presentation' slide" ), generalPage );
    generalLayout->addWidget( endOfPresentationSlide );
    TQWhatsThis::add( endOfPresentationSlide, i18n("<p>If this checkbox is selected, when the slideshow "
					"has finished a black slideshow containing the "
					"message 'End of presentation. Click to exit' will "
					"be shown.") );
    endOfPresentationSlide->setChecked( m_doc->spShowEndOfPresentationSlide() );
    endOfPresentationSlide->setDisabled( infiniteLoop->isEnabled() && getInfiniteLoop() );
    connect( infiniteLoop, TQT_SIGNAL( toggled(bool) ), endOfPresentationSlide, TQT_SLOT( setDisabled(bool) ) );

    presentationDuration = new TQCheckBox( i18n( "Measure presentation &duration" ), generalPage );
    generalLayout->addWidget( presentationDuration );
    TQWhatsThis::add( presentationDuration, i18n("<p>If this checkbox is selected, the time that "
						"each slide was displayed for, and the total time "
						"for the presentation will be measured.</p> "
						"<p>The times will be displayed at the end of the "
						"presentation.</p> "
						"<p>This can be used during rehearsal to check "
						"coverage for each issue in the presentation, "
						"and to verify that the presentation duration "
						"is correct.</p>" ) );
    presentationDuration->setChecked( m_doc->presentationDuration() );

    // presentation pen (color and width)

    TQGroupBox* penGroup = new TQGroupBox( 2, Qt::Horizontal, i18n("Presentation Pen") , generalPage );
    generalLayout->addWidget( penGroup );
    TQWhatsThis::add( penGroup, i18n("<p>This part of the dialog allows you to configure the "
				    "<em>drawing mode</em>, which allows you to add additional "
				    "information, emphasise particular content, or to correct "
				    "errors during the presentation by drawing on the slides "
				    "using the mouse.</p>"
				    "<p>You can configure the color of the drawing pen and the "
				    "width of the pen.</p>" ) );
    penGroup->layout()->setSpacing(KDialog::marginHint());
    penGroup->layout()->setMargin(KDialog::spacingHint());
    //TQGridLayout *grid = new TQGridLayout(penGroup->layout(), 3, 2 );

    TQLabel* label = new TQLabel( i18n( "Color:" ), penGroup );
    //grid->addWidget( label, 0, 0 );
    penColor = new KColorButton( m_doc->presPen().color(), m_doc->presPen().color(), penGroup );
    //grid->addWidget( penColor, 0, 1 );

    label = new TQLabel( i18n( "Width:" ), penGroup );
    // grid->addWidget( label, 1, 0 );
    penWidth = new TQSpinBox( 1, 10, 1, penGroup );
    penWidth->setSuffix( i18n(" pt") );
    penWidth->setValue( m_doc->presPen().width() );
    //grid->addWidget( penWidth, 1, 1 );

    generalLayout->addStretch();
}

void KPrPgConfDia::setupPageSlides()
{
    TQFrame* slidesPage = addPage( i18n("&Slides") );
    TQWhatsThis::add( slidesPage, i18n("<p>This dialog allows you to configure which slides "
				      "are used in the presentation. Slides that are not "
				      "selected will not be displayed during the slide "
				      "show.</p>") );
    TQGridLayout *slidesLayout = new TQGridLayout( slidesPage,7 , 2, 0, KDialog::spacingHint());


    TQButtonGroup *group=new TQVButtonGroup( slidesPage );
    group->setRadioButtonExclusive( true );

    m_customSlide = new TQRadioButton( i18n( "Custom slide show" ), group, "customslide" );

    connect( m_customSlide, TQT_SIGNAL( clicked () ), this, TQT_SLOT( radioButtonClicked() ) );

    TQHBox *box = new TQHBox( group );

    m_labelCustomSlide = new TQLabel( i18n( "Custom slide:" ),box );

    m_customSlideCombobox = new TQComboBox( box );
    m_customSlideCombobox->insertStringList( m_doc->presentationList() );

    m_selectedSlide = new TQRadioButton( i18n( "Selected pages:" ), group, "selectedslide" );
    slidesLayout->addMultiCellWidget( group, 0,2,0,1 );
    connect( m_selectedSlide, TQT_SIGNAL( clicked () ), this, TQT_SLOT( radioButtonClicked() ) );

    slides = new TQListView( slidesPage );
    slidesLayout->addMultiCellWidget( slides, 3, 3, 0, 1 );
    slidesLayout->setRowStretch( 3, 10 );
    slides->addColumn( i18n("Slide") );
    slides->setSorting( -1 );
    slides->header()->hide();

    for ( int i = m_doc->getPageNums() - 1; i >= 0; --i )
    {
        KPrPage *page=m_doc->pageList().at( i );
        TQCheckListItem* item = new TQCheckListItem( slides,
                                                   page->pageTitle(),
                                                   TQCheckListItem::CheckBox );
        item->setOn( page->isSlideSelected() );
    }

    TQHBox* buttonGroup = new TQHBox( slidesPage );
    buttonGroup->setSpacing( KDialog::spacingHint() );

    TQPushButton* selectAllButton = new TQPushButton( i18n( "Select &All" ), buttonGroup );
    connect( selectAllButton, TQT_SIGNAL( clicked() ), this, TQT_SLOT( selectAllSlides() ) );

    TQPushButton* deselectAllButton = new TQPushButton( i18n( "&Deselect All" ), buttonGroup );
    connect( deselectAllButton, TQT_SIGNAL( clicked() ), this, TQT_SLOT( deselectAllSlides() ) );

    TQWidget* spacer = new TQWidget( buttonGroup );

    spacer->setSizePolicy( TQSizePolicy( TQSizePolicy::Minimum, TQSizePolicy::Expanding ) );
    slidesLayout->addMultiCellWidget( buttonGroup, 4, 4, 0, 1 );

    if ( !m_doc->presentationName().isEmpty() )
    {
        m_customSlide->setChecked( true );
        m_customSlideCombobox->setCurrentText( m_doc->presentationName() );
    }
    else
        m_selectedSlide->setChecked( true );

    if ( m_customSlideCombobox->count()==0 )
    {
        m_customSlide->setEnabled( false );
        m_labelCustomSlide->setEnabled( false );
        m_customSlideCombobox->setEnabled( false );
    }
    radioButtonClicked();
}

KPrPgConfDia::~KPrPgConfDia()
{
}

void KPrPgConfDia::radioButtonClicked()
{
    if ( m_customSlide->isChecked() )
    {
        m_labelCustomSlide->setEnabled( true );
        m_customSlideCombobox->setEnabled( true );
        slides->setEnabled( false );
    }
    else
    {
        m_labelCustomSlide->setEnabled( false );
        m_customSlideCombobox->setEnabled( false );
        slides->setEnabled( true );
    }
}

bool KPrPgConfDia::getInfiniteLoop() const
{
    return infiniteLoop->isChecked();
}

bool KPrPgConfDia::getShowEndOfPresentationSlide() const
{
    return endOfPresentationSlide->isChecked();
}

bool KPrPgConfDia::getManualSwitch() const
{
    return m_manualButton->isChecked();
}

bool KPrPgConfDia::getPresentationDuration() const
{
    return presentationDuration->isChecked();
}

TQPen KPrPgConfDia::getPen() const
{
    return TQPen( penColor->color(), penWidth->value() );
}

TQValueList<bool> KPrPgConfDia::getSelectedSlides() const
{
    TQValueList<bool> selectedSlides;

    TQListViewItem *item = slides->firstChild();
    while( item )
    {
        TQCheckListItem *checkItem = dynamic_cast<TQCheckListItem*>( item );
        bool selected = false;
        if( checkItem ) selected = checkItem->isOn();
        item = item->nextSibling();
        selectedSlides.append( selected );
    }
    return selectedSlides;
}

void KPrPgConfDia::selectAllSlides()
{
    TQListViewItem *item = slides->firstChild();
    while( item )
    {
        TQCheckListItem *checkItem = dynamic_cast<TQCheckListItem*>( item );
        if( checkItem ) checkItem->setOn( true );
        item = item->nextSibling();
    }
}

void KPrPgConfDia::deselectAllSlides()
{
    TQListViewItem *item = slides->firstChild();
    while( item )
    {
        TQCheckListItem *checkItem = dynamic_cast<TQCheckListItem*>( item );
        if( checkItem ) checkItem->setOn( false );
        item = item->nextSibling();
    }
}

TQString KPrPgConfDia::presentationName() const
{
    if ( m_customSlide->isChecked() )
        return m_customSlideCombobox->currentText();
    else
        return TQString();
}

#include "KPrGradient.h"
#include "KPrPgConfDia.moc"
