/* This file is part of the KDE project
   Copyright (C) 2005 Cedric Pasteur <cedric.pasteur@free.fr>
   Copyright (C) 2004-2006 Jaroslaw Staniek <js@iidea.pl>

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this program; see the file COPYING.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include "kexidbtextedit.h"
#include "kexidblineedit.h"
#include <kexidb/queryschema.h>

#include <kapplication.h>
#include <kstdaccel.h>
#include <kdebug.h>

#include <tqpainter.h>

KexiDBTextEdit::KexiDBTextEdit(TQWidget *parent, const char *name)
 : KTextEdit(parent, name)
 , KexiDBTextWidgetInterface()
 , KexiFormDataItemInterface()
 , m_menuExtender(TQT_TQOBJECT(this), this)
 , m_slotTextChanged_enabled(true)
{
	connect(this, TQT_SIGNAL(textChanged()), this, TQT_SLOT(slotTextChanged()));
	installEventFilter(this);
}

KexiDBTextEdit::~KexiDBTextEdit()
{
}

void KexiDBTextEdit::setInvalidState( const TQString& displayText )
{
	setReadOnly(true);
//! @todo move this to KexiDataItemInterface::setInvalidStateInternal() ?
	if (focusPolicy() & TQ_TabFocus)
		setFocusPolicy(TQ_ClickFocus);
	KTextEdit::setText(displayText);
}

void KexiDBTextEdit::setValueInternal(const TQVariant& add, bool removeOld)
{
	if (m_columnInfo && m_columnInfo->field->type()==KexiDB::Field::Boolean) {
//! @todo temporary solution for booleans!
		KTextEdit::setText( add.toBool() ? "1" : "0" );
	}
	else {
		if (removeOld)
			KTextEdit::setText( add.toString() );
		else
			KTextEdit::setText( m_origValue.toString() + add.toString() );
	}
}

TQVariant KexiDBTextEdit::value()
{
	return text();
}

void KexiDBTextEdit::slotTextChanged()
{
	if (!m_slotTextChanged_enabled)
		return;
	signalValueChanged();
}

bool KexiDBTextEdit::valueIsNull()
{
	return text().isNull();
}

bool KexiDBTextEdit::valueIsEmpty()
{
	return text().isEmpty();
}

bool KexiDBTextEdit::isReadOnly() const
{
	return KTextEdit::isReadOnly();
}

void KexiDBTextEdit::setReadOnly( bool readOnly )
{
	KTextEdit::setReadOnly( readOnly );
	TQPalette p = palette();
	TQColor c(readOnly ? lighterGrayBackgroundColor(kapp->palette()) : p.color(TQPalette::Normal, TQColorGroup::Base));
	setPaper( c );
	p.setColor(TQColorGroup::Base, c);
	p.setColor(TQColorGroup::Background, c);
	setPalette( p );
}

void KexiDBTextEdit::setText( const TQString & text, const TQString & context )
{
	KTextEdit::setText(text, context);
}

TQWidget* KexiDBTextEdit::widget()
{
	return this;
}

bool KexiDBTextEdit::cursorAtStart()
{
	int para, index;
	getCursorPosition ( &para, &index );
	return para==0 && index==0;
}

bool KexiDBTextEdit::cursorAtEnd()
{
	int para, index;
	getCursorPosition ( &para, &index );
	return (paragraphs()-1)==para && (paragraphLength(paragraphs()-1)-1)==index;
}

void KexiDBTextEdit::clear()
{
	setText(TQString(), TQString());
}

void KexiDBTextEdit::setColumnInfo(KexiDB::QueryColumnInfo* cinfo)
{
	KexiFormDataItemInterface::setColumnInfo(cinfo);
	if (!cinfo)
		return;
	KexiDBTextWidgetInterface::setColumnInfo(m_columnInfo, this);
}

void KexiDBTextEdit::paintEvent ( TQPaintEvent *pe )
{
	KTextEdit::paintEvent( pe );
	TQPainter p(this);
	KexiDBTextWidgetInterface::paint( this, &p, text().isEmpty(), alignment(), hasFocus() );
}

TQPopupMenu * KexiDBTextEdit::createPopupMenu(const TQPoint & pos)
{
	TQPopupMenu *contextMenu = KTextEdit::createPopupMenu(pos);
	m_menuExtender.createTitle(contextMenu);
	return contextMenu;
}

void KexiDBTextEdit::undo()
{
	cancelEditor();
}

void KexiDBTextEdit::setDisplayDefaultValue(TQWidget* widget, bool displayDefaultValue)
{
	KexiFormDataItemInterface::setDisplayDefaultValue(widget, displayDefaultValue);
	// initialize display parameters for default / entered value
	KexiDisplayUtils::DisplayParameters * const params 
		= displayDefaultValue ? m_displayParametersForDefaultValue : m_displayParametersForEnteredValue;
	TQPalette pal(palette());
	pal.setColor(TQPalette::Active, TQColorGroup::Text, params->textColor);
	setPalette(pal);
	setFont(params->font);
//! @todo support rich text...
/*	m_slotTextChanged_enabled = false;
		//for rich text...
		const TQString origText( text() );
		KTextEdit::setText(TQString());
		setCurrentFont(params->font);
		setColor(params->textColor);
		KTextEdit::setText(origText);
	m_slotTextChanged_enabled = true;*/
}

void KexiDBTextEdit::moveCursorToEnd()
{
	KTextEdit::setCursorPosition(paragraphs()-1, paragraphLength( paragraphs()-1 ));
}

void KexiDBTextEdit::moveCursorToStart()
{
	KTextEdit::setCursorPosition(0 /*para*/, 0 /*index*/);
}

void KexiDBTextEdit::selectAll()
{
	KTextEdit::selectAll();
}

void KexiDBTextEdit::keyPressEvent( TQKeyEvent *ke )
{
	// for instance, Windows uses Ctrl+Tab for moving between tabs, so do not steal this shortcut
	if (KStdAccel::tabNext().contains( KKey(ke) ) || KStdAccel::tabPrev().contains( KKey(ke) )) {
		ke->ignore();
		return;
	}
	KTextEdit::keyPressEvent(ke);
}

#include "kexidbtextedit.moc"
