/*
 *  Copyright (c) 2002 Patrick Julien  <freak@codepimps.org>
 *  Copyright (c) 2005 Adrian Page <adrian@pagenet.plus.com>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */
#ifndef KIS_STRATEGY_COLORSPACE_RGB_F32_H_
#define KIS_STRATEGY_COLORSPACE_RGB_F32_H_

#include <tqcolor.h>

#include <klocale.h>

#include <koffice_export.h>

#include "kis_global.h"
#include "kis_f32_base_colorspace.h"

class KisColorSpaceFactoryRegistry;

class KRITATOOL_EXPORT KisRgbF32ColorSpace : public KisF32BaseColorSpace {
public:
    KisRgbF32ColorSpace(KisColorSpaceFactoryRegistry * parent, KisProfile *p);
    virtual ~KisRgbF32ColorSpace();

    virtual bool willDegrade(ColorSpaceIndependence independence)
        {
            if (independence == TO_RGBA8 || independence == TO_LAB16)
                return true;
            else
                return false;
        };



public:
    void setPixel(TQ_UINT8 *pixel, float red, float green, float blue, float alpha) const;
    void getPixel(const TQ_UINT8 *pixel, float *red, float *green, float *blue, float *alpha) const;

    virtual void fromTQColor(const TQColor& c, TQ_UINT8 *dst, KisProfile * profile = 0);
    virtual void fromTQColor(const TQColor& c, TQ_UINT8 opacity, TQ_UINT8 *dst, KisProfile * profile = 0);

    virtual void toTQColor(const TQ_UINT8 *src, TQColor *c, KisProfile * profile = 0);
    virtual void toTQColor(const TQ_UINT8 *src, TQColor *c, TQ_UINT8 *opacity, KisProfile * profile = 0);

    virtual TQ_UINT8 difference(const TQ_UINT8 *src1, const TQ_UINT8 *src2);
    virtual void mixColors(const TQ_UINT8 **colors, const TQ_UINT8 *weights, TQ_UINT32 nColors, TQ_UINT8 *dst) const;
    virtual void invertColor(TQ_UINT8 * src, TQ_INT32 nPixels);
    virtual void convolveColors(TQ_UINT8** colors, TQ_INT32 * kernelValues, KisChannelInfo::enumChannelFlags channelFlags, TQ_UINT8 *dst, TQ_INT32 factor, TQ_INT32 offset, TQ_INT32 nColors) const;
    virtual TQ_UINT8 intensity8(const TQ_UINT8 * src) const;

    virtual TQValueVector<KisChannelInfo *> channels() const;
    virtual TQ_UINT32 nChannels() const;
    virtual TQ_UINT32 nColorChannels() const;
    virtual TQ_UINT32 pixelSize() const;


    virtual TQImage convertToTQImage(const TQ_UINT8 *data, TQ_INT32 width, TQ_INT32 height,
                       KisProfile *  dstProfile,
                       TQ_INT32 renderingIntent,
                       float exposure = 0.0f);

    virtual KisCompositeOpList userVisiblecompositeOps() const;


protected:

    virtual void bitBlt(TQ_UINT8 *dst,
                TQ_INT32 dstRowStride,
                const TQ_UINT8 *src,
                TQ_INT32 srcRowStride,
                const TQ_UINT8 *srcAlphaMask,
                TQ_INT32 maskRowStride,
                TQ_UINT8 opacity,
                TQ_INT32 rows,
                TQ_INT32 cols,
                const KisCompositeOp& op);

    void compositeOver(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, float opacity);
    void compositeMultiply(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, float opacity);
    void compositeDivide(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, float opacity);
    void compositeScreen(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, float opacity);
    void compositeOverlay(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, float opacity);
    void compositeDodge(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, float opacity);
    void compositeBurn(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, float opacity);
    void compositeDarken(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, float opacity);
    void compositeLighten(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, float opacity);
    void compositeHue(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, float opacity);
    void compositeSaturation(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, float opacity);
    void compositeValue(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, float opacity);
    void compositeColor(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, float opacity);
    void compositeErase(TQ_UINT8 *dst, TQ_INT32 dstRowStride, const TQ_UINT8 *src, TQ_INT32 srcRowStride, const TQ_UINT8 *mask, TQ_INT32 maskRowStride, TQ_INT32 rows, TQ_INT32 columns, float opacity);

private:
    friend class KisRgbF32ColorSpaceTester;

    static const TQ_UINT8 PIXEL_BLUE = 0;
    static const TQ_UINT8 PIXEL_GREEN = 1;
    static const TQ_UINT8 PIXEL_RED = 2;
    static const TQ_UINT8 PIXEL_ALPHA = 3;

    struct Pixel {
        float blue;
        float green;
        float red;
        float alpha;
    };

    // For Alpha Composite
    struct F32Mult {
        inline float operator()(const float& a, const float& b) const {
            return a * b;
        }
    };
    struct Uint8ToF32 {
        inline float operator()(const TQ_UINT8 src) const {
            return UINT8_TO_FLOAT(src);
        }
    };
    struct F32OpacityTest {
        inline bool operator()(const float& opacity) const {
            return opacity > F32_OPACITY_TRANSPARENT + 1e-6; // #define EPSILON in the .cc
        }
    };
};

// FIXME: lcms doesn't support 32-bit float
#define F32_LCMS_TYPE TYPE_BGRA_16

class KisRgbF32ColorSpaceFactory : public KisColorSpaceFactory
{
public:
    /**
     * Chalk definition for use in .kra files and internally: unchanging name +
     * i18n'able description.
     */
    virtual KisID id() const { return KisID("RGBAF32", i18n("RGB (32-bit float/channel)")); };

    /**
     * lcms colorspace type definition.
     */
    virtual TQ_UINT32 colorSpaceType() { return F32_LCMS_TYPE; };

    virtual icColorSpaceSignature colorSpaceSignature() { return icSigRgbData; };

    virtual KisColorSpace *createColorSpace(KisColorSpaceFactoryRegistry * parent, KisProfile *p) { return new KisRgbF32ColorSpace(parent, p); };

    virtual TQString defaultProfile() { return "sRGB built-in - (lcms internal)"; };
};

#endif // KIS_STRATEGY_COLORSPACE_RGB_F32_H_

