/***************************************************************************
 *
 * knetworkmanager-devicestore_dbus.cpp - A NetworkManager frontend for KDE
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Timo Hoenig        <thoenig@suse.de>, <thoenig@nouse.net>
 *         Valentine Sinitsyn <e_val@inbox.ru>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

#include <tqwidget.h>
#include <tqlineedit.h>
#include <tqgroupbox.h>
#include <tqlayout.h>
#include <tqhostaddress.h>

#include "knetworkmanager-connection_setting_ipv4_widget.h"
#include "knetworkmanager-connection_setting_ipv4.h"

using namespace ConnectionSettings;

#define IP_INPUT_MASK "900.900.900.900"

IPv4WidgetImpl::IPv4WidgetImpl(Connection* conn, TQWidget* parent, const char* name, WFlags fl)
	: WidgetInterface(parent, name, fl)
{
	_ipv4_setting = dynamic_cast<ConnectionSettings::IPv4*> (conn->getSetting(NM_SETTING_IP4_CONFIG_SETTING_NAME));

	TQVBoxLayout* layout = new TQVBoxLayout(this, 1, 1);
	_mainWid = new ConnectionSettingIPv4Widget(this);
	layout->addWidget(_mainWid);

	Init();
}

void
IPv4WidgetImpl::Init()
{
	_mainWid->groupIPConfig->setChecked(_ipv4_setting->getMethod() == IPv4::METHOD_MANUAL);

	connect(_mainWid->groupIPConfig, TQT_SIGNAL(toggled(bool)), this, TQT_SLOT(slotIPConfigEnabled(bool)));

	_mainWid->txtIP->setInputMask(IP_INPUT_MASK);
	_mainWid->txtNetmask->setInputMask(IP_INPUT_MASK);
	_mainWid->txtGateway->setInputMask(IP_INPUT_MASK);

	if (!_ipv4_setting->getAddresses().isEmpty())
	{
		_mainWid->txtIP->setText(_ipv4_setting->getAddresses()[0].address.toString());
		_mainWid->txtNetmask->setText(_ipv4_setting->getAddresses()[0].netmask.toString());
		_mainWid->txtGateway->setText(_ipv4_setting->getAddresses()[0].gateway.toString());
	}

	if (!_ipv4_setting->getDNS().isEmpty())
	{
		TQValueList<TQHostAddress> hosts = _ipv4_setting->getDNS();
		TQStringList list;
		for (TQValueList<TQHostAddress>::Iterator it = hosts.begin(); it != hosts.end(); ++it)
		{
			list.append((*it).toString());
		}
		_mainWid->txtDNSAddresses->setText(list.join(" "));
	}

	if (!_ipv4_setting->getDNSSearch().isEmpty())
		_mainWid->txtDNSSearch->setText(_ipv4_setting->getDNSSearch().join(" "));

	// connect the signals after setting up the values
	connect(_mainWid->txtIP, TQT_SIGNAL(textChanged(const TQString&)), this, TQT_SLOT(slotIPAddressChanged(const TQString&)));
	connect(_mainWid->txtNetmask, TQT_SIGNAL(textChanged(const TQString&)), this, TQT_SLOT(slotNetmaskChanged(const TQString&)));
	connect(_mainWid->txtGateway, TQT_SIGNAL(textChanged(const TQString&)), this, TQT_SLOT(slotGatewayChanged(const TQString&)));
	connect(_mainWid->txtDNSAddresses, TQT_SIGNAL(textChanged(const TQString&)), this, TQT_SLOT(slotDNSAddressesChanged(const TQString&)));
	connect(_mainWid->txtDNSSearch, TQT_SIGNAL(textChanged(const TQString&)), this, TQT_SLOT(slotDNSSearchChanged(const TQString&)));
}

void
IPv4WidgetImpl::Activate()
{

}

void
IPv4WidgetImpl::slotDNSAddressesChanged(const TQString& adr)
{
	// TODO: use , and ; for splitting too
	TQStringList list = TQStringList::split(" ", adr);
	TQValueList<TQHostAddress> hosts;
	for (TQStringList::Iterator it = list.begin(); it != list.end(); ++it)
	{
		TQHostAddress host(*it);
		if (!host.isNull())
			hosts.append(host);
	}
	_ipv4_setting->setDNS(hosts);
}


void
IPv4WidgetImpl::slotDNSSearchChanged(const TQString& search)
{
	// TODO: use , and ; for splitting too
	_ipv4_setting->setDNSSearch(TQStringList::split(" ", search));
}

void
IPv4WidgetImpl::slotIPConfigEnabled(bool enabled)
{
	_ipv4_setting->setMethod(enabled ? IPv4::METHOD_MANUAL : IPv4::METHOD_DHCP );
}

void
IPv4WidgetImpl::slotIPAddressChanged(const TQString& ip)
{
	TQHostAddress ipadr(ip);
	if (!ipadr.isNull())
	{
		TQValueList<IPv4Address> addrs = _ipv4_setting->getAddresses();
		if (addrs.size() > 0)
			addrs[0].address = ipadr;
		else
		{
			IPv4Address adr;
			adr.address = ipadr;
			addrs.append(adr);
		}
		_ipv4_setting->setAddresses(addrs);

		// if netmask is not set yet we preset it to a default value depending on the network class
		if (_mainWid->txtNetmask->text() == "...")
		{
			if ( (ipadr.toIPv4Address() & 0xFF000000) < 0xDF000000)
			{
				if ( (ipadr.toIPv4Address() & 0xFF000000) >= 0xC0000000)
					_mainWid->txtNetmask->setText("255.255.255.0"); // class C
				else if ( (ipadr.toIPv4Address() & 0xFF000000) >= 0x80000000)
					_mainWid->txtNetmask->setText("255.255.0.0"); // class B
				else
					_mainWid->txtNetmask->setText("255.0.0.0"); // class A
			}
		}
	}
}

void
IPv4WidgetImpl::slotNetmaskChanged(const TQString& ip)
{
	TQValueList<IPv4Address> addrs = _ipv4_setting->getAddresses();
	addrs[0].netmask = TQHostAddress(ip);
	_ipv4_setting->setAddresses(addrs);
}

void
IPv4WidgetImpl::slotGatewayChanged(const TQString& ip)
{
	TQValueList<IPv4Address> addrs = _ipv4_setting->getAddresses();
	addrs[0].gateway = TQHostAddress(ip);
	_ipv4_setting->setAddresses(addrs);
}

#include "knetworkmanager-connection_setting_ipv4_widget.moc"
