/***************************************************************************
 *
 * knetworkmanager-connection_editor.cpp - A NetworkManager frontend for KDE
 *
 * Copyright (C) 2005, 2006 Novell, Inc.
 *
 * Author: Helmut Schaa        <hschaa@suse.de>, <helmut.schaa@gmx.de>
 * Author: Timothy Pearson <kb9vqf@pearsoncomputing.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 **************************************************************************/

// qt headers
#include <tqwidget.h>
#include <tqcombobox.h>
#include <tqtabwidget.h>
#include <tqpushbutton.h>
#include <tqwidgetstack.h>
#include <tqapplication.h>
#include <tqlabel.h>
#include <tqpopupmenu.h>

// kde headers
#include <kiconloader.h>
#include <kdebug.h>
#include <kpushbutton.h>
#include <klistview.h>
#include <klocale.h>
#include <kcombobox.h>

// knm headers
#include "knetworkmanager-connection_setting_info.h"
#include "knetworkmanager-connection.h"
#include "knetworkmanager-wired_connection.h"
#include "knetworkmanager-wireless_connection.h"
#include "knetworkmanager-vpn_connection.h"
#include "knetworkmanager-connection_store.h"
#include "knetworkmanager-connection_editor.h"
#include "knetworkmanager-connection_settings_dialog.h"
#include "knetworkmanager-storage.h"
#include "knetworkmanager-vpnservice.h"
#include "knetworkmanager-vpnmanager.h"

using namespace ConnectionSettings;

/*
 * ConnectionListViewItem
 */
class ConnectionListViewItem : public KListViewItem
{
	public:

	ConnectionListViewItem(TQListView* parent, GenericConnection* connection)
		: KListViewItem(parent)
		, _conn(connection)
	{
		Info* info = _conn->getInfoSetting();
		if (info)
		{
			setText(0, info->getName());
			setText(1, info->getDevType());
			// TODO: Move to a Factory
			if (info->getDevType() == NM_SETTING_WIRED_SETTING_NAME)
				setPixmap(0, KGlobal::iconLoader()->loadIcon("wired", KIcon::Small));
			else if (info->getDevType() == NM_SETTING_WIRELESS_SETTING_NAME)
				setPixmap(0, KGlobal::iconLoader()->loadIcon("wireless", KIcon::Small));
			else if (info->getDevType() == NM_SETTING_VPN_SETTING_NAME)
				setPixmap(0, KGlobal::iconLoader()->loadIcon("encrypted", KIcon::Small));
			else
				setPixmap(0, KGlobal::iconLoader()->loadIcon("help", KIcon::Small));
		}
	}

	GenericConnection* _conn;
};

/*
 * Constructor
 */
ConnectionEditorImpl::ConnectionEditorImpl(TQWidget* parent, const char* name, bool modal, WFlags fl)
	: ConnectionEditor(parent, name, modal, fl)
{

	// TODO: enable combobox if implemented
	cboConnectionType->hide();

	// TODO: Editmode is not ready yet, hide the button
//	pbEdit->hide();


	pbNew->setIconSet(KGlobal::iconLoader()->loadIcon("add", KIcon::Small));
	pbDelete->setIconSet(KGlobal::iconLoader()->loadIcon("remove", KIcon::Small));
	pbEdit->setIconSet(KGlobal::iconLoader()->loadIcon("edit", KIcon::Small));

	TQPopupMenu* popup = new TQPopupMenu(pbNew);
	// TODO: move to a factory class
	popup->insertItem(KGlobal::iconLoader()->loadIcon("wireless", KIcon::Small), i18n("Wireless"), this, TQT_SLOT(slotNewWirelessConnection()));
	popup->insertItem(KGlobal::iconLoader()->loadIcon("wired", KIcon::Small), i18n("Wired"), this, TQT_SLOT(slotNewWiredConnection()));

	if (!VPNManager::getVPNServices().isEmpty())
		popup->insertItem(KGlobal::iconLoader()->loadIcon("encrypted", KIcon::Small), i18n("VPN"), this, TQT_SLOT(slotNewVPNConnection()));

	pbNew->setPopup(popup);

	connect(pbClose, TQT_SIGNAL(clicked()), this, TQT_SLOT(close()));
	connect(pbDelete, TQT_SIGNAL(clicked()), this, TQT_SLOT(slotRemoveCurrentConnection()));
	connect(pbEdit, TQT_SIGNAL(clicked()), this, TQT_SLOT(slotEditCurrentConnection()));

	// show all connections
	fillConnectionList();
}

/*
 * Destructor
 */
ConnectionEditorImpl::~ConnectionEditorImpl()
{
	// remove the popupmenu
	if (pbNew->popup())
		delete pbNew->popup();
}

/*
 * New Wireless connection
 */
void ConnectionEditorImpl::slotNewWirelessConnection()
{
	// create a new wireless connection
	slotEditNewConnection(new WirelessConnection());
}

/*
 * New Wired connection
 */
void ConnectionEditorImpl::slotNewWiredConnection()
{
	slotEditNewConnection(new WiredConnection());
}

/*
 * New VPN connection
 */
void ConnectionEditorImpl::slotNewVPNConnection()
{
	slotEditNewConnection(new VPNConnection());
}

/*
 *
 */
void ConnectionEditorImpl::slotEditNewConnection(Connection* conn)
{
	// open a dialog for editing the connection
	ConnectionSettingsDialogImpl* dlg = new ConnectionSettingsDialogImpl(conn, true, NULL, this, "connect_something", false, TQt::WDestructiveClose);
	connect(dlg, TQT_SIGNAL(connectionSaved()), this, TQT_SLOT(slotRefershConnectionList()));
	dlg->show();
}

void ConnectionEditorImpl::slotRefershConnectionList()
{
	fillConnectionList();
}

/*
 * Edit the selected connection
 */
void ConnectionEditorImpl::slotEditCurrentConnection()
{
	ConnectionListViewItem* item = dynamic_cast<ConnectionListViewItem*>(lvConnections->currentItem());
	if (!item)
		return;

	Connection* conn = item->_conn;
	Storage* storage = Storage::getInstance();
	bool hasSecretsStored = storage->hasSecretsStored(conn);

	// we need the secrets for editing
	if (hasSecretsStored)
		storage->restoreAllSecrets(conn);

	ConnectionSettingsDialogImpl* dlg = new ConnectionSettingsDialogImpl(conn, false, NULL, this, "connect_something", false, TQt::WDestructiveClose);
	dlg->show();

	// save all connections (if not done already)
	storage->saveConnections();
}


/*
 * Delete the selected connection
 */
void ConnectionEditorImpl::slotRemoveCurrentConnection()
{
	ConnectionListViewItem* item = dynamic_cast<ConnectionListViewItem*>(lvConnections->currentItem());
	if (!item)
		return;

	ConnectionStore* cstore = ConnectionStore::getInstance();
	Connection* conn = item->_conn;

	lvConnections->takeItem(item);
	delete item;
	
	cstore->removeConnection(conn);
}

/*
 * Fill the connection list
 */
void ConnectionEditorImpl::fillConnectionList()
{
	ConnectionStore* cstore = ConnectionStore::getInstance();
	TQValueList<Connection*> conns = cstore->getConnections();
	TQValueList<Connection*>::Iterator it = conns.begin();

	lvConnections->clear();

	for (; it != conns.end(); ++it)
	{
		GenericConnection* conn = dynamic_cast<GenericConnection*>(*it);
		if (conn)
		{
			Info* info = conn->getInfoSetting();
			if (info)
			{
				new ConnectionListViewItem(lvConnections, conn);
			}
			else
				kdWarning() << k_funcinfo << "Connection without Info setting, drop it." << endl;
		}
		else
			kdWarning() << k_funcinfo << "non-generic connection, dropping it." << endl;
		
	}
}

#include "knetworkmanager-connection_editor.moc"
