/***************************************************************************
 *   kio-locate: KDE I/O Slave for the locate command                      *
 *                                                                         *
 *   Copyright (C) 2005 by Tobi Vollebregt                                 *
 *   tobivollebregt@gmail.com                                              *
 *                                                                         *
 *   Thanks to Google's Summer Of Code Program!                            *
 *                                                                         *
 *   Copyright (C) 2004 by Armin Straub                                    *
 *   linux@arminstraub.de                                                  *
 *                                                                         *
 *   This program was initially written by Michael Schuerig.               *
 *   Although I have completely rewritten it, most ideas are adopted       *
 *   from his original work.                                               *
 *                                                                         *
 *   Copyright (C) 2002 by Michael Schuerig                                *
 *   michael@schuerig.de                                                   *
 *                                                                         *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include <tqtextcodec.h>

#include <kdebug.h>
#include <kstandarddirs.h>

#include "locater.h"


Locater::Locater(TQObject *parent, const char *name)
	: TQObject(parent, name), m_process(TQTextCodec::codecForLocale())
{
	DEBUGSTR << "Locater::Locater" << endl;

	connect(&m_process, TQT_SIGNAL(processExited(KProcess*)),
			 this, TQT_SLOT(finished(KProcess*)));
	connect(&m_process, TQT_SIGNAL(readReady(KProcIO*)),
			 this, TQT_SLOT(gotOutput(KProcIO*)));

	setupLocate();
}


Locater::~Locater()
{
	DEBUGSTR << "Locater::~Locater" << endl;
}


void Locater::setupLocate(const TQString& binary, const TQString& additionalArguments)
{
	DEBUGSTR << "Locater::setupLocate(" << binary << ", " << additionalArguments << ")" << endl;

    // Automatically choose the correct binary if not specified.
    if (binary.isEmpty()) {
        if (!KStandardDirs::findExe("slocate").isNull()) {
            m_binary = "slocate";
        } else if (!KStandardDirs::findExe("rlocate").isNull()) {
            m_binary = "rlocate";
        } else {
            m_binary = "locate";
        }
        DEBUGSTR << "Using binary:" << m_binary << endl;
    } else {
        m_binary = binary;
    }
	m_additionalArguments = additionalArguments;
    m_binaryExists = KStandardDirs::findExe(m_binary) != TQString();
}


bool Locater::locate(const TQString& pattern, bool ignoreCase, bool regExp)
{
	DEBUGSTR << "Locater::locate(" << pattern << "," << ignoreCase << "," << regExp << ")" << endl;

	m_process.resetAll();
	m_process << m_binary;
	if (!m_additionalArguments.isEmpty()) {
		m_process << m_additionalArguments;
	}
	if (ignoreCase) {
		// m_process << "--ignore-case";
		m_process << "-i";
	}
	if (regExp) {
		m_process << "-r";
	}
	m_process << pattern;

	return m_process.start(KProcess::Block, false);
}

void Locater::stop()
{
	DEBUGSTR << "Locater::stop()" << endl;

	m_process.kill();
	emit finished();
}


void Locater::gotOutput(KProcIO* /*proc*/)
{
	//DEBUGSTR << "Locater::gotOutput" << endl;

	TQStringList items;
	TQString line;

	while (m_process.readln(line) != -1) {
		//DEBUGSTR << "OUTPUT>> '" << line << "'" << endl;

		items << line;
	}

	emit found(items);
}


void Locater::finished(KProcess* /*proc*/)
{
	DEBUGSTR << "Locater::finished" << endl;

	emit finished();
}


#include "locater.moc"
