/* This file is part of the KDE project
   Copyright (C) 2002 Harald Fernengel <harry@kdevelop.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include <tqvbox.h>
#include <tqlabel.h>
#include <tqheader.h>

#include <klocale.h>
#include <kpushbutton.h>
#include <klistbox.h>
#include <klistview.h>
#include <kstdguiitem.h>

#include "ksavealldialog.h"

namespace
{

class CheckURL : public TQCheckListItem
{
public:
	CheckURL( TQListView * lv, KURL const & url )
		: TQCheckListItem( lv, url.path(), TQCheckListItem::CheckBox),
		_url( url )
	{}

	KURL const & url() const { return _url; }

private:
    KURL _url;
};

}


KSaveSelectDialog::KSaveSelectDialog( KURL::List const & filelist, KURL::List const & ignorelist, TQWidget * parent ) :
  KDialogBase( parent, "SaveAllDialog", true, i18n("Save Modified Files?"),
	       Ok | User1 | Close )
{
  TQVBox *top = makeVBoxMainWidget();

  (void)new TQLabel( i18n("The following files have been modified. Save them?"), top );

	_listview = new KListView( top );
	_listview->addColumn( "" );
	_listview->header()->hide();
	_listview->setResizeMode( TQListView::LastColumn );

	setButtonOK( KGuiItem( i18n("Save &Selected"), TQString(), i18n("Saves all selected files") ) );
	setButtonText( User1, i18n("Save &None") );
	setButtonText( Close, KStdGuiItem::cancel().text() );
	setButtonTip( User1, i18n("Lose all modifications") );
	setButtonTip( Close, i18n("Cancels the action") );
	
	KURL::List::ConstIterator it = filelist.begin();
	while ( it != filelist.end() )
	{
		if ( !ignorelist.contains( *it ) )
		{
			TQCheckListItem * x = new CheckURL( _listview, *it );
			x->setOn( true );
		}
		++it;
	}

	connect( this, TQT_SIGNAL(closeClicked()), this, TQT_SLOT(cancel()) );
	connect( this, TQT_SIGNAL(okClicked()), this, TQT_SLOT(save()) );
	connect( this, TQT_SIGNAL(user1Clicked()), this, TQT_SLOT(saveNone()) );
}

KSaveSelectDialog::~KSaveSelectDialog() {}

void KSaveSelectDialog::saveNone( )
{
	// deselect all
	CheckURL * item = static_cast<CheckURL*>( _listview->firstChild() );
	while ( item )
	{
		item->setOn( false );
		item = static_cast<CheckURL*>( item->nextSibling() );
	}

	TQDialog::accept();
}

void KSaveSelectDialog::save( )
{
	TQDialog::accept();
}

void KSaveSelectDialog::cancel( )
{
	TQDialog::reject();
}

KURL::List KSaveSelectDialog::filesToSave( )
{
	KURL::List filelist;
	CheckURL const * item = static_cast<CheckURL*>( _listview->firstChild() );
	while ( item )
	{
		if ( item->isOn() )
		{
			filelist << item->url();
		}
		item = static_cast<CheckURL*>( item->nextSibling() );
	}
	return filelist;
}

KURL::List KSaveSelectDialog::filesNotToSave( )
{
	KURL::List filelist;
	CheckURL const * item = static_cast<CheckURL*>( _listview->firstChild() );
	while ( item )
	{
		if ( ! item->isOn() )
		{
			filelist << item->url();
		}
		item = static_cast<CheckURL*>( item->nextSibling() );
	}
	return filelist;
}


KSaveAllDialog::KSaveAllDialog( const TQStringList& filenames, TQWidget* parent ) :
  KDialogBase( parent, "SaveAllDialog", true, i18n("Save Modified Files?"),
	       Ok | User1 | Close )
{
  m_result = Cancel;

  TQVBox *top = makeVBoxMainWidget();

  (void)new TQLabel( i18n("The following files have been modified. Save them?"), top );
  KListBox* lb = new KListBox( top );
  lb->setMinimumHeight( lb->fontMetrics().height() * 5 );
  lb->insertStringList( filenames );

  setButtonOK( KGuiItem( i18n("Save &All"), TQString(), i18n("Saves all modified files") ) );
  setButtonText( User1, i18n("Save &None") );
  setButtonText( Close, KStdGuiItem::cancel().text() );
  setButtonTip( User1, i18n("Lose all modifications") );
  setButtonTip( Close, i18n("Cancels the action") );

  connect( this, TQT_SIGNAL(closeClicked()), this, TQT_SLOT(cancel()) );
  connect( this, TQT_SIGNAL(okClicked()), this, TQT_SLOT(saveAll()) );
  connect( this, TQT_SIGNAL(user1Clicked()), this, TQT_SLOT(revert()) );
}

KSaveAllDialog::~KSaveAllDialog()
{
}

void KSaveAllDialog::revert()
{
  m_result = Revert;
  TQDialog::accept();
}

void KSaveAllDialog::saveAll()
{
  m_result = SaveAll;
  TQDialog::accept();
}

void KSaveAllDialog::cancel()
{
  m_result = Cancel;
  TQDialog::reject();
}

#include "ksavealldialog.moc"
