/***************************************************************************
 *   Copyright (C) 1999-2001 by Bernd Gehrmann                             *
 *   bernd@kdevelop.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "makeitem.h"

#include <tqstylesheet.h>

#include <kdebug.h>
#include <klocale.h>

#include "ktexteditor/cursorinterface.h"

MakeItem::MakeItem()
{
}

MakeItem::MakeItem( const TQString& text )
	: m_text( text )
{
}

MakeItem::~MakeItem()
{
}

TQString MakeItem::color( bool bright_bg )
{
	switch ( type() )
	{
	case Error:
		return bright_bg ? "maroon" : "red";
	case Warning:
		return bright_bg ? "#666" : "#999";
	case Diagnostic:
		return bright_bg ? "black" : "white";
	default:
		return bright_bg ? "navy" : "blue";
	}
}

TQString MakeItem::icon()
{
	switch ( type() )
	{
	case Error:
	case Warning:
		return "<img src=\"error\"></img><nobr> </nobr>";
	case Diagnostic:
		return "<img src=\"warning\"></img><nobr> </nobr>";
	default:
		return "<img src=\"message\"></img><nobr> </nobr>";
	}
}

TQString MakeItem::text( EOutputLevel )
{
	return TQStyleSheet::escape( m_text );
}

TQString MakeItem::formattedText( EOutputLevel level, bool bright_bg )
{
  TQString txt = text(level);
	if (txt.isEmpty())
		return "<br>";
	if ( level == eFull )
	{
		return txt;
	}
	else
	{
		return TQString("<code>")
        .append( icon() ).append("<font color=\"").append( color( bright_bg) ).append("\">")
        .append( txt ).append("</font></code>").append( br() );
	}
}

TQString MakeItem::br()
{
    // TQt >= 3.1 doesn't need a <br>.
    static const TQString br;
    return br;
}

ErrorItem::ErrorItem( const TQString& fn, int ln, const TQString& tx, const TQString& line, bool isWarning, bool isInstatiationInfo, const TQString& compiler )
	: MakeItem( line )
	, fileName(fn)
	, lineNum(ln)
	, m_error(tx)
	, m_isWarning(isWarning | isInstatiationInfo)
	, m_isInstatiationInfo( isInstatiationInfo )
	, m_compiler(compiler)
{}

ErrorItem::~ErrorItem()
{
}

bool ErrorItem::append( const TQString& text )
{
	if ( !text.startsWith("   ") )
		return false;
	if ( text.startsWith("   ") && (m_compiler == "intel") )
		return false;
	m_text += text;
	m_error += text;
	m_error = m_error.simplifyWhiteSpace();
	m_text = m_text.simplifyWhiteSpace();
	return true;
}

ExitStatusItem::ExitStatusItem( bool normalExit, int exitStatus )
	: m_normalExit( normalExit )
	, m_exitStatus( exitStatus )
{
//	kdDebug() << "ExitStatusItem: normalExit=" << normalExit << "; exitStatus=" << exitStatus << endl;
	m_text = i18n("*** Compilation aborted ***");
	if ( m_normalExit )
		if (m_exitStatus )
			m_text = i18n("*** Exited with status: %1 ***").arg( m_exitStatus );
        	else
			m_text = i18n("*** Success ***");
}

TQString ExitStatusItem::text( EOutputLevel )
{
	return m_text;	
}

bool DirectoryItem::m_showDirectoryMessages = true;

TQString EnteringDirectoryItem::text( EOutputLevel outputLevel )
{
	if ( outputLevel < eFull )
		return i18n("Entering directory %1").arg( directory );
	return m_text;
}

TQString ExitingDirectoryItem::text( EOutputLevel outputLevel )
{
	if ( outputLevel < eFull )
		return i18n("Leaving directory %1").arg( directory );
	return m_text;
}

TQString ActionItem::text( EOutputLevel outputLevel )
{
	if ( outputLevel < eFull )
	{
		if ( m_tool.isEmpty() )
			return TQString(m_action).append(" <b>").append(m_file).append("</b>");
		return TQString(m_action).append(" <b>").append(m_file).append("</b>").append(" (").append(m_tool).append(")");
	}
	return MakeItem::text( outputLevel );
}
