/* This file is part of the KDE project
   Copyright (C) 2000-2001 Bernd Gehrmann <bernd@kdevelop.org>
   Copyright (C) 2003 Alexander Dymo <cloudtemple@mksat.net>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#ifndef _FLAGBOXES_H_
#define _FLAGBOXES_H_

#include <kfile.h>

/**
@file flagboxes.h
Support classes for compiler plugins.
*/

#include <tqlistview.h>
#include <tqcheckbox.h>
#include <tqradiobutton.h>
#include <tqptrlist.h>
#include <tqstringlist.h>

class TQSpinBox;
class FlagListBox;
class FlagListToolTip;
class FlagCheckBoxController;
class FlagRadioButtonController;
class FlagPathEditController;
class FlagEditController;
class KLineEdit;
class TQPushButton;
class KURLRequester;

/**List item holding a compiler flag.*/
class FlagListItem : public TQCheckListItem
{
public:
    FlagListItem(FlagListBox *parent, const TQString &flagstr,
                 const TQString &description);
    FlagListItem(FlagListBox *parent, const TQString &flagstr,
                 const TQString &description, const TQString &offstr);
    ~FlagListItem()
        {}
private:
    TQString flag;
    TQString off;
    TQString desc;
    friend class FlagListToolTip;
    friend class FlagListBox;
};


/**List box item holding a compiler flag.*/
class FlagListBox : public TQListView
{
    Q_OBJECT
  TQ_OBJECT
public:
    FlagListBox( TQWidget *parent=0, const char *name=0 );
    ~FlagListBox()
        {}

    void readFlags(TQStringList *list);
    void writeFlags(TQStringList *list);
};


/**Check box item holding a compiler flag.*/
class FlagCheckBox : public TQCheckBox
{
    Q_OBJECT
  TQ_OBJECT
public:
    FlagCheckBox(TQWidget *parent, FlagCheckBoxController *controller,
                 const TQString &flagstr, const TQString &description);
    FlagCheckBox(TQWidget *parent, FlagCheckBoxController *controller,
                 const TQString &flagstr, const TQString &description,
                 const TQString &offstr);
    FlagCheckBox(TQWidget *parent, FlagCheckBoxController *controller,
                 const TQString &flagstr, const TQString &description,
                 const TQString &offstr, const TQString &defstr);
    ~FlagCheckBox()
        {}

private:
    TQString flag;
    TQString off;
    TQString def;
    bool includeOff;
    bool useDef;
    bool defSet;
    friend class FlagCheckBoxController;
};

/**Radiobutton item holding an option of a compiler flag.*/
class FlagRadioButton : public TQRadioButton
{
    Q_OBJECT
  TQ_OBJECT
public:
    FlagRadioButton(TQWidget *parent, FlagRadioButtonController *controller,
                 const TQString &flagstr, const TQString &description);
    ~FlagRadioButton()
        {}

private:
    TQString flag;
    friend class FlagRadioButtonController;
};

/**Path editor if path is a compiler flag.*/
class FlagPathEdit: public TQWidget
{
    Q_OBJECT
  TQ_OBJECT
public:
    /**If the pathDelimiter is not empty then path edit can contain a list of paths*/
    FlagPathEdit(TQWidget *parent, TQString pathDelimiter, FlagPathEditController *controller,
                 const TQString &flagstr, const TQString &description, KFile::Mode mode = KFile::Directory);

    ~FlagPathEdit() {}

    void setText(const TQString text);
    bool isEmpty();
    TQString text();

private slots:
    void showPathDetails();

private:
    KLineEdit *edit;
    TQPushButton *details;
    KURLRequester *url;

    TQString delimiter;
    TQString flag;
    TQString m_description;
    friend class FlagPathEditController;
};

/**List editor if list is a compiler flag.*/
class FlagListEdit: public TQWidget
{
    Q_OBJECT
  TQ_OBJECT
public:
    /**If the listDelimiter is not empty then list edit can contain a list of entries*/
    FlagListEdit(TQWidget *parent, TQString listDelimiter, FlagEditController *controller,
                 const TQString &flagstr, const TQString &description);

    ~FlagListEdit() {}

    void setText(const TQString text);
    void appendText(const TQString text);
    bool isEmpty();
    TQString text();
    TQStringList flags();

private slots:
    void showListDetails();

private:
    KLineEdit *edit;
    TQPushButton *details;

    TQString delimiter;
    TQString flag;
    TQString m_description;
    friend class FlagEditController;
};

/**Spin editor of a compiler flag.*/
class FlagSpinEdit: public TQWidget
{
public:
    FlagSpinEdit(TQWidget *parent, int minVal, int maxVal, int incr, int defaultVal, FlagEditController *controller,
                 const TQString &flagstr, const TQString &description);
    ~FlagSpinEdit() {}

    void setText(const TQString text);
    TQString text();
    bool isDefault();

    TQString flags();

private:
    int m_defaultVal;
    TQString flag;

    TQSpinBox *spb;

    friend class FlagEditController;
};

/**Controller for path editors.*/
class FlagPathEditController
{
public:
    FlagPathEditController();
    ~FlagPathEditController();

    void readFlags(TQStringList *list);
    void writeFlags(TQStringList *list);

private:
    void addPathEdit(FlagPathEdit *item);
    TQPtrList<FlagPathEdit> plist;
    friend class FlagPathEdit;
};

/**Controller for flag editors.*/
class FlagEditController
{
public:
    FlagEditController();
    ~FlagEditController();

    void readFlags(TQStringList *list);
    void writeFlags(TQStringList *list);

private:
    void addListEdit(FlagListEdit *item);
    void addSpinBox(FlagSpinEdit *item);
    TQPtrList<FlagListEdit> plist;
    TQPtrList<FlagSpinEdit>  slist;
    friend class FlagListEdit;
    friend class FlagSpinEdit;
};

/**Controller for check box editors.*/
class FlagCheckBoxController
{
public:
    /**"multi key" is a list of options like -vxyz (-vx -vy -vz)
       multiKeys must contain a list of option names like {-v}
       in the above example.
    */
    FlagCheckBoxController(TQStringList multiKeys = TQStringList());
    ~FlagCheckBoxController()
        {}

    void readFlags(TQStringList *list);
    void writeFlags(TQStringList *list);

    void addCheckBox(FlagCheckBox *item);
private:
    TQPtrList<FlagCheckBox> cblist;

    TQStringList m_multiKeys;
};

/**Controller for radiobutton editors.*/
class FlagRadioButtonController
{
public:
    /**multiKeys is a list of options like -vxyz (-vx -vy -vz)
       multiKeys must contain a list of option names like {-v}
       in the above example.
    */
    FlagRadioButtonController(TQStringList multiKeys = TQStringList());
    ~FlagRadioButtonController()
        {}

    void readFlags(TQStringList *list);
    void writeFlags(TQStringList *list);

    void addRadioButton(FlagRadioButton *item);
private:
    TQPtrList<FlagRadioButton> cblist;

    TQStringList m_multiKeys;
};

#endif
