/* This file is part of KDevelop
    Copyright (C) 2002,2003 Roberto Raggi <roberto@kdevelop.org>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public License
    along with this library; see the file COPYING.LIB.  If not, write to
    the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA 02110-1301, USA.
*/

#ifndef DRIVER_H
#define DRIVER_H

#include "JavaAST.hpp"

#include <tqpair.h>
#include <tqvaluestack.h>
#include <tqstringlist.h>
#include <tqmap.h>

class JavaLexer;
class JavaRecognizer;

class Problem
{
public:
    enum
    {
	Level_Error = 0,
	Level_Warning,
	Level_Todo,
	Level_Fixme
    };

public:
    Problem() {}
    Problem( const Problem& source )
	: m_text( source.m_text ), m_line( source.m_line ),
	  m_column( source.m_column ), m_level( source.m_level ) {}
    Problem( const TQString& text, int line, int column, int level=Level_Error )
	: m_text( text ), m_line( line ), m_column( column ), m_level(level) {}

    Problem& operator = ( const Problem& source )
    {
	m_text = source.m_text;
	m_line = source.m_line;
	m_column = source.m_column;
	m_level = source.m_level;
	return( *this );
    }

    bool operator == ( const Problem& p ) const
    {
	return m_text == p.m_text && m_line == p.m_line && m_column == p.m_column && m_level == p.m_level;
    }

    TQString text() const { return m_text; }
    int line() const { return m_line; }
    int column() const { return m_column; }
    int level() const { return m_level; }

private:
    TQString m_text;
    int m_line;
    int m_column;
    int m_level;
};

class SourceProvider
{
public:
    SourceProvider() {}
    virtual ~SourceProvider() {}

    virtual TQString contents( const TQString& fileName ) = 0;
    virtual bool isModified( const TQString& fileName ) = 0;

private:
    SourceProvider( const SourceProvider& source );
    void operator = ( const SourceProvider& source );
};

class Driver
{
public:
    Driver();
    virtual ~Driver();

    SourceProvider* sourceProvider();
    void setSourceProvider( SourceProvider* sourceProvider );

    virtual void reset();

    virtual void parseFile( const TQString& fileName, bool onlyPreProcesss=false, bool force=false );
    virtual void fileParsed( const TQString& fileName );
    virtual void remove( const TQString& fileName );

    virtual void addProblem( const TQString& fileName, const Problem& problem );

    TQString currentFileName() const { return m_currentFileName; }
    RefJavaAST takeTranslationUnit( const TQString& fileName );
    RefJavaAST translationUnit( const TQString& fileName ) const;
    TQValueList<Problem> problems( const TQString& fileName ) const;

    TQStringList includePaths() const { return m_includePaths; }
    virtual void addIncludePath( const TQString &path );

    const TQMap<TQString, RefJavaAST> &parsedUnits() const { return m_parsedUnits; }

protected:
    virtual void setupLexer( JavaLexer* lexer );
    virtual void setupParser( JavaRecognizer* parser );

private:
    TQValueList<Problem>& findOrInsertProblemList( const TQString& fileName );

private:
    TQString m_currentFileName;
    TQMap< TQString, TQValueList<Problem> > m_problems;
    TQMap< TQString, RefJavaAST > m_parsedUnits;
    TQStringList m_includePaths;
    JavaLexer *lexer;
    SourceProvider* m_sourceProvider;

private:
    Driver( const Driver& source );
    void operator = ( const Driver& source );
};

#endif
