/* ****************************************************************************
  This file is part of KBabel

  Copyright (C) 2002-2005  Stanislav Visnovsky
                            <visnovsky@kde.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  In addition, as a special exception, the copyright holders give
  permission to link the code of this program with any edition of
  the TQt library by Trolltech AS, Norway (or with modified versions
  of TQt that use the same license as TQt), and distribute linked
  combinations including the two.  You must obey the GNU General
  Public License in all respects for all of the code used other than
  TQt. If you modify this file, you may extend this exception to
  your version of the file, but you are not obligated to do so.  If
  you do not wish to do so, delete this exception statement from
  your version.

**************************************************************************** */
#ifndef CONTEXT_H
#define CONTEXT_H

#include <tqvaluelist.h>
#include <tqwidget.h>

#include <ktexteditor/document.h>
#include <ktexteditor/view.h>

#include <kbproject.h>

class TQComboBox;
class TQVBoxLayout;
class TQLineEdit;
class KListEditor;
class KConfig;
class KURL;

struct ContextInfo
{
    TQString path;
    uint line;
};

/**
 * @short Class for displaying source code context
 *
 * Widget for displaying source code context of for the given GNU gettext comment.
 * The searched paths can be configured using variables.
 *
 * The possible variables are:
 * - \@POFILEDIR\@   absolute directory of the PO file (to create paths relatives to the PO file)
 * - \@PACKAGE\@     name of the PO file
 * - \@PACKAGEDIR\@  relative directory of the PO file (relative to \@CODEROOT\@)
 * - \@CODEROOT\@    base directory (especially of the catalog manager)
 * - \@COMMENTPATH\@ (relative) path given as source code reference in a comment of the PO file
 *
 * @note The difference between \@POFILEDIR\@ and a path constructed by
 * \@CODEROOT\@\@PACKAGEDIR\@/ is that \@POFILEDIR\@ will also work if the file is external
 * to the catalog manager's root
 *
 * @note It requires a KPart implementing KTextEditor interface with selections.
 * @author Stanislav Visnovsky <visnovsky@kde.org>
 */
class KDE_EXPORT SourceContext : public TQWidget
{
    Q_OBJECT
  TQ_OBJECT
public:
    SourceContext(TQWidget* parent, KBabel::Project::Ptr project); 
    
    void setProject(KBabel::Project::Ptr project);

public slots:
    /**
     * Try to find the corresponding file and load it to this widget.
     * @param packageDir	path of the package, where to find the source file
     * @param packageName	name of the package, where to find the source file
     * @param gettextComment	comment string with context as generated by xgettext (can start with #:)
     * @param urlPoFile         URL of the PO file
     * @todo even if @p urlPoFile is an URL SourceContext::resolvePath is not remote-aware yet
     */
    void setContext( const TQString& packageDir, const TQString& packageName, const TQString& gettextComment, const KURL& urlPoFile );

private:
    /**
     * Get a list of paths from the source references in the comment @p gettextComment
     * @param packageDir        path of the package, where to find the source file
     * @param packageName       name of the package, where to find the source file
     * @param gettextComment    comment string with context as generated by xgettext (can start with #:)
     * @param urlPoFile         URL of the PO file
     * @todo even if @p urlPoFile is an URL SourceContext::resolvePath is not remote-aware yet
     * @private 
     */
    TQValueList<ContextInfo> resolvePath( const TQString& packageDir, const TQString& packageName, const TQString& gettextComment, const KURL& urlPoFile );
    bool loadPart();

    /// Parent widget (for KIO::NetAccess member functions)
    TQWidget* m_parent;
    KTextEditor::Document* _part;
    KTextEditor::View* _view;
    TQComboBox *_referenceCombo;
    TQVBoxLayout *_layout;
    
    TQValueList<ContextInfo> _referenceList;
    
    KBabel::Project::Ptr _project;
private slots:
    void updateToSelected(int index);
};

#endif // CONTEXT_H
