/*
 * acljobs.h
 *
 * Copyright (c) 2004 David Faure <faure@kde.org>
 *
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of this program with any edition of
 *  the TQt library by Trolltech AS, Norway (or with modified versions
 *  of TQt that use the same license as TQt), and distribute linked
 *  combinations including the two.  You must obey the GNU General
 *  Public License in all respects for all of the code used other than
 *  TQt.  If you modify this file, you may extend this exception to
 *  your version of the file, but you are not obligated to do so.  If
 *  you do not wish to do so, delete this exception statement from
 *  your version.
 */

#ifndef KMACLJOBS_H
#define KMACLJOBS_H

#include <kio/job.h>
#include <tqvaluevector.h>

namespace KMail {

  /// One entry in the ACL list: user and permissions
  struct ACLListEntry {
    ACLListEntry() {} // for TQValueVector
    ACLListEntry( const TQString& u, const TQString& irl, int p )
      : userId( u ), internalRightsList( irl ), permissions( p ), changed( false ) {}
    TQString userId;
    TQString internalRightsList; ///< protocol-dependent string (e.g. IMAP rights list)
    int permissions; ///< based on the ACLPermissions enum
    bool changed; ///< special flag for KMFolderCachedImap
  };

  typedef TQValueVector<ACLListEntry> ACLList;

/**
 * This namespace contains functions that return jobs for ACL operations.
 *
 * The current implementation is tied to IMAP.
 * If someone wants to extend this to other protocols, turn the class into a namespace
 * and use virtual methods.
 */
namespace ACLJobs {

  // Used by KMFolderCachedImap and KMFolderImap, no better place for that yet, until we have a
  // common base class for both
  enum ACLFetchState {
    NotFetchedYet, ///< The user rights/ACL have not been fetched from the server yet, we don't know them
    Ok,            ///< The user rights/ACL have been fetched from the server sucessfully
    FetchFailed    ///< The attempt to fetch the user rights/ACL from the server failed
  };

  /// Bitfield modelling the possible permissions.
  /// This is modelled after the imap4 permissions except that Read is "rs".
  /// The semantics of the bits is protocol-dependent.
  enum ACLPermissions {
    List = 1,
    Read = 2,
    WriteFlags = 4,
    Insert = 8,
    Create = 16,
    Delete = 32,
    Administer = 64,
    Post = 128,
    WriteSeenFlag = 256,
    // alias for "all read/write permissions except admin"
    AllWrite = List | Read | WriteFlags | Insert | Post | Create | Delete | WriteSeenFlag,
    // alias for "all permissions"
    All = List | Read | WriteFlags | Insert | Post | Create | Delete | Administer | WriteSeenFlag
  };
  /// Set the permissions for a given user on a given url
  KIO::SimpleJob* setACL( KIO::Slave* slave, const KURL& url, const TQString& user, unsigned int permissions );

  class DeleteACLJob;
  /// Delete the permissions for a given user on a given url
  DeleteACLJob* deleteACL( KIO::Slave* slave, const KURL& url, const TQString& user );

  class GetACLJob;
  /// List all ACLs for a given url
  GetACLJob* getACL( KIO::Slave* slave, const KURL& url );

  class GetUserRightsJob;
  /// Get the users' rights for a given url
  GetUserRightsJob* getUserRights( KIO::Slave* slave, const KURL& url );

  class MultiSetACLJob;
  /// Set and delete a list of permissions for different users on a given url
  MultiSetACLJob* multiSetACL( KIO::Slave* slave, const KURL& url, const ACLList& acl );

  /// List all ACLs for a given url
  class GetACLJob : public KIO::SimpleJob
  {
    Q_OBJECT
  TQ_OBJECT
  public:
    GetACLJob( const KURL& url, const TQByteArray &packedArgs,
               bool showProgressInfo );

    const ACLList& entries() const { return m_entries; }

  protected slots:
    void slotInfoMessage( KIO::Job*, const TQString& );
  private:
    ACLList m_entries;
  };

  /// Get the users' rights for a given url
  class GetUserRightsJob : public KIO::SimpleJob
  {
    Q_OBJECT
  TQ_OBJECT
  public:
    GetUserRightsJob( const KURL& url, const TQByteArray &packedArgs,
                      bool showProgressInfo );
    unsigned int permissions() const { return m_permissions; }

  protected slots:
    void slotInfoMessage( KIO::Job*, const TQString& );
  private:
    unsigned int m_permissions;
  };

  /// Delete the permissions for a given user on a given url
  /// This class only exists to store the userid in the job
  class DeleteACLJob : public KIO::SimpleJob
  {
    Q_OBJECT
  TQ_OBJECT
  public:
    DeleteACLJob( const KURL& url, const TQString& userId,
                  const TQByteArray &packedArgs,
                  bool showProgressInfo );

    TQString userId() const { return mUserId; }

  private:
    TQString mUserId;
  };

  /// Set and delete a list of permissions for different users on a given url
  class MultiSetACLJob : public KIO::Job {
    Q_OBJECT
  TQ_OBJECT

  public:
    MultiSetACLJob( KIO::Slave* slave, const KURL& url, const ACLList& acl, bool showProgressInfo );

  signals:
    // Emitted when a given user's permissions were successfully changed.
    // This allows the caller to keep track of what exactly was done (and handle errors better)
    void aclChanged( const TQString& userId, int permissions );

  protected slots:
    virtual void slotStart();
    virtual void slotResult( KIO::Job *job );

  private:
    KIO::Slave* mSlave;
    const KURL mUrl;
    const ACLList mACLList;
    ACLList::const_iterator mACLListIterator;
  };


#ifndef NDEBUG
  TQString permissionsToString( unsigned int permissions );
#endif
}

} // namespace

#endif /* KMACLJOBS_H */
