/*
 * filetransfer.h - File Transfer
 * Copyright (C) 2004  Justin Karneges
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef XMPP_FILETRANSFER_H
#define XMPP_FILETRANSFER_H

#include "im.h"

namespace XMPP
{
	class S5BConnection;
	struct FTRequest;

	class FileTransfer : public TQObject
	{
		Q_OBJECT
  TQ_OBJECT
	public:
		enum { ErrReject, ErrNeg, ErrConnect, ErrProxy, ErrStream };
		enum { Idle, Requesting, Connecting, WaitingForAccept, Active };
		~FileTransfer();

		void setProxy(const Jid &proxy);

		// send
		void sendFile(const Jid &to, const TQString &fname, TQ_LLONG size, const TQString &desc);
		TQ_LLONG offset() const;
		TQ_LLONG length() const;
		int dataSizeNeeded() const;
		void writeFileData(const TQByteArray &a);

		// receive
		Jid peer() const;
		TQString fileName() const;
		TQ_LLONG fileSize() const;
		TQString description() const;
		bool rangeSupported() const;
		void accept(TQ_LLONG offset=0, TQ_LLONG length=0);

		// both
		void close(); // reject, or stop sending/receiving
		S5BConnection *s5bConnection() const; // active link

	signals:
		void accepted(); // indicates S5BConnection has started
		void connected();
		void readyRead(const TQByteArray &a);
		void bytesWritten(int);
		void error(int);

	private slots:
		void ft_finished();
		void s5b_connected();
		void s5b_connectionClosed();
		void s5b_readyRead();
		void s5b_bytesWritten(int);
		void s5b_error(int);
		void doAccept();

	private:
		class Private;
		Private *d;

		void reset();

		friend class FileTransferManager;
		FileTransfer(FileTransferManager *, TQObject *parent=0);
		void man_waitForAccept(const FTRequest &req);
		void takeConnection(S5BConnection *c);
	};

	class FileTransferManager : public TQObject
	{
		Q_OBJECT
  TQ_OBJECT
	public:
		FileTransferManager(Client *);
		~FileTransferManager();

		Client *client() const;
		FileTransfer *createTransfer();
		FileTransfer *takeIncoming();

	signals:
		void incomingReady();

	private slots:
		void pft_incoming(const FTRequest &req);

	private:
		class Private;
		Private *d;

		friend class Client;
		void s5b_incomingReady(S5BConnection *);

		friend class FileTransfer;
		TQString link(FileTransfer *);
		void con_accept(FileTransfer *);
		void con_reject(FileTransfer *);
		void unlink(FileTransfer *);
	};

	class JT_FT : public Task
	{
		Q_OBJECT
  TQ_OBJECT
	public:
		JT_FT(Task *parent);
		~JT_FT();

		void request(const Jid &to, const TQString &id, const TQString &fname, TQ_LLONG size, const TQString &desc, const TQStringList &streamTypes);
		TQ_LLONG rangeOffset() const;
		TQ_LLONG rangeLength() const;
		TQString streamType() const;

		void onGo();
		bool take(const TQDomElement &);

	private:
		class Private;
		Private *d;
	};

	struct FTRequest
	{
		Jid from;
		TQString iq_id, id;
		TQString fname;
		TQ_LLONG size;
		TQString desc;
		bool rangeSupported;
		TQStringList streamTypes;
	};
	class JT_PushFT : public Task
	{
		Q_OBJECT
  TQ_OBJECT
	public:
		JT_PushFT(Task *parent);
		~JT_PushFT();

		void respondSuccess(const Jid &to, const TQString &id, TQ_LLONG rangeOffset, TQ_LLONG rangeLength, const TQString &streamType);
		void respondError(const Jid &to, const TQString &id, int code, const TQString &str);

		bool take(const TQDomElement &);

	signals:
		void incoming(const FTRequest &req);
	};
}

#endif
