/*
 * configaccess.cpp
 *
 * Copyright (c) 2001 Frerich Raabe <raabe@kde.org>
 *
 * This program is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS
 * FOR A PARTICULAR PURPOSE. For licensing and distribution details, check the
 * accompanying file 'COPYING'.
 */
#include "configaccess.h"
#include "newsengine.h"

#include <tqregexp.h>

#include <kdebug.h>
#include <kglobal.h>
#include <kglobalsettings.h>

static NewsSourceBase::Data NewsSourceDefault[DEFAULT_NEWSSOURCES] = {
	// Arts ---------------
		NewsSourceBase::Data(
		TQString::fromLatin1("Bureau 42"),
		TQString::fromLatin1("http://www.bureau42.com/rdf/"),
		TQString::fromLatin1("http://www.bureau42.com/favicon.ico"),
		NewsSourceBase::Arts, 5, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("eFilmCritic"),
		TQString::fromLatin1("http://efilmcritic.com/fo.rdf"),
		TQString::fromLatin1("http://efilmcritic.com/favicon.ico"),
		NewsSourceBase::Arts, 3, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("superhits.ch"),
		TQString::fromLatin1("http://www.superhits.ch/cgi-bin/superhits.cgi?page=rdf"),
		TQString::fromLatin1("http://www.superhits.ch/favicon.ico"),
		NewsSourceBase::Arts, 10, false, false, TQString::fromLatin1("de")),
	// Business -----------
		NewsSourceBase::Data(
		TQString::fromLatin1("Internet.com Business"),
		TQString::fromLatin1("http://headlines.internet.com/internetnews/bus-news/news.rss"),
		TQString(),
		NewsSourceBase::Business, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("TradeSims"),
		TQString::fromLatin1("http://www.tradesims.com/AEX.rdf"),
		TQString(),
		NewsSourceBase::Business, 10, false, false),
	// Computers ----------
		NewsSourceBase::Data(
		TQString::fromLatin1( "linuxartist.org" ),
		TQString::fromLatin1( "http://www.linuxartist.org/backend.php"),
		TQString::fromLatin1( "http://www.linuxartist.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false ),
		NewsSourceBase::Data(
		TQString::fromLatin1("KDE Deutschland"),
		TQString::fromLatin1("http://www.kde.de/nachrichten/nachrichten.rdf"),
		TQString::fromLatin1("http://www.kde.de/favicon.ico"),
		NewsSourceBase::Computers, 10, true, false, TQString::fromLatin1("de")),
		NewsSourceBase::Data(
		TQString::fromLatin1("KDE France"),
		TQString::fromLatin1("http://www.kde-france.org/backend-breves.php3"),
		TQString(),
		NewsSourceBase::Computers, 10, true, false, TQString::fromLatin1("fr")),
		NewsSourceBase::Data(
		TQString::fromLatin1("FreeBSD Project News"),
		TQString::fromLatin1("http://www.freebsd.org/news/news.rdf"),
		TQString::fromLatin1("http://www.freebsd.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("dot.kde.org"),
		TQString::fromLatin1("http://www.kde.org/dotkdeorg.rdf"),
		TQString::fromLatin1("http://www.kde.org/favicon.ico"),
		NewsSourceBase::Computers, 10, true, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("GNOME News"),
		TQString::fromLatin1("http://www.gnomedesktop.org/backend.php"),
		TQString(),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("Slashdot"),
		TQString::fromLatin1("http://slashdot.org/slashdot.rdf"),
		TQString::fromLatin1("http://slashdot.org/favicon.ico"),
		NewsSourceBase::Computers, 10, true, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("Ask Slashdot"),
		TQString::fromLatin1("http://slashdot.org/askslashdot.rdf"),
		TQString::fromLatin1("http://slashdot.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("Slashdot: Features"),
		TQString::fromLatin1("http://slashdot.org/features.rdf"),
		TQString::fromLatin1("http://slashdot.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("Slashdot: Apache"),
		TQString::fromLatin1("http://slashdot.org/apache.rdf"),
		TQString::fromLatin1("http://slashdot.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("Slashdot: Books"),
		TQString::fromLatin1("http://slashdot.org/books.rdf"),
		TQString::fromLatin1("http://slashdot.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("Jabber News"),
		TQString::fromLatin1("http://www.jabber.org/news/rss.xml"),
		TQString(),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("Freshmeat"),
		TQString::fromLatin1("http://freshmeat.net/backend/fm.rdf"),
		TQString::fromLatin1("http://freshmeat.net/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("Linux Weekly News"),
		TQString::fromLatin1("http://www.lwn.net/headlines/rss"),
		TQString::fromLatin1("http://www.lwn.net/favicon.ico"),
		NewsSourceBase::Computers, 10, true, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("heise online news"),
		TQString::fromLatin1("http://www.heise.de/newsticker/heise.rdf"),
		TQString::fromLatin1("http://www.heise.de/favicon.ico"),
		NewsSourceBase::Computers, 10, true, false, TQString::fromLatin1("de")),
		NewsSourceBase::Data(
		TQString::fromLatin1("RUS-CERT Ticker"),
		TQString::fromLatin1("http://cert.uni-stuttgart.de/ticker/rus-cert.rdf"),
		TQString::fromLatin1("http://cert.uni-stuttgart.de/favicon.ico"),
		NewsSourceBase::Computers, 10, true, false, TQString::fromLatin1("de")),
		NewsSourceBase::Data(
		TQString::fromLatin1("RUS-CERT Elsewhere"),
		TQString::fromLatin1("http://cert.uni-stuttgart.de/ticker/rus-cert-elsewhere.rdf"),
		TQString::fromLatin1("http://cert.uni-stuttgart.de/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false, TQString::fromLatin1("de")),
		NewsSourceBase::Data(
		TQString::fromLatin1("Kuro5hin"),
		TQString::fromLatin1("http://kuro5hin.org/backend.rdf"),
		TQString::fromLatin1("http://kuro5hin.org/favicon.ico"),
		NewsSourceBase::Computers, 10, true, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("Prolinux"),
		TQString::fromLatin1("http://www.pl-forum.de/backend/pro-linux.rdf"),
		TQString::fromLatin1("http://www.prolinux.de/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false, TQString::fromLatin1("de")),
		NewsSourceBase::Data(
		TQString::fromLatin1("Linuxde.org"),
		TQString::fromLatin1("http://www.linuxde.org/backends/news.rdf"),
		TQString::fromLatin1("http://www.linuxde.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false, TQString::fromLatin1("de")),
		NewsSourceBase::Data(
		TQString::fromLatin1("LinuxSecurity.com"),
		TQString::fromLatin1("http://www.linuxsecurity.com/linuxsecurity_hybrid.rdf"),
		TQString(),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("Linux Game Tome"),
		TQString::fromLatin1("http://happypenguin.org/html/news.rdf"),
		TQString(),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("Telefragged"),
		TQString::fromLatin1("http://www.telefragged.com/cgi-bin/rdf.pl"),
		TQString(),
		NewsSourceBase::Computers, 10, false, false),
/*		NewsSourceBase::Data(
		TQString::fromLatin1("Gimp News"),
		TQString::fromLatin1("http://www.xach.com/gimp/news/channel.rdf"),
		TQString(),
		NewsSourceBase::Computers, 10, false, false),*/
		NewsSourceBase::Data(
		TQString::fromLatin1("Mozilla"),
		TQString::fromLatin1("http://www.mozilla.org/news.rdf"),
		TQString::fromLatin1("http://www.mozillazine.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("MozillaZine"),
		TQString::fromLatin1("http://www.mozillazine.org/contents.rdf"),
		TQString::fromLatin1("http://www.mozillazine.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("Daemon News"),
		TQString::fromLatin1("http://daily.daemonnews.org/ddn.rdf.php3"),
		TQString(),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("use Perl;"),
		TQString::fromLatin1("http://use.perl.org/useperl.rdf"),
		TQString(),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("desktopian.org"),
		TQString::fromLatin1("http://www.desktopian.org/includes/headlines.xml"),
		TQString::fromLatin1("http://www.desktopian.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("Root prompt"),
		TQString::fromLatin1("http://www.rootprompt.org/rss/"),
		TQString::fromLatin1("http://www.rootprompt.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("SecurityFocus"),
		TQString::fromLatin1("http://www.securityfocus.com/rss/news.xml"),
		TQString::fromLatin1("http://www.securityfocus.com/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("LinuxNewbie"),
		TQString::fromLatin1("http://www.linuxnewbie.org/news.cdf"),
		TQString::fromLatin1("http://www.linuxnewbie.org/favicon.ico"),
		NewsSourceBase::Computers, 5, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("Arstechnica"),
		TQString::fromLatin1("http://arstechnica.com/etc/rdf/ars.rdf"),
		TQString::fromLatin1("http://arstechnica.com/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("amiga-news.de - deutschsprachige Amiga Nachrichten"),
		TQString::fromLatin1("http://www.amiga-news.de/de/backends/news/index.rss"),
		TQString::fromLatin1("http://www.amiga-news.de/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false, TQString::fromLatin1("de")),
		NewsSourceBase::Data(
		TQString::fromLatin1("amiga-news.de - english Amiga news"),
		TQString::fromLatin1("http://www.amiga-news.de/en/backends/news/index.rss"),
		TQString::fromLatin1("http://www.amiga-news.de/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("RadioTux)"),
		TQString::fromLatin1("http://blog.radiotux.de/feed/"),
		TQString(),
		NewsSourceBase::Computers, 10, false, false, TQString::fromLatin1("de")),
		NewsSourceBase::Data(
		TQString::fromLatin1("kdenews.unixcode.org"),
		TQString::fromLatin1("http://kdenews.unixcode.org/?node=news&action=rss"),
		TQString(),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("FreshPorts - the place for ports"),
		TQString::fromLatin1("http://www.freshports.org/news.php"),
		TQString::fromLatin1("http://www.freshports.org/favicon.ico"),
		NewsSourceBase::Computers, 20, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("NetPhoenix"),
		TQString::fromLatin1("http://www.netphoenix.at/rss/netphoenix.php"),
		TQString::fromLatin1("http://www.netphoenix.at/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false, TQString::fromLatin1("de")),
		NewsSourceBase::Data(
		TQString::fromLatin1("ShortNews - by www.netphoenix.at"),
		TQString::fromLatin1("http://www.netphoenix.at/rss/shortnews.php"),
		TQString::fromLatin1("http://www.netphoenix.at/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false, TQString::fromLatin1("de")),
		NewsSourceBase::Data(
		TQString::fromLatin1("zez.org - about code "),
		TQString::fromLatin1("http://zez.org/article/rssheadlines"),
		TQString(),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("BSDatwork.com"),
		TQString::fromLatin1("http://BSDatwork.com/backend.php"),
		TQString::fromLatin1("http://BSDatwork.com/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("FreshSource - the place for source"),
		TQString::fromLatin1("http://www.freshsource.org/news.php"),
		TQString::fromLatin1("http://www.freshsource.org/favicon.ico"),
		NewsSourceBase::Computers, 20, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("The FreeBSD Diary"),
		TQString::fromLatin1("http://www.freebsddiary.org/news.php"),
		TQString::fromLatin1("http://www.freebsddiary.org/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("MaximumBSD"),
		TQString::fromLatin1("http://www.maximumbsd.com/backend/mb.rdf"),
		TQString::fromLatin1("http://www.maximumbsd.com/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("BR-Linux.org"),
		TQString::fromLatin1("http://br-linux.org/noticias/index.rdf"),
		TQString::fromLatin1("http://br-linux.org/noticias/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("OSNews.com"),
		TQString::fromLatin1("http://www.osnews.com/files/recent.rdf"),
		TQString::fromLatin1("http://www.osnews.com/favicon.ico"),
		NewsSourceBase::Computers, 10, false, false),
	// Miscellaneous ------
		NewsSourceBase::Data(
		TQString::fromLatin1("tagesschau.de"),
		TQString::fromLatin1("http://www.tagesschau.de/newsticker.rdf"),
		TQString::fromLatin1("http://www.tagesschau.de/favicon.ico"),
		NewsSourceBase::Misc, 10, true, false, TQString::fromLatin1("de")),
		NewsSourceBase::Data(
		TQString::fromLatin1("N24.de"),
		TQString::fromLatin1("http://www.n24.de/rss/?rubrik=home"),
		TQString::fromLatin1("http://www.n24.de/favicon.ico"),
		NewsSourceBase::Misc, 10, true, false, TQString::fromLatin1("de")),
/*
		NewsSourceBase::Data(
		TQString::fromLatin1("CNN"),
		TQString::fromLatin1("http://www.cnn.com/cnn.rss"),
		TQString::fromLatin1("http://www.cnn.com/favicon.ico"),
		NewsSourceBase::Misc, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("CNN Europe"),
		TQString::fromLatin1("http://europe.cnn.com/cnn.rss"),
		TQString::fromLatin1("http://europe.cnn.com/favicon.ico"),
		NewsSourceBase::Misc, 10, false, false),
*/
		NewsSourceBase::Data(
		TQString::fromLatin1("HotWired"),
		TQString::fromLatin1("http://www.hotwired.com/webmonkey/meta/headlines.rdf"),
		TQString::fromLatin1("http://www.hotwired.com/favicon.ico"),
		NewsSourceBase::Misc, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1("The Register"),
		TQString::fromLatin1("http://www.theregister.co.uk/tonys/slashdot.rdf"),
		TQString::fromLatin1("http://www.theregister.co.uk/favicon.ico"),
		NewsSourceBase::Misc, 10, false, false),
		NewsSourceBase::Data(
		TQString::fromLatin1( "Christian Science Monitor" ),
		TQString::fromLatin1( "http://www.csmonitor.com/rss/csm.rss"),
		TQString::fromLatin1( "http://www.csmonitor.com/favicon.ico"),
		NewsSourceBase::Misc, 10, false, false ),
	// Magazines ------
		NewsSourceBase::Data(
		TQString::fromLatin1("Revista do Linux"),
		TQString::fromLatin1("http://www.revistadolinux.com.br/noticias/arquivo/noticias.rdf"),
		TQString::fromLatin1("http://www.revistadolinux.com.br/favicon.ico"),
		NewsSourceBase::Magazines, 10, false, false /*, TQString::fromLatin1("br")*/ ),
		NewsSourceBase::Data(
		TQString::fromLatin1("Spiegel.de"),
		TQString::fromLatin1("http://www.spiegel.de/schlagzeilen/rss/0,5291,,00.xml"),
		TQString::fromLatin1("http://www.spiegel.de/favicon.ico"),
		NewsSourceBase::Magazines, 10, true, false, TQString::fromLatin1("de")),
		NewsSourceBase::Data(
		TQString::fromLatin1("FAZ.de"),
		TQString::fromLatin1("http://www.faz.net/s/Rub/Tpl~Epartner~SRss_.xml"),
		TQString::fromLatin1("http://www.faz.net/favicon.ico"),
		NewsSourceBase::Magazines, 10, true, false, TQString::fromLatin1("de")),
	// Recreation
		NewsSourceBase::Data(
		TQString::fromLatin1("Segfault"),
		TQString::fromLatin1("http://segfault.org/stories.xml"),
		TQString::fromLatin1("http://segfault.org/favicon.ico"),
		NewsSourceBase::Recreation, 10, false, false),
	// Society
		NewsSourceBase::Data(
		TQString::fromLatin1("nippon.it"),
		TQString::fromLatin1("http://www.nippon.it/backend.it.php"),
		TQString::fromLatin1("http://www.nippon.it/favicon.ico"),
		NewsSourceBase::Society, 10, false, false, TQString::fromLatin1("it")),
		NewsSourceBase::Data( 
		TQString::fromLatin1( "gflash" ),
		TQString::fromLatin1( "http://www.gflash.de/backend.php"),
		TQString::fromLatin1( "http://www.gflash.de/favicon.ico"),
		NewsSourceBase::Society, 10, false, false, TQString::fromLatin1( "de" ) ),
		NewsSourceBase::Data( 
		TQString::fromLatin1( "Quintessenz" ),
		TQString::fromLatin1( "http://quintessenz.at/cgi-bin/rdf"),
		TQString::fromLatin1( "http://quintessenz.at/favicon.ico"),
		NewsSourceBase::Society, 9, false, false, TQString::fromLatin1( "de" ) )
};

ArticleFilter::ArticleFilter(const TQString &action, const TQString &newsSource,
		const TQString &condition, const TQString &expression, bool enabled)
	: m_action(action),
	m_newsSource(newsSource),
	m_condition(condition),
	m_expression(expression),
	m_enabled(enabled)
{
}

bool ArticleFilter::matches(Article::Ptr a) const
{
	if (!enabled() ||
	    (a->newsSource()->data().name != newsSource() &&
	    newsSource() != i18n("all news sources")))
		return false;
	
	bool matches;

	if (condition() == i18n("contain"))
		matches = a->headline().contains(expression());
	else if (condition() == i18n("do not contain"))
		matches = !a->headline().contains(expression());
	else if (condition() == i18n("equal"))
		matches = (a->headline() == expression());
	else if (condition() == i18n("do not equal"))
		matches = (a->headline() != expression());
	else { // condition() == i18n("match")
		TQRegExp regexp = TQRegExp(expression());
		matches = regexp.exactMatch(a->headline());
	}
	
	if (action() == i18n("Show"))
		matches = !matches;

	return matches;
}
	
ConfigAccess::ConfigAccess()
	: m_defaultCfg(new KConfig(TQString(), true, false))
{
	m_cfg = m_defaultCfg;
}

ConfigAccess::ConfigAccess(KConfig *config)
	: m_cfg(config), m_defaultCfg(0L)
{
	m_cfg->setGroup("KNewsTicker");
}

ConfigAccess::~ConfigAccess()
{
	delete m_defaultCfg;
}

unsigned int ConfigAccess::interval() const
{
	return m_cfg->readNumEntry("Update interval", 30);
}

void ConfigAccess::setInterval(const unsigned int interval)
{
	m_cfg->writeEntry("Update interval", interval);
	m_cfg->sync();
}

unsigned int ConfigAccess::mouseWheelSpeed() const
{
	return m_cfg->readNumEntry("Mouse wheel speed", 5);
}

void ConfigAccess::setMouseWheelSpeed(const unsigned int mouseWheelSpeed)
{
	m_cfg->writeEntry("Mouse wheel speed", mouseWheelSpeed);
	m_cfg->sync();
}

TQFont ConfigAccess::font() const
{
	TQFont font = KGlobalSettings::fixedFont();
	return m_cfg->readFontEntry("Font", &font);
}

void ConfigAccess::setFont(const TQFont &font)
{
	m_cfg->writeEntry("Font", font);
	m_cfg->sync();
}

bool ConfigAccess::customNames() const
{
	return m_cfg->readBoolEntry("Custom names", false);
}

void ConfigAccess::setCustomNames(bool customNames)
{
	m_cfg->writeEntry("Custom names", customNames);
	m_cfg->sync();
}

bool ConfigAccess::scrollMostRecentOnly() const
{
	return m_cfg->readBoolEntry("Scroll most recent headlines only", false);
}

void ConfigAccess::setScrollMostRecentOnly(bool scrollMostRecentOnly)
{
	m_cfg->writeEntry("Scroll most recent headlines only", scrollMostRecentOnly);
	m_cfg->sync();
}

bool ConfigAccess::offlineMode() const
{
	return m_cfg->readBoolEntry("Offline mode", false);
}

void ConfigAccess::setOfflineMode(bool offlineMode)
{
	m_cfg->writeEntry("Offline mode", offlineMode);
	m_cfg->sync();
}

TQStringList ConfigAccess::newsSources() const
{
	TQStringList tempList = m_cfg->readListEntry("News sources");
	if (tempList.isEmpty())
		for (unsigned int i = 0; i < DEFAULT_NEWSSOURCES; i++)
			tempList << NewsSourceDefault[i].name;
	return tempList;
}

ArticleFilter::List ConfigAccess::filters() const
{
	return m_cfg->readIntListEntry("Filters");
}

void ConfigAccess::setNewsSources(const TQStringList &newsSources)
{
	m_cfg->writeEntry("News sources", newsSources);
	m_cfg->sync();
}

void ConfigAccess::setFilters(const ArticleFilter::List &filters)
{
	m_cfg->writeEntry("Filters", filters);
	m_cfg->sync();
}

unsigned int ConfigAccess::scrollingSpeed() const
{
	return m_cfg->readNumEntry("Scrolling speed", 20);
}

void ConfigAccess::setScrollingSpeed(const unsigned int scrollingSpeed)
{
	m_cfg->writeEntry("Scrolling speed", scrollingSpeed);
	m_cfg->sync();
}

unsigned int ConfigAccess::scrollingDirection() const
{
	return m_cfg->readNumEntry("Scrolling direction", 0);
}

void ConfigAccess::setScrollingDirection(const unsigned int scrollingDirection)
{
	m_cfg->writeEntry("Scrolling direction", scrollingDirection);
	m_cfg->sync();
}

TQColor ConfigAccess::foregroundColor() const
{
	return m_cfg->readColorEntry("Foreground color", &TQt::black);
}

void ConfigAccess::setForegroundColor(const TQColor &foregroundColor)
{
	m_cfg->writeEntry("Foreground color", foregroundColor);
	m_cfg->sync();
}

TQColor ConfigAccess::backgroundColor() const
{
	return m_cfg->readColorEntry("Background color", &TQt::white);
}

void ConfigAccess::setBackgroundColor(const TQColor &backgroundColor)
{
	m_cfg->writeEntry("Background color", backgroundColor);
	m_cfg->sync();
}

TQColor ConfigAccess::highlightedColor() const
{
	return m_cfg->readColorEntry("Highlighted color", &TQt::red);
}

void ConfigAccess::setHighlightedColor(const TQColor &highlightedColor)
{
	m_cfg->writeEntry("Highlighted color", highlightedColor);
	m_cfg->sync();
}

bool ConfigAccess::underlineHighlighted() const
{
	return m_cfg->readBoolEntry("Underline highlighted headlines", true);
}

void ConfigAccess::setUnderlineHighlighted(bool underlineHighlighted)
{
	m_cfg->writeEntry("Underline highlighted headlines", underlineHighlighted);
	m_cfg->sync();
}

NewsSourceBase *ConfigAccess::newsSource(const TQString &newsSource)
{
	NewsSourceBase::Data nsd;

	if (m_cfg->hasGroup(newsSource)) {
		m_cfg->setGroup(newsSource);
		nsd.name = newsSource;
		nsd.sourceFile = m_cfg->readPathEntry("Source file");
		nsd.isProgram = m_cfg->readBoolEntry("Is program", false);
		nsd.subject = static_cast<NewsSourceBase::Subject>
			(m_cfg->readNumEntry("Subject", NewsSourceBase::Computers));
		nsd.icon = m_cfg->readEntry("Icon");
		nsd.maxArticles = m_cfg->readNumEntry("Max articles", 10);
		nsd.enabled = m_cfg->readBoolEntry("Enabled", true);
		nsd.language = m_cfg->readEntry("Language", TQString::fromLatin1("C"));
		m_cfg->setGroup("KNewsTicker");
	} else for (unsigned int i = 0; i < DEFAULT_NEWSSOURCES; i++)
		if (NewsSourceDefault[i].name == newsSource) {
			nsd = NewsSourceDefault[i];
			if (nsd.enabled)
				nsd.enabled = (nsd.language == TQString::fromLatin1("C") ||
						KGlobal::locale()->languageList().contains(nsd.language));
			break;
		}
	
	if (nsd.isProgram)
		return new ProgramNewsSource(nsd, this);
	else 
		return new SourceFileNewsSource(nsd, this);
	
	return 0L;
}

ArticleFilter ConfigAccess::filter(const unsigned int filterNo) const
{
	ArticleFilter f;
	f.setId(filterNo);

	if (m_cfg->hasGroup(TQString::fromLatin1("Filter #%1").arg(filterNo))) {
		m_cfg->setGroup(TQString::fromLatin1("Filter #%1").arg(filterNo));
		f.setAction(m_cfg->readEntry("Action", i18n("Show")));
		f.setNewsSource(m_cfg->readEntry("News source", i18n("all news sources")));
		f.setCondition(m_cfg->readEntry("Condition", i18n("contain")));
		f.setExpression(m_cfg->readEntry("Expression"));
		f.setEnabled(m_cfg->readBoolEntry("Enabled", true));
		m_cfg->setGroup("KNewsTicker");
	}

	return f;
}

void ConfigAccess::setNewsSource(const NewsSourceBase::Data &ns)
{
	m_cfg->setGroup(ns.name);
	m_cfg->writePathEntry("Source file", ns.sourceFile);
	m_cfg->writeEntry("Is program", ns.isProgram);
	m_cfg->writeEntry("Max articles", ns.maxArticles);
	m_cfg->writeEntry("Subject", ns.subject);
	m_cfg->writeEntry("Icon", ns.icon);
	m_cfg->writeEntry("Enabled", ns.enabled);
	m_cfg->writeEntry("Language", ns.language);
	m_cfg->setGroup("KNewsTicker");
	m_cfg->sync();
}

void ConfigAccess::setFilter(const ArticleFilter &f)
{
	m_cfg->setGroup(TQString::fromLatin1("Filter #%1").arg(f.id()));
	m_cfg->writeEntry("Action", f.action());
	m_cfg->writeEntry("News source", f.newsSource());
	m_cfg->writeEntry("Condition", f.condition());
	m_cfg->writeEntry("Expression", f.expression());
	m_cfg->writeEntry("Enabled", f.enabled());
	m_cfg->setGroup("KNewsTicker");
	m_cfg->sync();
}

bool ConfigAccess::showIcons() const
{
	return m_cfg->readBoolEntry("Show icons", true);
}

void ConfigAccess::setShowIcons(bool showIcons)
{
	m_cfg->writeEntry("Show icons", showIcons);
	m_cfg->sync();
}

bool ConfigAccess::slowedScrolling() const
{
	return m_cfg->readBoolEntry("Slowed scrolling", false);
}

void ConfigAccess::setSlowedScrolling(bool slowedScrolling)
{
	m_cfg->writeEntry("Slowed scrolling", slowedScrolling);
	m_cfg->sync();
}

