/*
  Copyright (c) 2002-2004 Jan Schaefer <j_schaef@informatik.uni-kl.de>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

*/

#ifndef SAMBAFILE_H
#define SAMBAFILE_H


#include <tqdict.h>
#include <tqstring.h>
#include <tqobject.h>
#include <kio/job.h>

#include "sambashare.h"


class KSimpleConfig;
class KProcess;
class KConfig;
class KTempFile;

class SambaFile;

class SambaFileLoadException
{
  public:
    SambaFileLoadException(const TQString & msg) { message = msg; };
    
    TQString getMessage() { return message; };
  protected:
    TQString message;
};

class SambaConfigFile : public TQDict<SambaShare>
{
public:
  SambaConfigFile(SambaFile*);
  TQString getDefaultValue(const TQString & name);
  TQStringList getShareList();
  void addShare(const TQString & name, SambaShare* share);
  SambaShare* addShare(const TQString & name);
  void removeShare(const TQString & name);
protected:
  SambaFile* _sambaFile;
  TQStringList _shareList;
};

class SambaFile : public TQObject
{
Q_OBJECT
  TQ_OBJECT
public: 
	SambaFile(const TQString & _path, bool _readonly=true);
	~SambaFile();

  /** No descriptions */
  TQString findShareByPath(const TQString & path) const;
  void removeShareByPath(const TQString & path);
  
  
  SambaShare* newShare(const TQString & name);
  SambaShare* newShare(const TQString & name, const TQString & path);
  SambaShare* newPrinter(const TQString & name, const TQString & printer);

  void removeShare(const TQString & share);
  void removeShare(SambaShare* share);

	void renameShare(const TQString & oldName, const TQString & newName);

  SambaShare* getShare(const TQString & share) const;
  
  /**
   * Returns a list of all shared directories
   **/
  SambaShareList* getSharedDirs() const;

  /**
   * Returns a list of all shared printers
   **/
  SambaShareList* getSharedPrinters() const;

  /**
   * Returns a name which isn't already used for a share
   * if the alreadyUsedName parameter is given, then
   * a name based on this name is returned.
   * E.g.: if public is already used, the method could return 
   * public2
   **/
	TQString getUnusedName(const TQString alreadyUsedName=TQString()) const;

  /**
   * Returns all values of the global section
   * which are returned by the testparam program
   * if the values were already loaded then these
   * values are returned except the reload parameter
   * is true
   **/
  SambaShare* getTestParmValues(bool reload=false);

  static TQString findSambaConf();


  int getSambaVersion();

  /**
   * Load all data from the smb.conf file
   * Can load a remote file
   **/
  bool load();

  bool save();
  
  /**
   * Reads the local smb.conf
   */
  bool openFile();
  
  bool isRemoteFile();

  /**
   * Save all data to the specified file
   * if successful returns true otherwise false
   **/
  bool saveTo(const TQString & path);

protected:
  bool readonly;
  bool changed;
  TQString path;
  TQString localPath;
  SambaConfigFile *_sambaConfig;
  SambaShare* _testParmValues;
  TQString _parmOutput;
  int _sambaVersion;

  void parseParmStdOutput();
  SambaConfigFile* getSambaConfigFile(KSimpleConfig* config);
  KSimpleConfig* getSimpleConfig(SambaConfigFile* sambaConfig, const TQString & filename);



private:
  void copyConfigs(KConfig* first, KConfig* second);
  KTempFile * _tempFile;
public slots:

  /**
   * Saves all changes to the smb.conf file
   * if the user is not allowed she'll be asked
   * for a root password
   **/
  bool slotApply();
protected slots:
  void testParmStdOutReceived(KProcess *proc, char *buffer, int buflen);
  void slotJobFinished( KIO::Job *);
  void slotSaveJobFinished( KIO::Job *);

signals:
  void canceled(const TQString &);
  void completed();  
  
};

#endif
