/*
 *  ksokoban - a Sokoban game for KDE
 *  Copyright (C) 1998  Anders Widell  <d95-awi@nada.kth.se>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#ifndef PLAYFIELD_H
#define PLAYFIELD_H

#include <tqwidget.h>
#include <tqstring.h>
#include <tqfont.h>
#include <tqfontmetrics.h>
#include <tqpixmap.h>
#include <tqimage.h>
#include <tqbrush.h>

#include "ImageData.h"
#include "LevelMap.h"
class MapDelta;
class MoveSequence;
class Move;
#include "PathFinder.h"

class History;
class Bookmark;
class LevelCollection;
class TQPainter;
class TQCursor;

class PlayField : public TQWidget {
  Q_OBJECT
  TQ_OBJECT
public:
  PlayField(TQWidget *parent, const char *name=0, WFlags f=0);
  ~PlayField ();

  bool canMoveNow();
  int animDelay() { return animDelay_; }

  void setSize(int w, int h);
  void level(int _l) { levelMap_->level(_l); }
  LevelCollection  *collection() const { return levelMap_->collection(); }
  void setBookmark(Bookmark *bm);
  void goToBookmark(Bookmark *bm);

  int level() const;
  const TQString &collectionName();
  int totalMoves() const;
  int totalPushes() const;

  void updateCollectionXpm();
  void updateTextXpm();
  void updateLevelXpm();
  void updateStepsXpm();
  void updatePushesXpm();

public slots:
  void nextLevel();
  void previousLevel();
  void undo();
  void redo();
  void restartLevel();
  void changeCollection(LevelCollection *collection);
  void changeAnim(int num);

protected:
  ImageData *imageData_;
  LevelMap  *levelMap_;
  History   *history_;
  int        lastLevel_;
  MoveSequence  *moveSequence_;
  MapDelta  *mapDelta_;
  bool       moveInProgress_;
  bool       dragInProgress_;
  PathFinder pathFinder_;
  int        animDelay_;
  const TQCursor* cursor_;

  void levelChange ();
  void paintSquare (int x, int y, TQPainter &paint);
  void paintDelta ();
  void paintEvent (TQPaintEvent *e);
  void paintPainterClip(TQPainter& paint, int x, int y, int w, int h);
  void paintPainter(TQPainter& paint, const TQRect& rect);
  void resizeEvent (TQResizeEvent *e);
  void mouseMoveEvent(TQMouseEvent* e);
  void keyPressEvent (TQKeyEvent *);
  void focusInEvent (TQFocusEvent *);
  void focusOutEvent (TQFocusEvent *);
  void mousePressEvent (TQMouseEvent *);
  void mouseReleaseEvent(TQMouseEvent*);
  void leaveEvent(TQEvent*);
  void wheelEvent (TQWheelEvent *);
  void step (int _x, int _y);
  void push (int _x, int _y);
  virtual void timerEvent (TQTimerEvent *);
  void stopDrag();
  void dragObject(int xpixel, int ypixel);
  void highlight();
  void changeCursor(const TQCursor* c);
  void eatKeyPressEvents();

private:
  int size_, xOffs_, yOffs_;
  int highlightX_, highlightY_;
  int dragX_, dragY_;
  int lastMouseXPos_, lastMouseYPos_;
  int mousePosX_, mousePosY_;
  int wheelDelta_;

  int x2pixel (int x) const { return size_*x+xOffs_; }
  int y2pixel (int y) const { return size_*y+yOffs_; }

  int pixel2x (int x) const { return (x-xOffs_)/size_; }
  int pixel2y (int y) const { return (y-yOffs_)/size_; }

  void startMoving (Move *m);
  void startMoving (MoveSequence *ms);
  void stopMoving ();

  TQRect pnumRect_, ptxtRect_, snumRect_, stxtRect_, lnumRect_, ltxtRect_;
  TQRect collRect_;

  const TQString levelText_, stepsText_, pushesText_;
  TQPixmap pnumXpm_, ptxtXpm_, snumXpm_, stxtXpm_, lnumXpm_, ltxtXpm_;
  TQPixmap collXpm_;
  TQPixmap dragXpm_;
  TQImage  dragImage_;
  TQFont         statusFont_;
  TQFontMetrics  statusMetrics_;
  TQBrush  background_;
  TQBrush  floor_;

};

#endif  /* PLAYFIELD_H */
