// Copyright (c) 2002-2003 Rob Kaper <cap@capsi.com>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License version 2.1 as published by the Free Software Foundation.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public License
// along with this library; see the file COPYING.LIB.  If not, write to
// the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.

#ifndef LIBATLANTIK_NETWORK_H
#define LIBATLANTIK_NETWORK_H

#include <tqmap.h>

#include <kextsock.h>
#include "libatlantic_export.h"
class TQDomNode;
class TQTextStream;

class AtlanticCore;

class Player;
class Estate;
class EstateGroup;
class Trade;
class Auction;

class LIBATLANTIC_EXPORT AtlantikNetwork : public KExtendedSocket
{
Q_OBJECT
  TQ_OBJECT

public:
	AtlantikNetwork(AtlanticCore *atlanticCore);
	virtual ~AtlantikNetwork(void);
	void setName(TQString name);
	void cmdChat(TQString msg);

private slots:
	void writeData(TQString msg);
	void rollDice();
	void endTurn();
	void newGame(const TQString &gameType);
	void reconnect(const TQString &cookie);
	void startGame();
	void buyEstate();
	void auctionEstate();
	void estateToggleMortgage(Estate *estate);
	void estateHouseBuy(Estate *estate);
	void estateHouseSell(Estate *estate);
	void jailCard();
	void jailPay();
	void jailRoll();
	void newTrade(Player *player);
	void kickPlayer(Player *player);
	void tokenConfirmation(Estate *);
	void tradeUpdateEstate(Trade *trade, Estate *estate, Player *player);
	void tradeUpdateMoney(Trade *trade, unsigned int money, Player *pFrom, Player *pTo);
	void tradeReject(Trade *trade);
	void tradeAccept(Trade *trade);
	void auctionBid(Auction *auction, int amount);
	void changeOption(int, const TQString &value);
	void slotLookupFinished(int count);
	void slotConnectionSuccess();
	void slotConnectionFailed(int error);

public slots:
	void serverConnect(const TQString host, int port);
	void joinGame(int gameId);
	void leaveGame();
	void slotRead();
	void setImage(const TQString &name);

signals:
	/**
	 * A new estate was created. This signal might be replaced with one in
	 * the AtlanticCore class in the future, but it is here now because we
	 * do not want GUI implementations to create a view until the
	 * estateupdate message has been fully parsed.
	 *
	 * @param estate	Created Estate object.
	 */
	void newEstate(Estate *estate);

	/**
	 * A new estate group was created. This signal might be replaced with
	 * one in the AtlanticCore class in the future, but it is here now
	 * because we do not want GUI implementations to create a view until the
	 * estategroupupdate message has been fully parsed.
	 *
	 * @param estateGroup	Created EstateGroup object.
	 */
	void newEstateGroup(EstateGroup *estateGroup);

	void msgInfo(TQString);
	void msgError(TQString);
	void msgChat(TQString, TQString);
	void msgStatus(const TQString &data, const TQString &icon = TQString());
	void networkEvent(const TQString &data, const TQString &icon);

	void displayDetails(TQString text, bool clearText, bool clearButtons, Estate *estate = 0);
	void addCommandButton(TQString command, TQString caption, bool enabled);
	void addCloseButton();

	void gameOption(TQString title, TQString type, TQString value, TQString edit, TQString command);
	void endConfigUpdate();

	void gameConfig();
	void gameInit();
	void gameRun();
	void gameEnd();

	/**
	 * The trade has been completed. Emitted after all necessary estate and
	 * player updates are processed.
	 *
	 * @param trade Trade
	 */
	void msgTradeUpdateAccepted(Trade *trade);

	/**
	 * One of the players rejected the trade and the trade object has been
	 * deleted from the server.
	 *
	 * @param trade Trade
	 * @param playerId Unique player identifier of rejecting player
	 */
	void msgTradeUpdateRejected(Trade *trade, int playerId);

	void newAuction(Auction *auction);
	void auctionCompleted(Auction *auction);
	void receivedHandshake();
	void clientCookie(TQString cookie);

private:
	void processMsg(const TQString &msg);
	void processNode(TQDomNode);

	AtlanticCore *m_atlanticCore;
	TQTextStream *m_textStream;

	int m_playerId;
	TQString m_serverVersion;

	TQMap<Player *, int> m_playerLocationMap;
	TQMap<int, Auction *> m_auctions;
};

#endif
