// Copyright (c) 2002-2004 Rob Kaper <cap@capsi.com>
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// version 2 as published by the Free Software Foundation.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; see the file COPYING.  If not, write to
// the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
// Boston, MA 02110-1301, USA.

#include <iostream>

#include <tqcheckbox.h>
#include <tqradiobutton.h>

#include <kdebug.h>
#include <kdialog.h>
#include <klocale.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>

#include <atlantic_core.h>
#include <configoption.h>
#include <game.h>
#include <player.h>

#include "selectconfiguration_widget.moc"

SelectConfiguration::SelectConfiguration(AtlanticCore *atlanticCore, TQWidget *parent, const char *name) : TQWidget(parent, name)
{
	m_atlanticCore = atlanticCore;
	m_game = 0;

	m_mainLayout = new TQVBoxLayout(this, KDialog::marginHint());
	Q_CHECK_PTR(m_mainLayout);

	// Game configuration.
	m_configBox = new TQVGroupBox(i18n("Game Configuration"), this, "configBox");
	m_mainLayout->addWidget(m_configBox);

	// Player buttons.
	TQHBoxLayout *playerButtons = new TQHBoxLayout(m_mainLayout, KDialog::spacingHint());
	playerButtons->setMargin(0);

	playerButtons->addItem(new TQSpacerItem(20, 20, TQSizePolicy::Expanding, TQSizePolicy::Minimum));

	//Qt::Vertical spacer.
	m_mainLayout->addItem(new TQSpacerItem(20, 20, TQSizePolicy::Minimum, TQSizePolicy::Expanding));

	// Server buttons.
	TQHBoxLayout *serverButtons = new TQHBoxLayout(m_mainLayout, KDialog::spacingHint());
	serverButtons->setMargin(0);

	m_backButton = new KPushButton(SmallIcon("back"), i18n("Leave Game"), this);
	serverButtons->addWidget(m_backButton);

	connect(m_backButton, TQT_SIGNAL(clicked()), this, TQT_SIGNAL(leaveGame()));

	serverButtons->addItem(new TQSpacerItem(20, 20, TQSizePolicy::Expanding, TQSizePolicy::Minimum));

	m_startButton = new KPushButton(SmallIconSet("forward"), i18n("Start Game"), this);
	serverButtons->addWidget(m_startButton);
	m_startButton->setEnabled(false);

	connect(m_startButton, TQT_SIGNAL(clicked()), this, TQT_SIGNAL(startGame()));

	Player *playerSelf = m_atlanticCore->playerSelf();
	playerChanged(playerSelf);
	connect(playerSelf, TQT_SIGNAL(changed(Player *)), this, TQT_SLOT(playerChanged(Player *)));
	
	emit statusMessage(i18n("Retrieving configuration list..."));
}

void SelectConfiguration::initGame()
{
	emit statusMessage(i18n("Game started. Retrieving full game data..."));
}

void SelectConfiguration::addConfigOption(ConfigOption *configOption)
{
	// FIXME: only bool types supported!
	TQCheckBox *checkBox = new TQCheckBox(configOption->description(), m_configBox, "checkbox");
	m_configMap[(TQObject *)checkBox] = configOption;
	m_configBoxMap[configOption] = checkBox;
	
	checkBox->setChecked( configOption->value().toInt() );
	checkBox->setEnabled( configOption->edit() && m_atlanticCore->selfIsMaster() );
	checkBox->show();

	connect(checkBox, TQT_SIGNAL(clicked()), this, TQT_SLOT(changeOption()));
	connect(configOption, TQT_SIGNAL(changed(ConfigOption *)), this, TQT_SLOT(optionChanged(ConfigOption *)));
}

void SelectConfiguration::gameOption(TQString title, TQString type, TQString value, TQString edit, TQString command)
{
	// Find if option exists in GUI yet
	if (TQCheckBox *checkBox = dynamic_cast<TQCheckBox *>(m_checkBoxMap[command]))
	{
		checkBox->setChecked(value.toInt());
		checkBox->setEnabled(edit.toInt());
		return;
	}

	// Create option
	if (type == "bool")
	{
		TQCheckBox *checkBox = new TQCheckBox(title, m_configBox, "checkbox");
		m_optionCommandMap[(TQObject *)checkBox] = command;
		m_checkBoxMap[command] = checkBox;
		checkBox->setChecked(value.toInt());
		checkBox->setEnabled(edit.toInt());
		checkBox->show();

		connect(checkBox, TQT_SIGNAL(clicked()), this, TQT_SLOT(optionChanged()));
	}
	// TODO: create options other than type=bool

	// TODO: Enable edit for master only
}

void SelectConfiguration::changeOption()
{
	ConfigOption *configOption = m_configMap[(TQObject *)TQT_BASE_OBJECT_NAME::sender()];
	if (configOption)
	{
		kdDebug() << "checked " << ((TQCheckBox *)TQT_BASE_OBJECT_NAME::sender())->isChecked() << endl;
		emit changeOption( configOption->id(), TQString::number( ((TQCheckBox *)TQT_BASE_OBJECT_NAME::sender())->isChecked() ) );
	}
}

void SelectConfiguration::optionChanged(ConfigOption *configOption)
{
	TQCheckBox *checkBox = m_configBoxMap[configOption];
	if (checkBox)
	{
		checkBox->setText( configOption->description() );
		checkBox->setChecked( configOption->value().toInt() );
		checkBox->setEnabled( configOption->edit() && m_atlanticCore->selfIsMaster() );
	}
}

void SelectConfiguration::optionChanged()
{
	TQString command = m_optionCommandMap[(TQObject *)TQT_BASE_OBJECT_NAME::sender()];

	if (TQCheckBox *checkBox = m_checkBoxMap[command])
	{
		command.append(TQString::number(checkBox->isChecked()));
		emit buttonCommand(command);
	}
}

void SelectConfiguration::slotEndUpdate()
{
	emit statusMessage(i18n("Retrieved configuration list."));
}

void SelectConfiguration::playerChanged(Player *player)
{
	kdDebug() << "playerChanged" << endl;

	if (player->game() != m_game)
	{
		kdDebug() << "playerChanged::change" << endl;

		if (m_game)
			disconnect(m_game, TQT_SIGNAL(changed(Game *)), this, TQT_SLOT(gameChanged(Game *)));

		m_game = player->game();

		if (m_game)
			connect(m_game, TQT_SIGNAL(changed(Game *)), this, TQT_SLOT(gameChanged(Game *)));
	}
}

void SelectConfiguration::gameChanged(Game *game)
{
	m_startButton->setEnabled( game->master() == m_atlanticCore->playerSelf() );

	for (TQMapIterator<ConfigOption *, TQCheckBox *> it = m_configBoxMap.begin() ; it != m_configBoxMap.end() ; ++it)
		(*it)->setEnabled( it.key()->edit() && m_atlanticCore->selfIsMaster() );
}
