// TQString.cs - A Qt to C# binding.
//
// Copyright (C) 2002  Adam Treat (manyoso@yahoo.com)
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
//
// Static file customized to allow unicode strings and
// other features


namespace Qt {

	using Qt;
	using System;
	using System.Runtime.InteropServices;

	public class TQString : QtSupport, IDisposable {

		public enum SectionFlags {
			SectionDefault = 0,
			SectionSkipEmpty = 1,
			SectionIncludeLeadingSep = 2,
			SectionIncludeTrailingSep = 4,
			SectionCaseInsensitiveSeps = 8
		}

		
		[DllImport("libqtc")]
		private static extern IntPtr qt_QString_null ();
		public static readonly TQString Null = new TQString (qt_QString_null ());
		
		// TODO: Is there a good reason why this C# constructor is public?
		[DllImport("libqtc")]
		private static extern IntPtr qt_new_QString ();
		public TQString () : this (QNull.Instance)
		{
			rawObject = qt_new_QString ();
		}

		// TODO: Make sure there is a TQChar constructor that
		// handles unicode conversion
		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_new_QString1 (IntPtr arg1);
		public TQString (TQChar arg1) : this (QNull.Instance)
		{
			rawObject = qt_new_QString1 (arg1.RawObject);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_new_QString2 (IntPtr arg1);
		public TQString (TQString arg1) : this (QNull.Instance)
		{
			rawObject = qt_new_QString2 (arg1.RawObject);
		}

		// TODO: Verify that this works. We have weak support for
		// QByteArray
		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_new_QString3 (IntPtr arg1);
		public TQString (TQByteArray arg1) : this (QNull.Instance)
		{
			rawObject = qt_new_QString3 (arg1.RawObject);
		}

		// TODO: This signature is wrong. The C++ version is
		// takes a pointer to a sequence of QChars, not a single
		// QChar
		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_new_QString4 (IntPtr unicode, uint length);
		public TQString (TQChar unicode, uint length) : this (QNull.Instance)
		{
			rawObject = qt_new_QString4 (unicode.RawObject, length);
			
		}
	
		[DllImport("libqtc")]
		private static extern IntPtr qt_QString_setUnicodeCodes(
			IntPtr raw,
			[MarshalAs(UnmanagedType.LPWStr)] string s,
			uint len);
		
		public TQString( string s ): this( QNull.Instance )
		{
			rawObject = qt_new_QString ();
			rawObject = qt_QString_setUnicodeCodes(
				rawObject, s, (uint) s.Length );
		}

		internal TQString (IntPtr ptr) : this (QNull.Instance)
		{
			rawObject = ptr;

		}

		internal TQString (QNull dummy) : base (QNull.Instance) {}
		
		
		[DllImport("libqtc")]
		private static extern void qt_del_QString (IntPtr str);
		~TQString ()
		{
			qt_del_QString (rawObject);
		}

		
		[DllImport("libqtc")]
		private static extern ushort qt_QChar_unicode (IntPtr raw);			
		
		public char this[int index]
		{
			get
			{
				return (char) qt_QChar_unicode(
					qt_QString_at( rawObject, (uint) index ) );
			}
		}
		
		public static implicit operator string ( TQString s )
		{
			int length = (int) s.Length();
			char[] t = new char[length];
		
			for (int i = 0; i < length; i++ )
				t[i] = s[i];
		
			return new String(t);
		}
		
		public static implicit operator TQString ( string s )
		{
			return new TQString(s);
		}		
		
		public override string ToString ()
		{
			return (string) this;	
		}
		
		public static TQString operator + ( TQString a, TQString b )
		{
			TQString result = new TQString (qt_new_QString2( a.rawObject ));
			result.rawObject = 
				qt_QString_append2( result.rawObject, b.rawObject );
			return result;		
		}		
		
		// TODO: Do we need ==, !=, Equals(), etc?

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern bool qt_QString_isNull (IntPtr raw);
		public bool IsNull ()
		{
			return qt_QString_isNull (rawObject);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern bool qt_QString_isEmpty (IntPtr raw);
		public bool IsEmpty ()
		{
			return qt_QString_isEmpty (rawObject);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern uint qt_QString_length (IntPtr raw);
		public uint Length ()
		{
			return qt_QString_length (rawObject);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern void qt_QString_truncate (IntPtr raw, uint pos);
		public void Truncate (uint pos)
		{
			qt_QString_truncate (rawObject, pos);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_fill (IntPtr raw, IntPtr c, int len);
		public TQString Fill (TQChar c, int len)
		{
			return new TQString (qt_QString_fill (rawObject, c.RawObject, len));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_arg (IntPtr raw, long a, int fieldwidth, int Q_base);
		public TQString Arg (long a, int fieldwidth, int Q_base)
		{
			return new TQString (qt_QString_arg (rawObject, a, fieldwidth, Q_base));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_arg1 (IntPtr raw, ulong a, int fieldwidth, int Q_base);
		public TQString Arg (ulong a, int fieldwidth, int Q_base)
		{
			return new TQString (qt_QString_arg1 (rawObject, a, fieldwidth, Q_base));

		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_arg2 (IntPtr raw, int a, int fieldwidth, int Q_base);
		public TQString Arg (int a, int fieldwidth, int Q_base)
		{
			return new TQString (qt_QString_arg2 (rawObject, a, fieldwidth, Q_base));

		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_arg3 (IntPtr raw, uint a, int fieldwidth, int Q_base);
		public TQString Arg (uint a, int fieldwidth, int Q_base)
		{
			return new TQString (qt_QString_arg3 (rawObject, a, fieldwidth, Q_base));

		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_arg4 (IntPtr raw, short a, int fieldwidth, int Q_base);
		public TQString Arg (short a, int fieldwidth, int Q_base)
		{
			return new TQString (qt_QString_arg4 (rawObject, a, fieldwidth, Q_base));

		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_arg5 (IntPtr raw, ushort a, int fieldwidth, int Q_base);
		public TQString Arg (ushort a, int fieldwidth, int Q_base)
		{
			return new TQString (qt_QString_arg5 (rawObject, a, fieldwidth, Q_base));

		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_arg6 (IntPtr raw, char a, int fieldwidth);
		public TQString Arg (char a, int fieldwidth)
		{
			return new TQString (qt_QString_arg6 (rawObject, a, fieldwidth));

		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_arg7 (IntPtr raw, IntPtr a, int fieldwidth);
		public TQString Arg (TQChar a, int fieldwidth)
		{
			return new TQString (qt_QString_arg7 (rawObject, a.RawObject, fieldwidth));

		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_arg8 (IntPtr raw, IntPtr a, int fieldwidth);
		public TQString Arg (TQString a, int fieldwidth)
		{
			return new TQString (qt_QString_arg8 (rawObject, a.RawObject, fieldwidth));

		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_arg9 (IntPtr raw, double a, int fieldwidth, char fmt, int prec);
		public TQString Arg (double a, int fieldwidth, char fmt, int prec)
		{
			return new TQString (qt_QString_arg9 (rawObject, a, fieldwidth, fmt, prec));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern int qt_QString_find (IntPtr raw, IntPtr c, int index, bool cs);
		public int Find (TQChar c, int index, bool cs)
		{
			return qt_QString_find (rawObject, c.RawObject, index, cs);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern int qt_QString_find1 (IntPtr raw, char c, int index, bool cs);
		public int Find (char c, int index, bool cs)
		{
			return qt_QString_find1 (rawObject, c, index, cs);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern int qt_QString_find2 (IntPtr raw, IntPtr str, int index, bool cs);
		public int Find (TQString str, int index, bool cs)
		{
			return qt_QString_find2 (rawObject, str.RawObject, index, cs);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern int qt_QString_find3 (IntPtr raw, IntPtr arg1, int index);
		public int Find (TQRegExp arg1, int index)
		{
			return qt_QString_find3 (rawObject, arg1.RawObject, index);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern int qt_QString_find4 (IntPtr raw, string str, int index);
		public int Find (string str, int index)
		{
			return qt_QString_find4 (rawObject, str, index);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern int qt_QString_findRev (IntPtr raw, IntPtr c, int index, bool cs);
		public int FindRev (TQChar c, int index, bool cs)
		{
			return qt_QString_findRev (rawObject, c.RawObject, index, cs);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern int qt_QString_findRev1 (IntPtr raw, char c, int index, bool cs);
		public int FindRev (char c, int index, bool cs)
		{
			return qt_QString_findRev1 (rawObject, c, index, cs);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern int qt_QString_findRev2 (IntPtr raw, IntPtr str, int index, bool cs);
		public int FindRev (TQString str, int index, bool cs)
		{
			return qt_QString_findRev2 (rawObject, str.RawObject, index, cs);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern int qt_QString_findRev3 (IntPtr raw, IntPtr arg1, int index);
		public int FindRev (TQRegExp arg1, int index)
		{
			return qt_QString_findRev3 (rawObject, arg1.RawObject, index);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern int qt_QString_findRev4 (IntPtr raw, string str, int index);
		public int FindRev (string str, int index)
		{
			return qt_QString_findRev4 (rawObject, str, index);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern int qt_QString_contains (IntPtr raw, IntPtr c, bool cs);
		public int Contains (TQChar c, bool cs)
		{
			return qt_QString_contains (rawObject, c.RawObject, cs);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern int qt_QString_contains1 (IntPtr raw, char c, bool cs);
		public int Contains (char c, bool cs)
		{
			return qt_QString_contains1 (rawObject, c, cs);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern int qt_QString_contains2 (IntPtr raw, string str, bool cs);
		public int Contains (string str, bool cs)
		{
			return qt_QString_contains2 (rawObject, str, cs);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern int qt_QString_contains3 (IntPtr raw, IntPtr str, bool cs);
		public int Contains (TQString str, bool cs)
		{
			return qt_QString_contains3 (rawObject, str.RawObject, cs);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern int qt_QString_contains4 (IntPtr raw, IntPtr arg1);
		public int Contains (TQRegExp arg1)
		{
			return qt_QString_contains4 (rawObject, arg1.RawObject);
		}

		// TODO: This is wrong. sep should be a TQChar* in C++ terms
		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_section (IntPtr raw, IntPtr sep, int start, int end, TQString.SectionFlags SectionFlags);
		public TQString Section (TQChar sep, int start, int end, TQString.SectionFlags SectionFlags)
		{
			return new TQString (qt_QString_section (rawObject, sep.RawObject, start, end, SectionFlags));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_section1 (IntPtr raw, char sep, int start, int end, TQString.SectionFlags SectionFlags);
		public TQString Section (char sep, int start, int end, TQString.SectionFlags SectionFlags)
		{
			return new TQString (qt_QString_section1 (rawObject, sep, start, end, SectionFlags));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_section2 (IntPtr raw, string in_sep, int start, int end, TQString.SectionFlags SectionFlags);
		public TQString Section (string in_sep, int start, int end, TQString.SectionFlags SectionFlags)
		{
			return new TQString (qt_QString_section2 (rawObject, in_sep, start, end, SectionFlags));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_section3 (IntPtr raw, IntPtr in_sep, int start, int end, TQString.SectionFlags SectionFlags);
		public TQString Section (TQString in_sep, int start, int end, TQString.SectionFlags SectionFlags)
		{
			return new TQString (qt_QString_section3 (rawObject, in_sep.RawObject, start, end, SectionFlags));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_section4 (IntPtr raw, IntPtr reg, int start, int end, TQString.SectionFlags SectionFlags);
		public TQString Section (TQRegExp reg, int start, int end, TQString.SectionFlags SectionFlags)
		{
			return new TQString (qt_QString_section4 (rawObject, reg.RawObject, start, end, SectionFlags));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_left (IntPtr raw, uint len);
		public TQString Left (uint len)
		{
			return new TQString (qt_QString_left (rawObject, len));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_right (IntPtr raw, uint len);
		public TQString Right (uint len)
		{
			return new TQString (qt_QString_right (rawObject, len));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_mid (IntPtr raw, uint index, uint len);
		public TQString Mid (uint index, uint len)
		{
			return new TQString (qt_QString_mid (rawObject, index, len));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_leftJustify (IntPtr raw, uint width, IntPtr fill, bool trunc);
		public TQString LeftJustify (uint width, TQChar fill, bool trunc)
		{
			return new TQString (qt_QString_leftJustify (rawObject, width, fill.RawObject, trunc));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_rightJustify (IntPtr raw, uint width, IntPtr fill, bool trunc);
		public TQString RightJustify (uint width, TQChar fill, bool trunc)
		{
			return new TQString (qt_QString_rightJustify (rawObject, width, fill.RawObject, trunc));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_lower (IntPtr raw);
		public TQString Lower ()
		{
			return new TQString (qt_QString_lower (rawObject));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_upper (IntPtr raw);
		public TQString Upper ()
		{
			return new TQString (qt_QString_upper (rawObject));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_stripWhiteSpace (IntPtr raw);
		public TQString StripWhiteSpace ()
		{
			return new TQString (qt_QString_stripWhiteSpace (rawObject));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_simplifyWhiteSpace (IntPtr raw);
		public TQString SimplifyWhiteSpace ()
		{
			return new TQString (qt_QString_simplifyWhiteSpace (rawObject));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_insert (IntPtr raw, uint index, IntPtr arg1);
		public TQString Insert (uint index, TQString arg1)
		{
			return new TQString (qt_QString_insert (rawObject, index, arg1.RawObject));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_insert1 (IntPtr raw, uint index, IntPtr arg1, uint len);
		public TQString Insert (uint index, TQChar arg1, uint len)
		{
			return new TQString (qt_QString_insert1 (rawObject, index, arg1.RawObject, len));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_insert2 (IntPtr raw, uint index, IntPtr arg1);
		public TQString Insert (uint index, TQChar arg1)
		{
			return new TQString (qt_QString_insert2 (rawObject, index, arg1.RawObject));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_insert3 (IntPtr raw, uint index, char c);
		public TQString Insert (uint index, char c)
		{
			return new TQString (qt_QString_insert3 (rawObject, index, c));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_append (IntPtr raw, char arg1);
		public TQString Append (char arg1)
		{
			return new TQString (qt_QString_append (rawObject, arg1));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_append1 (IntPtr raw, IntPtr arg1);
		public TQString Append (TQChar arg1)
		{
			return new TQString (qt_QString_append1 (rawObject, arg1.RawObject));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_append2 (IntPtr raw, IntPtr arg1);
		public TQString Append (TQString arg1)
		{
			return new TQString (qt_QString_append2 (rawObject, arg1.RawObject));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_prepend (IntPtr raw, char arg1);
		public TQString Prepend (char arg1)
		{
			return new TQString (qt_QString_prepend (rawObject, arg1));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_prepend1 (IntPtr raw, IntPtr arg1);
		public TQString Prepend (TQChar arg1)
		{
			return new TQString (qt_QString_prepend1 (rawObject, arg1.RawObject));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_prepend2 (IntPtr raw, IntPtr arg1);
		public TQString Prepend (TQString arg1)
		{
			return new TQString (qt_QString_prepend2 (rawObject, arg1.RawObject));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_remove (IntPtr raw, uint index, uint len);
		public TQString Remove (uint index, uint len)
		{
			return new TQString (qt_QString_remove (rawObject, index, len));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_replace (IntPtr raw, uint index, uint len, IntPtr arg1);
		public TQString Replace (uint index, uint len, TQString arg1)
		{
			return new TQString (qt_QString_replace (rawObject, index, len, arg1.RawObject));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_replace1 (IntPtr raw, uint index, uint len, IntPtr arg1, uint clen);
		public TQString Replace (uint index, uint len, TQChar arg1, uint clen)
		{
			return new TQString (qt_QString_replace1 (rawObject, index, len, arg1.RawObject, clen));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_replace2 (IntPtr raw, IntPtr arg1, IntPtr arg2);
		public TQString Replace (TQRegExp arg1, TQString arg2)
		{
			return new TQString (qt_QString_replace2 (rawObject, arg1.RawObject, arg2.RawObject));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern short qt_QString_toShort (IntPtr raw, bool ok, int Q_base);
		public short ToShort (bool ok, int Q_base)
		{
			return qt_QString_toShort (rawObject, ok, Q_base);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern ushort qt_QString_toUShort (IntPtr raw, bool ok, int Q_base);
		public ushort ToUShort (bool ok, int Q_base)
		{
			return qt_QString_toUShort (rawObject, ok, Q_base);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern int qt_QString_toInt (IntPtr raw, bool ok, int Q_base);
		public int ToInt (bool ok, int Q_base)
		{
			return qt_QString_toInt (rawObject, ok, Q_base);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern uint qt_QString_toUInt (IntPtr raw, bool ok, int Q_base);
		public uint ToUInt (bool ok, int Q_base)
		{
			return qt_QString_toUInt (rawObject, ok, Q_base);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern long qt_QString_toLong (IntPtr raw, bool ok, int Q_base);
		public long ToLong (bool ok, int Q_base)
		{
			return qt_QString_toLong (rawObject, ok, Q_base);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern ulong qt_QString_toULong (IntPtr raw, bool ok, int Q_base);
		public ulong ToULong (bool ok, int Q_base)
		{
			return qt_QString_toULong (rawObject, ok, Q_base);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern float qt_QString_toFloat (IntPtr raw, bool ok);
		public float ToFloat (bool ok)
		{
			return qt_QString_toFloat (rawObject, ok);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern double qt_QString_toDouble (IntPtr raw, bool ok);
		public double ToDouble (bool ok)
		{
			return qt_QString_toDouble (rawObject, ok);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_setNum (IntPtr raw, short arg1, int Q_base);
		public TQString SetNum (short arg1, int Q_base)
		{
			return new TQString (qt_QString_setNum (rawObject, arg1, Q_base));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_setNum1 (IntPtr raw, ushort arg1, int Q_base);
		public TQString SetNum (ushort arg1, int Q_base)
		{
			return new TQString (qt_QString_setNum1 (rawObject, arg1, Q_base));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_setNum2 (IntPtr raw, int arg1, int Q_base);
		public TQString SetNum (int arg1, int Q_base)
		{
			return new TQString (qt_QString_setNum2 (rawObject, arg1, Q_base));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_setNum3 (IntPtr raw, uint arg1, int Q_base);
		public TQString SetNum (uint arg1, int Q_base)
		{
			return new TQString (qt_QString_setNum3 (rawObject, arg1, Q_base));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_setNum4 (IntPtr raw, long arg1, int Q_base);
		public TQString SetNum (long arg1, int Q_base)
		{
			return new TQString (qt_QString_setNum4 (rawObject, arg1, Q_base));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_setNum5 (IntPtr raw, ulong arg1, int Q_base);
		public TQString SetNum (ulong arg1, int Q_base)
		{
			return new TQString (qt_QString_setNum5 (rawObject, arg1, Q_base));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_setNum6 (IntPtr raw, float arg1, char f, int prec);
		public TQString SetNum (float arg1, char f, int prec)
		{
			return new TQString (qt_QString_setNum6 (rawObject, arg1, f, prec));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_setNum7 (IntPtr raw, double arg1, char f, int prec);
		public TQString SetNum (double arg1, char f, int prec)
		{
			return new TQString (qt_QString_setNum7 (rawObject, arg1, f, prec));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern void qt_QString_setExpand (IntPtr raw, uint index, IntPtr c);
		public void SetExpand (uint index, TQChar c)
		{
			qt_QString_setExpand (rawObject, index, c.RawObject);
		}

		// TODO: The TQChar functions need to get fixed
		
		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_at (IntPtr raw, uint i);
		public TQChar At (uint i)
		{
			return LookupObject (qt_QString_at (rawObject, i), typeof (TQChar)) as TQChar;
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_constref (IntPtr raw, uint i);
		public TQChar Constref (uint i)
		{
			return LookupObject (qt_QString_constref (rawObject, i), typeof (TQChar)) as TQChar;
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_ref (IntPtr raw, uint i);
		public TQChar Q_ref (uint i)
		{
			return LookupObject (qt_QString_ref (rawObject, i), typeof (TQChar)) as TQChar;
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_unicode (IntPtr raw);
		public TQChar Unicode ()
		{
			return LookupObject (qt_QString_unicode (rawObject), typeof (TQChar)) as TQChar;
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern string qt_QString_ascii (IntPtr raw);
		public string Ascii ()
		{
			return qt_QString_ascii (rawObject);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern string qt_QString_latin1 (IntPtr raw);
		public string Latin1 ()
		{
			return qt_QString_latin1 (rawObject);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_utf8 (IntPtr raw);
		public TQCString Utf8 ()
		{
			return LookupObject (qt_QString_utf8 (rawObject), typeof (TQCString)) as TQCString;
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_local8Bit (IntPtr raw);
		public TQCString Local8Bit ()
		{
			return LookupObject (qt_QString_local8Bit (rawObject), typeof (TQCString)) as TQCString;
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_setUnicode (IntPtr raw, IntPtr unicode, uint len);
		public TQString SetUnicode (TQChar unicode, uint len)
		{
			return new TQString (qt_QString_setUnicode (rawObject, unicode.RawObject, len));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_setLatin1 (IntPtr raw, string arg1, int len);
		public TQString SetLatin1 (string arg1, int len)
		{
			return new TQString (qt_QString_setLatin1 (rawObject, arg1, len));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern int qt_QString_compare (IntPtr raw, IntPtr s);
		public int Compare (TQString s)
		{
			return qt_QString_compare (rawObject, s.RawObject);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern int qt_QString_localeAwareCompare (IntPtr raw, IntPtr s);
		public int LocaleAwareCompare (TQString s)
		{
			return qt_QString_localeAwareCompare (rawObject, s.RawObject);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern void qt_QString_compose (IntPtr raw);
		public void Compose ()
		{
			qt_QString_compose (rawObject);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern string qt_QString_data (IntPtr raw);
		public string Data ()
		{
			return qt_QString_data (rawObject);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern bool qt_QString_startsWith (IntPtr raw, IntPtr arg1);
		public bool StartsWith (TQString arg1)
		{
			return qt_QString_startsWith (rawObject, arg1.RawObject);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern bool qt_QString_endsWith (IntPtr raw, IntPtr arg1);
		public bool EndsWith (TQString arg1)
		{
			return qt_QString_endsWith (rawObject, arg1.RawObject);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern void qt_QString_setLength (IntPtr raw, uint newLength);
		public void SetLength (uint newLength)
		{
			qt_QString_setLength (rawObject, newLength);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern bool qt_QString_simpleText (IntPtr raw);
		public bool SimpleText ()
		{
			return qt_QString_simpleText (rawObject);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern bool qt_QString_isRightToLeft (IntPtr raw);
		public bool IsRightToLeft ()
		{
			return qt_QString_isRightToLeft (rawObject);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_number (long arg1, int Q_base);
		public static TQString Number (long arg1, int Q_base)
		{
			return new TQString (qt_QString_number (arg1, Q_base));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_number1 (ulong arg1, int Q_base);
		public static TQString Number (ulong arg1, int Q_base)
		{
			return new TQString (qt_QString_number1 (arg1, Q_base));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_number2 (int arg1, int Q_base);
		public static TQString Number (int arg1, int Q_base)
		{
			return new TQString (qt_QString_number2 (arg1, Q_base));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_number3 (uint arg1, int Q_base);
		public static TQString Number (uint arg1, int Q_base)
		{
			return new TQString (qt_QString_number3 (arg1, Q_base));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_number4 (double arg1, char f, int prec);
		public static TQString Number (double arg1, char f, int prec)
		{
			return new TQString (qt_QString_number4 (arg1, f, prec));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_fromLatin1 (string arg1, int len);
		public static TQString FromLatin1 (string arg1, int len)
		{
			return new TQString (qt_QString_fromLatin1 (arg1, len));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_fromUtf8 (TQString arg1, int len);
		public static TQString FromUtf8 (string arg1, int len)
		{
			return new TQString (qt_QString_fromUtf8 (arg1, len));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern IntPtr qt_QString_fromLocal8Bit (TQString arg1, int len);
		public static TQString FromLocal8Bit (string arg1, int len)
		{
			return new TQString (qt_QString_fromLocal8Bit (arg1, len));
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern int qt_QString_compare1 (IntPtr s1, IntPtr s2);
		public static int Compare (TQString s1, TQString s2)
		{
			return qt_QString_compare1 (s1.RawObject, s2.RawObject);
		}

		[DllImport("libqtc", CharSet=CharSet.Ansi)]
		private static extern int qt_QString_localeAwareCompare1 (IntPtr s1, IntPtr s2);
		public static int LocaleAwareCompare (TQString s1, TQString s2)
		{
			return qt_QString_localeAwareCompare1 (s1.RawObject, s2.RawObject);
		}

		public TQString Fill (TQChar c)
		{
			return Fill(c, (int) -1);
		}

		public TQString Arg (long a)
		{
			return Arg(a, (int) 0);
		}

		public TQString Arg (long a, int fieldwidth)
		{
			return Arg(a, fieldwidth, (int) 10);
		}

		public TQString Arg (ulong a)
		{
			return Arg(a, (int) 0);
		}

		public TQString Arg (ulong a, int fieldwidth)
		{
			return Arg(a, fieldwidth, (int) 10);
		}

		public TQString Arg (int a)
		{
			return Arg(a, (int) 0);
		}

		public TQString Arg (int a, int fieldwidth)
		{
			return Arg(a, fieldwidth, (int) 10);
		}

		public TQString Arg (uint a)
		{
			return Arg(a, (int) 0);
		}

		public TQString Arg (uint a, int fieldwidth)
		{
			return Arg(a, fieldwidth, (int) 10);
		}

		public TQString Arg (short a)
		{
			return Arg(a, (int) 0);
		}

		public TQString Arg (short a, int fieldwidth)
		{
			return Arg(a, fieldwidth, (int) 10);
		}

		public TQString Arg (ushort a)
		{
			return Arg(a, (int) 0);
		}

		public TQString Arg (ushort a, int fieldwidth)
		{
			return Arg(a, fieldwidth, (int) 10);
		}

		public TQString Arg (char a)
		{
			return Arg(a, (int) 0);
		}

		public TQString Arg (TQChar a)
		{
			return Arg(a, (int) 0);
		}

		public TQString Arg (TQString a)
		{
			return Arg(a, (int) 0);
		}

		public TQString Arg (double a)
		{
			return Arg(a, (int) 0);
		}

		public TQString Arg (double a, int fieldwidth)
		{
			return Arg(a, fieldwidth, (char) 'g');
		}

		public TQString Arg (double a, int fieldwidth, char fmt)
		{
			return Arg(a, fieldwidth, fmt, (int) -1);
		}

		public int Find (TQChar c)
		{
			return Find(c, (int) 0);
		}

		public int Find (TQChar c, int index)
		{
			return Find(c, index, true);
		}

		public int Find (char c)
		{
			return Find(c, (int) 0);
		}

		public int Find (char c, int index)
		{
			return Find(c, index, true);
		}

		public int Find (TQString str)
		{
			return Find(str, (int) 0);
		}

		public int Find (TQString str, int index)
		{
			return Find(str, index, true);
		}

		public int Find (TQRegExp arg1)
		{
			return Find(arg1, (int) 0);
		}

		public int Find (string str)
		{
			return Find(str, (int) 0);
		}

		public int FindRev (TQChar c)
		{
			return FindRev(c, (int) -1);
		}

		public int FindRev (TQChar c, int index)
		{
			return FindRev(c, index, true);
		}

		public int FindRev (char c)
		{
			return FindRev(c, (int) -1);
		}

		public int FindRev (char c, int index)
		{
			return FindRev(c, index, true);
		}

		public int FindRev (TQString str)
		{
			return FindRev(str, (int) -1);
		}

		public int FindRev (TQString str, int index)
		{
			return FindRev(str, index, true);
		}

		public int FindRev (TQRegExp arg1)
		{
			return FindRev(arg1, (int) -1);
		}

		public int FindRev (string str)
		{
			return FindRev(str, (int) -1);
		}

		public int Contains (TQChar c)
		{
			return Contains(c, true);
		}

		public int Contains (char c)
		{
			return Contains(c, true);
		}

		public int Contains (TQString str)
		{
			return Contains(str, true);
		}

		public int Contains (string str)
		{
			return Contains(str, true);
		}

		public TQString Section (TQChar sep, int start)
		{
			return Section(sep, start, (int) 1);
		}

		public TQString Section (TQChar sep, int start, int end)
		{
			return Section(sep, start, end, TQString.SectionFlags.SectionDefault);
		}

		public TQString Section (char sep, int start)
		{
			return Section(sep, start, (int) 1);
		}

		public TQString Section (char sep, int start, int end)
		{
			return Section(sep, start, end, TQString.SectionFlags.SectionDefault);
		}

		public TQString Section (TQString in_sep, int start)
		{
			return Section(in_sep, start, (int) 1);
		}

		public TQString Section (TQString in_sep, int start, int end)
		{
			return Section(in_sep, start, end, TQString.SectionFlags.SectionDefault);
		}

		public TQString Section (string in_sep, int start)
		{
			return Section(in_sep, start, (int) 1);
		}

		public TQString Section (string in_sep, int start, int end)
		{
			return Section(in_sep, start, end, TQString.SectionFlags.SectionDefault);
		}

		public TQString Section (TQRegExp reg, int start)
		{
			return Section(reg, start, (int) 1);
		}

		public TQString Section (TQRegExp reg, int start, int end)
		{
			return Section(reg, start, end, TQString.SectionFlags.SectionDefault);
		}

		public TQString Mid (uint index)
		{
			return Mid(index, (uint) 1);
		}

		public TQString LeftJustify (uint width)
		{
			return LeftJustify(width, new TQChar ());
		}

		public TQString LeftJustify (uint width, TQChar fill)
		{
			return LeftJustify(width, fill, false);
		}

		public TQString RightJustify (uint width)
		{
			return RightJustify(width, new TQChar ());
		}

		public TQString RightJustify (uint width, TQChar fill)
		{
			return RightJustify(width, fill, false);
		}

		public short ToShort ()
		{
			return ToShort(false);
		}

		public short ToShort (bool ok)
		{
			return ToShort(ok, (int) 10);
		}

		public ushort ToUShort ()
		{
			return ToUShort(false);
		}

		public ushort ToUShort (bool ok)
		{
			return ToUShort(ok, (int) 10);
		}

		public int ToInt ()
		{
			return ToInt(false);
		}

		public int ToInt (bool ok)
		{
			return ToInt(ok, (int) 10);
		}

		public uint ToUInt ()
		{
			return ToUInt(false);
		}

		public uint ToUInt (bool ok)
		{
			return ToUInt(ok, (int) 10);
		}

		public long ToLong ()
		{
			return ToLong(false);
		}

		public long ToLong (bool ok)
		{
			return ToLong(ok, (int) 10);
		}

		public ulong ToULong ()
		{
			return ToULong(false);
		}

		public ulong ToULong (bool ok)
		{
			return ToULong(ok, (int) 10);
		}

		public float ToFloat ()
		{
			return ToFloat(false);
		}

		public double ToDouble ()
		{
			return ToDouble(false);
		}

		public TQString SetNum (short arg1)
		{
			return SetNum(arg1, (int) 10);
		}

		public TQString SetNum (ushort arg1)
		{
			return SetNum(arg1, (int) 10);
		}

		public TQString SetNum (int arg1)
		{
			return SetNum(arg1, (int) 10);
		}

		public TQString SetNum (uint arg1)
		{
			return SetNum(arg1, (int) 10);
		}

		public TQString SetNum (long arg1)
		{
			return SetNum(arg1, (int) 10);
		}

		public TQString SetNum (ulong arg1)
		{
			return SetNum(arg1, (int) 10);
		}

		public TQString SetNum (float arg1)
		{
			return SetNum(arg1, (char) 'g');
		}

		public TQString SetNum (float arg1, char f)
		{
			return SetNum(arg1, f, (int) 6);
		}

		public TQString SetNum (double arg1)
		{
			return SetNum(arg1, (char) 'g');
		}

		public TQString SetNum (double arg1, char f)
		{
			return SetNum(arg1, f, (int) 6);
		}

		public TQString SetLatin1 (TQString arg1)
		{
			return SetLatin1(arg1, (int) -1);
		}

		public static TQString Number (long arg1)
		{
			return Number(arg1, (int) 10);
		}

		public static TQString Number (ulong arg1)
		{
			return Number(arg1, (int) 10);
		}

		public static TQString Number (int arg1)
		{
			return Number(arg1, (int) 10);
		}

		public static TQString Number (uint arg1)
		{
			return Number(arg1, (int) 10);
		}

		public static TQString Number (double arg1)
		{
			return Number(arg1, (char) 'g');
		}

		public static TQString Number (double arg1, char f)
		{
			return Number(arg1, f, (int) 6);
		}

		public static TQString FromLatin1 (TQString arg1)
		{
			return FromLatin1(arg1, (int) -1);
		}

		public static TQString FromUtf8 (TQString arg1)
		{
			return FromUtf8(arg1, (int) -1);
		}

		public static TQString FromLocal8Bit (TQString arg1)
		{
			return FromLocal8Bit(arg1, (int) -1);
		}
	}
}
