/***************************************************************************
                          JavaSlot.cpp  -  description
                             -------------------
    begin                : Tue Oct 31 2000
    copyright            : (C) 2000-2004 by Richard Dale
    email                : Richard_Dale@tipitina.demon.co.uk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdio.h>

#include <tqmetaobject.h>

#include "JavaSlot.moc"
#include <qtjava/QtSupport.h>

/*
 Converts a Java type signature, from a signal or slot, to the corresponding C++ Qt type.
 Used to connect a Java slot to a signal emitted by the underlying C++ Qt instance.
 Note that all Java slots and signals are of type jobjectArray, and so no conversion is needed for them.
*/
static const char * const javaToQtTypeSignatureMap[][2] = {
	{"()",														"()"},
	{"(boolean)",												"(bool)"},
	{"(boolean[],boolean[])",									"(bool*, bool*)"},
	{"(char)",													"(char)"},
	{"(double)",												"(double)"},
	{"(float)",													"(float)"},
	{"(int)",													"(int)"},
	{"(int,boolean)",											"(int,bool)"},
	{"(int,String,byte[])",										"(int,const TQCString&,const TQByteArray&)"},
	{"(int,TQPoint)",											"(int,const TQPoint&)"},
	{"(int,String)",											"(int,const TQString&)"},
	{"(int,int)",												"(int,int)"},
	{"(int,int,TQPoint)",										"(int,int,const TQPoint&)"},
	{"(int,int,int)",											"(int,int,int)"},
	{"(int,int,int,TQPoint)",									"(int,int,int,const TQPoint&)"},
	{"(int,int,TQNetworkOperation)",								"(int,int,TQNetworkOperation*)"},
	{"(int,TQDropEvent)",										"(int,TQDropEvent*)"},
	{"(int,TQIconViewItem,TQPoint)",								"(int,TQIconViewItem*,const TQPoint&)"},
	{"(int,TQListBoxItem,TQPoint)",								"(int,TQListBoxItem*,const TQPoint)"},
	{"(int,TQListViewItem,TQPoint,int)",							"(int,TQListViewItem*,const TQPoint,int)"},
	{"(long)",													"(long)"},
	{"(long,byte[])",											"(long,const TQByteArray&)"},
	{"(long,String)",											"(long,const TQString&)"},
	{"(short)",													"(short)"},
	{"(byte)",													"(unsigned char)"},
	{"(TQAction)",												"(TQAction*)"},
	{"(int)",													"(TQDir::SortSpec)"},
	{"(TQDockWindow)",											"(TQDockWindow*)"},
	{"(int)",													"(TQDockWindow::Place)"},
	{"(TQDropEvent)",											"(TQDropEvent*)"},
	{"(TQDropEvent,TQListViewItem)",								"(TQDropEvent*,TQListViewItem*)"},
	{"(TQDropEvent,ArrayList",									"(TQDropEvent*,const TQValueList<TQIconDragItem>&)"},
	{"(TQDropEvent,TQListViewItem,TQListViewItem)",				"(TQDropEvent*,TQListViewItem*,TQListViewItem*)"},
	{"(TQIconViewItem)",											"(TQIconViewItem*)"},
	{"(TQIconViewItem,TQPoint)",									"(TQIconViewItem*,const TQPoint&)"},
	{"(TQIconViewItem,String)",									"(TQIconViewItem*,const TQString&)"},
	{"(ArrayList,ArrayList,ArrayList)",							"(QList<TQListViewItem>,QList<TQListViewItem>,QList<TQListViewItem>)"},
	{"(TQListBoxItem)",											"(TQListBoxItem*)"},
	{"(TQListBoxItem,TQPoint)",									"(TQListBoxItem*,const TQPoint&)"},
	{"(TQListViewItem,int,String)",								"(TQListViewItem*,int,const TQString&)"},
	{"(TQListViewItem,TQListViewItem,TQListViewItem)",				"(TQListViewItem*,TQListViewItem*,TQListViewItem*)"},
	{"(TQListViewItem,TQPoint,int)",								"(TQListViewItem*,const TQPoint&,int)"},
	{"(TQListViewItem,String,int)",								"(TQListViewItem*,const TQString&,int)"},
	{"(TQListViewItem,int)",										"(TQListViewItem*,int)"},
	{"(TQListViewItem)",											"(TQListViewItem*)"},
	{"(TQNetworkOperation)",										"(TQNetworkOperation*)"},
	{"(TQObject)",												"(TQObject*)"},
	{"(TQPopupMenu)",											"(TQPopupMenu*)"},
	{"(int)",													"(TQSql::Op)"},
	{"(int)",													"(TQSqlCursor::Mode)"},
	{"(TQSqlRecord)",											"(TQSqlRecord*)"},
	{"(int)",													"(TQTextEdit::VerticalAlignment)"},
	{"(TQToolBar)",												"(TQToolBar*)"},
	{"(int)",													"(Orientation)"},
	{"(TQWidget)",												"(TQWidget*)"},
	{"(TQWidget,TQPoint)",										"(TQWidget*,const TQPoint&)"},
	{"(TQWidget,String)",										"(TQWidget*,const TQString&)"},
	{"(TQWidget,TQDropEvent)",									"(TQWidget*,const TQDropEvent*)"},
	{"(TQWidget,TQPixmap)",										"(TQWidget*,TQPixmap)"},
	{"byte[]",													"(const TQByteArray&)"},
	{"(byte[],TQNetworkOperation)",								"(const TQByteArray&,TQNetworkOperation*)"},
	{"(String,boolean)",										"(const char*,bool)"},
	{"(String,String,byte[],String,String,String)",				"(const char*,const TQString&,const TQByteArray&,const TQString&,const TQString&,const TQString&)"},
	{"(char)",													"(const TQChar&)"},
	{"(String)",												"(const TQCString&)"},
	{"(TQColor)",												"(const TQColor&)"},
	{"(TQColor,String)",											"(const TQColor&,const TQString&)"},
	{"(Calendar)",												"(const TQDate&)"},
	{"(Calendar,Calendar)",										"(const TQDate&,const TQDate&)"},
	{"(Calendar)",												"(const TQDateTime&)"},
	{"(TQDragMoveEvent,boolean)",								"(const TQDragMoveEvent*,bool&)"},
	{"(TQSize)",													"(const TQSize&)"},
	{"(TQFont)",													"(const TQFont&)"},
	{"(TQHttpResponseHeader)",									"(const TQHttpResponseHeader&)"},
	{"(TQImage,int)",											"(const TQImage&, int)"},
	{"(TQPixmap)",												"(const TQPixmap&)"},
	{"(TQPoint)",												"(const TQPoint&)"},
	{"(TQRect)",													"(const TQRect&)"},
	{"(TQSqlRecord)",											"(const TQSqlRecord*)"},
	{"(String)",												"(const TQString&)"},
	{"(String,boolean,String)",									"(const TQString&,bool,const TQString&)"},
	{"(String,String,String)",									"(const TQString&,const TQCString&,const TQString&)"},
	{"(String,TQLibrary)",										"(const TQString&,TQLibrary*)"},
	{"(String,TQPoint)",											"(const TQString&,const TQPoint&)"},
	{"(String,String)",											"(const TQString&,const TQString&)"},
	{"(String,ArrayList,int)",									"(const TQString&,const TQStringList&,unsigned int)"},
	{"(String,String,int)",										"(const TQString&,const TQString&,unsigned int)"},
	{"(String,int)",											"(const TQString&,int)"},
	{"(String,int,int)",										"(const TQString&,int,int)"},
	{"(String,int,int,int)",									"(const TQString&,int,int,int)"},
	{"(ArrayList)",												"(const TQStringList&)"},
	{"(Date)",													"(const TQTime&)"},
	{"(String,long,long,long)",									"(const TQString&,unsigned long,unsigned long,unsigned long)"},
	{"(TQUrlInfo)",												"(const TQUrlInfo&)"},
	{"(TQUrlInfo,TQNetworkOperation)",							"(const TQUrlInfo&,TQNetworkOperation*)"},
	{"(ArrayList,TQNetworkOperation)",							"(const TQValueList<TQUrlInfo>&,TQNetworkOperation*)"},
	{"(TQVariant)",												"(const TQVariant&)"},
	{"(long,long,long,String)",									"(const unsigned long&,const unsigned long&,const unsigned long&,const TQString&)"}
};

JavaSlot::JavaSlot(JNIEnv * env, jobject receiver, jstring slot)
{
	jobject		javaInvocation;
	jmethodID	cid;
	jclass		invocationClass;

	invocationClass = env->FindClass("org/kde/qt/Invocation");
	if (invocationClass == NULL) {
		return;
	}

	cid = env->GetMethodID(invocationClass, "<init>", "(Ljava/lang/Object;Ljava/lang/String;)V");
	if (cid == NULL) {
		return;
	}

	javaInvocation = env->NewObject(invocationClass, cid, receiver, slot);
	invocation = env->NewGlobalRef(javaInvocation);
	env->DeleteLocalRef(invocationClass);
}

JavaSlot::~JavaSlot()
{
	JNIEnv *	env;
	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	env->DeleteGlobalRef(invocation);
}

JavaSlotFactory * JavaSlot::_factory = 0L;
	
void
JavaSlot::setJavaSlotFactory(JavaSlotFactory * factory)
{
	if (_factory != 0L) {
		delete _factory;
	}
	
	_factory = factory;
}

JavaSlot *
JavaSlot::createJavaSlot(JNIEnv * env, jobject receiver, jstring slot)
{
	return _factory->createJavaSlot(env, receiver, slot);
}

const char *
JavaSlot::javaToQtSignalName(JNIEnv * env, jstring signal, TQMetaObject * smeta)
{
	char	signalName[200];
	char 	javaTypeSignature[200];
	char *	signalString = (char*) env->GetStringUTFChars(signal, (jboolean *) 0);
	
	if (	signalString[0] == '2'
			&& sscanf(signalString, "%[^(]%s", signalName, javaTypeSignature) == 2 )
	{
		env->ReleaseStringUTFChars(signal, signalString);
		return javaToQtSignalType(signalName, javaTypeSignature, smeta);
	}
	
	env->ReleaseStringUTFChars(signal, signalString);
	return "";
}


const char *
JavaSlot::javaToQtSignalType(const char * signalName, const char * javaTypeSignature, TQMetaObject * smeta)
{
static char qtSignalString[200];

	for (	unsigned int index = 0;
			index < sizeof(javaToQtTypeSignatureMap)/sizeof(*javaToQtTypeSignatureMap);
			index++ )
	{
		if (strcmp(javaTypeSignature, javaToQtTypeSignatureMap[index][0]) == 0) {
			(void) sprintf(qtSignalString, "%s%s", signalName, javaToQtTypeSignatureMap[index][1]);

			if (smeta == 0 || smeta->findSignal(((const char *) qtSignalString + 1), TRUE) >= 0) {
				return qtSignalString;
			}
		}
	}

	return "";
}

const char *
JavaSlot::javaToQtSlotName(JNIEnv * env, jstring slot, const char * signalString)
{
static char qtSlotString[200];
	char 	javaTypeSignature[200];
	char *	slotString = (char*) env->GetStringUTFChars(slot, (jboolean *) 0);

	if (sscanf(slotString, "%*[^(]%s", javaTypeSignature) == 1) {
		env->ReleaseStringUTFChars(slot, slotString);
		(void) sprintf(qtSlotString, "1invoke%s", javaToQtSlotType(javaTypeSignature, signalString));
		return qtSlotString;
	}

	(void) sprintf(qtSlotString, "1invoke%s", slotString);
	env->ReleaseStringUTFChars(slot, slotString);
	return qtSlotString;
}

const char *
JavaSlot::javaToQtSlotType(const char * javaTypeSignature, const char * signalString)
{
	for (	unsigned int index = 0;
			index < sizeof(javaToQtTypeSignatureMap)/sizeof(*javaToQtTypeSignatureMap);
			index++ )
	{
		if (	strcmp(javaTypeSignature, javaToQtTypeSignatureMap[index][0]) == 0
				&& (	signalString == 0
						|| TQObject::checkConnectArgs(signalString, (const TQObject *) 0, javaToQtTypeSignatureMap[index][1]) ) )
		{
			return javaToQtTypeSignatureMap[index][1];
		}
	}
	
	// If no matching C++ type signature is found, then just return the java one
	return javaTypeSignature;
}

void
JavaSlot::invoke()
{
	invoke((jobjectArray) 0);
}

void
JavaSlot::invoke(bool arg)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Z)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, arg);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(bool* arg1, bool* arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "([Z[Z)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	jbooleanArray jarg1 = QtSupport::fromBooleanPtr(env, arg1);
	jbooleanArray jarg2 = QtSupport::fromBooleanPtr(env, arg2);
	result = env->CallObjectMethod(invocation, mid, jarg1, jarg2);
	*arg1 = *(QtSupport::toBooleanPtr(env, jarg1));
	*arg2 = *(QtSupport::toBooleanPtr(env, jarg1));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(char arg)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(C)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, arg);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const char* arg1, bool arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Z)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid, QtSupport::fromCharString(env, (char *)arg1), (jboolean) arg2);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const char* arg1,const TQString& arg2,const TQByteArray& arg3,const TQString& arg4,const TQString& arg5,const TQString& arg6)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;[BLjava/lang/Object;Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromCharString(env, (char*)arg1),
									QtSupport::fromQString(env, (TQString*)&arg2),
									QtSupport::fromQByteArray(env, (TQByteArray *) &arg3),
									QtSupport::fromQString(env, (TQString*)&arg4),
									QtSupport::fromQString(env, (TQString*)&arg5),
									QtSupport::fromQString(env, (TQString*)&arg6) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQByteArray& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "([B)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQByteArray(env, (TQByteArray *) &arg1) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQByteArray& arg1,TQNetworkOperation* arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "([BLjava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQByteArray(env, (TQByteArray *) &arg1),
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.TQNetworkOperation") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQChar& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(C)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::fromQChar(env, (TQChar *) &arg1));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQColor& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.TQColor"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQColor& arg1,const TQString& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.TQColor"),
									QtSupport::fromQString(env, (TQString*)&arg2) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQCString& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::fromQCString(env, (TQCString*)&arg1));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQDate& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::fromQDate(env, (TQDate *) &arg1));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQDate& arg1,const TQDate& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQDate(env, (TQDate *) &arg1),
									QtSupport::fromQDate(env, (TQDate *) &arg2) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQDateTime& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::fromQDateTime(env, (TQDateTime *) &arg1));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQDragMoveEvent* arg1,bool& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Z)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.TQDragMoveEvent"), (jboolean) arg2);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQFont& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.TQFont"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQHttpResponseHeader& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.TQHttpResponseHeader"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQImage& arg1, int arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.TQImage"), (jint) arg2);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQPixmap& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.TQPixmap"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQPoint& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.TQPoint"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQRect& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.TQRect"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQSize& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.TQSize"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQSqlRecord* arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQSqlRecord"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQString& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid, QtSupport::fromQString(env, (TQString *) &arg1));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQString& arg1,bool arg2,const TQString& arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;ZLjava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQString(env, (TQString*) &arg1),
									(jboolean) arg2,
									QtSupport::fromQString(env, (TQString*) &arg3) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQString& arg1,const TQCString& arg2,const TQString& arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQString(env, (TQString*) &arg1),
									QtSupport::fromQString(env, (TQString*) &arg2),
									QtSupport::fromQString(env, (TQString*) &arg3) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQString& arg1,const TQPoint& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQString(env, (TQString *) &arg1),
									QtSupport::objectForQtKey(env, (void *) &arg2, "org.kde.qt.TQPoint") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQString& arg1,const TQString& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQString(env, (TQString *) &arg1),
									QtSupport::fromQString(env, (TQString *) &arg2) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQString& arg1,const TQString& arg2,unsigned int arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQString(env, (TQString *) &arg1),
									QtSupport::fromQString(env, (TQString *) &arg2),
									(jint) arg3 );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQString& arg1,const TQStringList& arg2,unsigned int arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQString(env, (TQString*) &arg1),
									QtSupport::arrayWithQStringList(env, (TQStringList*)&arg2),
									(jint) arg3 );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQString& arg1,int arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQString(env, (TQString *) &arg1),
									(jint) arg2 );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQString& arg1,int arg2,int arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;II)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQString(env, (TQString *) &arg1),
									(jint) arg2,
									(jint) arg3 );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQString& arg1,int arg2,int arg3,int arg4)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;III)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQString(env, (TQString *) &arg1),
									(jint) arg2,
									(jint) arg3,
									(jint) arg4 );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQString& arg1,TQLibrary* arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQString(env, (TQString *) &arg1),
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.TQLibrary") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQString& arg1,unsigned long arg2,unsigned long arg3,unsigned long arg4)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;JJJ)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQString(env, (TQString *) &arg1),
									(jlong) arg2, (jlong) arg3,(jlong) arg4 );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQStringList& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::arrayWithQStringList(env, (TQStringList *) &arg1));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQTime& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.TQTime"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQUrlInfo& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.TQUrlInfo") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQUrlInfo& arg1,TQNetworkOperation* arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.TQUrlInfo"),
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.TQNetworkOperation") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQValueList<TQUrlInfo>& arg1,TQNetworkOperation* arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::arrayWithQUrlInfoList(env, (TQValueList<TQUrlInfo> *) &arg1),
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.TQNetworkOperation") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const TQVariant& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.TQVariant"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const unsigned long& arg1,const unsigned long& arg2,const unsigned long& arg3,const TQString& arg4)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(IIILjava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									(jlong) arg1,
									(jlong) arg2,
									(jlong) arg3,
									QtSupport::fromQString(env, (TQString*) &arg4) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(double arg)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(D)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, arg);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(float arg)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(F)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, arg);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, arg);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1, bool arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(IZ)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, (jint) arg1, (jboolean) arg2);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,const TQCString& arg2,const TQByteArray& arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(ILjava/lang/Object;[B)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									(jlong) arg1,
									QtSupport::fromQCString(env, (TQCString*)&arg2),
									QtSupport::fromQByteArray(env, (TQByteArray *) &arg3) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,const TQPoint& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(ILjava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									(jint) arg1,
									QtSupport::objectForQtKey(env, (void *) &arg2, "org.kde.qt.TQPoint") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,const TQString& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(ILjava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid, (jint) arg1, QtSupport::fromQString(env, (TQString *) &arg2));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,int arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(II)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, (jint) arg1, (jint) arg2);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,int arg2,const TQPoint& arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(IILjava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									(jint) arg1,
									(jint) arg2,
									QtSupport::objectForQtKey(env, (void *) &arg3, "org.kde.qt.TQPoint") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,int arg2,int arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(III)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, (jint) arg1, (jint) arg2, (jint) arg3);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,int arg2,int arg3,const TQPoint& arg4)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(IIILjava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									(jint) arg1,
									(jint) arg2,
									(jint) arg3,
									QtSupport::objectForQtKey(env, (void *) &arg4, "org.kde.qt.TQPoint"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,int arg2,TQNetworkOperation* arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(IILjava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation,
									mid,
									arg1,
									arg2,
									QtSupport::objectForQtKey(env, (void *) arg3, "org.kde.qt.TQNetworkOperation") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,TQDropEvent* arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(ILjava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation,
									mid,
									arg1,
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.TQDropEvent") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,TQIconViewItem* arg2,const TQPoint& arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(ILjava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									(jint) arg1,
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.TQIconViewItem"),
									QtSupport::objectForQtKey(env, (void *) &arg3, "org.kde.qt.TQPoint") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,TQListBoxItem* arg2,const TQPoint arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(ILjava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									(jint) arg1,
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.TQListBoxItem"),
									QtSupport::objectForQtKey(env, (void *) &arg3, "org.kde.qt.TQPoint") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,TQListViewItem* arg2,const TQPoint arg3,int arg4)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(ILjava/lang/Object;Ljava/lang/Object;I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									(jint) arg1,
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.TQListBoxItem"),
									QtSupport::objectForQtKey(env, (void *) &arg3, "org.kde.qt.TQPoint"),
									(jint) arg4 );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(long arg)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(J)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, arg);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(long arg1,const TQByteArray& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(J[B)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									(jlong) arg1,
									QtSupport::fromQByteArray(env, (TQByteArray *) &arg2) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(long arg1,const TQString& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(JLjava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid, (jlong) arg1, QtSupport::fromQString(env, (TQString *) &arg2));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQAction * arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQAction"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQDir::SortSpec arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, (jint) arg1);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQDockWindow * arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQDockWindow"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQDockWindow::Place arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid, arg1);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQDropEvent* arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQDropEvent"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQDropEvent* arg1,const TQValueList<TQIconDragItem>& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQDropEvent"),
									QtSupport::arrayWithQIconDragItemList(env, (TQValueList<TQIconDragItem> *) &arg2) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQDropEvent* arg1,TQListViewItem*arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQDropEvent"),
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.TQListViewItem") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQDropEvent*arg1,TQListViewItem*arg2,TQListViewItem*arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQDropEvent"),
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.TQListViewItem"),
									QtSupport::objectForQtKey(env, (void *) arg3, "org.kde.qt.TQListViewItem") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQIconViewItem* arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQIconViewItem"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQIconViewItem* arg1,const TQPoint& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQIconViewItem"),
									QtSupport::objectForQtKey(env, (void *) &arg2, "org.kde.qt.TQPoint") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQIconViewItem* arg1,const TQString& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQIconViewItem"),
									QtSupport::fromQString(env, (TQString *) &arg2) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQListBoxItem* arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQListBoxItem"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQListBoxItem* arg1,const TQPoint& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQListBoxItem"),
									QtSupport::objectForQtKey(env, (void *) &arg2, "org.kde.qt.TQPoint") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QList<TQListViewItem> arg1,QList<TQListViewItem> arg2,QList<TQListViewItem> arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.QList"),
									QtSupport::objectForQtKey(env, (void *) &arg2, "org.kde.qt.QList"),
									QtSupport::objectForQtKey(env, (void *) &arg3, "org.kde.qt.QList") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQListViewItem*arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQListViewItem"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQListViewItem* arg1,const TQPoint& arg2,int arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQListBoxItem"),
									QtSupport::objectForQtKey(env, (void *) &arg2, "org.kde.qt.TQPoint"),
									(jint) arg3 );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQListViewItem* arg1,const TQString& arg2,int arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQListViewItem"),
									QtSupport::fromQString(env, (TQString *) &arg2),
									(jint) arg3 );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQListViewItem* arg1,int arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQListViewItem"),
									(jint) arg2 );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQListViewItem* arg1, int arg2, const TQString& arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;ILjava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQListViewItem"),
									(jint) arg2,
									QtSupport::fromQString(env, (TQString *) &arg3) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQListViewItem* arg1,TQListViewItem*arg2,TQListViewItem*arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQListViewItem"),
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.TQListViewItem"),
									QtSupport::objectForQtKey(env, (void *) arg3, "org.kde.qt.TQListViewItem") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQNetworkOperation* arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQNetworkOperation") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQObject* arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQObject") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQPopupMenu* arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQPopupMenu") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQSqlCursor::Mode arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, (jint) arg1);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQSql::Op arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, (jint) arg1);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQSqlRecord* arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQSqlRecord") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQTextEdit::VerticalAlignment arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, (jint) arg1);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQToolBar* arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQToolBar"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(Orientation arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid, (jint) arg1);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQWidget* arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQWidget"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQWidget* arg1,const TQDropEvent* arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, 
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQWidget"),
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.TQDropEvent") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQWidget* arg1,const TQPoint& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, 
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQWidget"),
									QtSupport::objectForQtKey(env, (void *) &arg2, "org.kde.qt.TQPoint") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQWidget* arg1,const TQString& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQWidget"),
									QtSupport::fromQString(env, (TQString *) &arg2) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(TQWidget* arg1,TQPixmap arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, 
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.TQWidget"),
									QtSupport::objectForQtKey(env, (void *) &arg2, "org.kde.qt.TQPixmap") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(short arg)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(S)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, arg);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(unsigned char arg)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(B)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, arg);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(jobjectArray args)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "([Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, args);
	env->PopLocalFrame(0);
}

JavaSlotFactory::JavaSlotFactory()
{
}

JavaSlot *
JavaSlotFactory::createJavaSlot(JNIEnv * env, jobject receiver, jstring slot)
{
	return new JavaSlot(env, receiver, slot);
}

JavaSignal::JavaSignal()
{
}

JavaSignal::~JavaSignal()
{
}

void
JavaSignal::emitArgs(jobjectArray args)
{
	emit signalJava(args);
}


