/***************************************************************************
* $Id$
**
* Copyright (C) 1992-2000 Trolltech AS.  All rights reserved.
**
* This file is part of an example program for Qt.  This example
* program may be used, distributed and modified without limitation.
**
****************************************************************************/

import org.kde.qt.*;
import java.util.HashMap;

class TextEdit  extends QMainWindow
{


private    QAction actionTextBold,
	actionTextUnderline,
	actionTextItalic,
	actionTextColor,
	actionAlignLeft,
	actionAlignCenter,
	actionAlignRight,
	actionAlignJustify;
    QComboBox comboStyle,
	comboFont,
	comboSize;
private    QTabWidget tabWidget;
private    HashMap filenames = new HashMap();



TextEdit( )
{
	this(null, null);
}

TextEdit( QWidget parent, String name )
{
    super( parent, name );
    setupFileActions();
    setupEditActions();
    setupTextActions();

    tabWidget = new QTabWidget( this );
    connect( tabWidget, SIGNAL(" currentChanged( QWidget  )"),
	     this, SLOT(" editorChanged( QWidget  )") );
    setCentralWidget( tabWidget );

    if ( qApp().args().length == 0 ) {
	load( "example.html" );
    } else {
	for ( int i = 0; i < qApp().args().length; ++i )
	    load( qApp().args()[ i ] );
    }
}

void setupFileActions()
{
    QToolBar tb = new QToolBar( this );
    tb.setLabel( "File Actions" );
    QPopupMenu menu = new QPopupMenu( this );
    menuBar().insertItem( tr( "&File" ), menu );

    QAction a;
    a = new QAction( tr( "New" ), new QIconSet(new QPixmap( "filenew.xpm" )), tr( "&New..." ), new QKeySequence(CTRL + Key_N), this, "fileNew" );
    connect( a, SIGNAL(" activated()"), this, SLOT(" fileNew()") );
    a.addTo( tb );
    a.addTo( menu );
    a = new QAction( tr( "Open" ), new QIconSet(new QPixmap( "fileopen.xpm" )), tr( "&Open..." ), new QKeySequence(CTRL + Key_O), this, "fileOpen" );
    connect( a, SIGNAL(" activated()"), this, SLOT(" fileOpen()") );
    a.addTo( tb );
    a.addTo( menu );
    menu.insertSeparator();
    a = new QAction( tr( "Save" ), new QIconSet(new QPixmap( "filesave.xpm" )), tr( "&Save..." ), new QKeySequence(CTRL + Key_S), this, "fileSave" );
    connect( a, SIGNAL(" activated()"), this, SLOT(" fileSave()") );
    a.addTo( tb );
    a.addTo( menu );
    a = new QAction( tr( "Save As" ), new QIconSet(new QPixmap()), tr( "Save &As..." ), new QKeySequence(0), this, "fileSaveAs" );
    connect( a, SIGNAL(" activated()"), this, SLOT(" fileSaveAs()") );
    a.addTo( menu );
    menu.insertSeparator();
    a = new QAction( tr( "Print" ), new QIconSet(new QPixmap( "fileprint.xpm" )), tr( "&Print..." ), new QKeySequence(CTRL + Key_P), this, "filePrint" );
    connect( a, SIGNAL(" activated()"), this, SLOT(" filePrint()") );
    a.addTo( tb );
    a.addTo( menu );
    a = new QAction( tr( "Close" ), new QIconSet(new QPixmap()), tr( "&Close" ), new QKeySequence(0), this, "fileClose" );
    connect( a, SIGNAL(" activated()"), this, SLOT(" fileClose()") );
    a.addTo( menu );
    a = new QAction( tr( "Exit" ), new QIconSet(new QPixmap()), tr( "E&xit" ), new QKeySequence(0), this, "fileExit" );
    connect( a, SIGNAL(" activated()"), this, SLOT(" fileExit()") );
    a.addTo( menu );
}

void setupEditActions()
{
    QToolBar tb = new QToolBar( this );
    tb.setLabel( "Edit Actions" );
    QPopupMenu menu = new QPopupMenu( this );
    menuBar().insertItem( tr( "&Edit" ), menu );

    QAction a;
    a = new QAction( tr( "Undo" ), new QIconSet(new QPixmap( "editundo.xpm" )), tr( "&Undo" ), new QKeySequence(CTRL + Key_Z), this, "editUndo" );
    connect( a, SIGNAL(" activated()"), this, SLOT(" editUndo()") );
    a.addTo( tb );
    a.addTo( menu );
    a = new QAction( tr( "Redo" ), new QIconSet(new QPixmap( "editredo.xpm" )), tr( "&Redo" ), new QKeySequence(CTRL + Key_Y), this, "editRedo" );
    connect( a, SIGNAL(" activated()"), this, SLOT(" editRedo()") );
    a.addTo( tb );
    a.addTo( menu );
    menu.insertSeparator();
    a = new QAction( tr( "Copy" ), new QIconSet(new QPixmap( "editcopy.xpm" )), tr( "&Copy" ), new QKeySequence(CTRL + Key_C), this, "editCopy" );
    connect( a, SIGNAL(" activated()"), this, SLOT(" editCopy()") );
    a.addTo( tb );
    a.addTo( menu );
    a = new QAction( tr( "Cut" ), new QIconSet(new QPixmap( "editcut.xpm" )), tr( "Cu&t" ), new QKeySequence(CTRL + Key_X), this, "editCut" );
    connect( a, SIGNAL(" activated()"), this, SLOT(" editCut()") );
    a.addTo( tb );
    a.addTo( menu );
    a = new QAction( tr( "Paste" ), new QIconSet(new QPixmap( "editpaste.xpm" )), tr( "&Paste" ), new QKeySequence(CTRL + Key_V), this, "editPaste" );
    connect( a, SIGNAL(" activated()"), this, SLOT(" editPaste()") );
    a.addTo( tb );
    a.addTo( menu );
}

void setupTextActions()
{
    QToolBar tb = new QToolBar( this );
    tb.setLabel( "Format Actions" );
    QPopupMenu menu = new QPopupMenu( this );
    menuBar().insertItem( tr( "F&ormat" ), menu );

    comboStyle = new QComboBox( false, tb );
    comboStyle.insertItem( "Standard" );
    comboStyle.insertItem( "Bullet List (Disc)" );
    comboStyle.insertItem( "Bullet List (Circle)" );
    comboStyle.insertItem( "Bullet List (Square)" );
    comboStyle.insertItem( "Ordered List (Decimal)" );
    comboStyle.insertItem( "Ordered List (Alpha lower)" );
    comboStyle.insertItem( "Ordered List (Alpha upper)" );
    connect( comboStyle, SIGNAL(" activated( int )"),
	     this, SLOT(" textStyle( int )") );

    comboFont = new QComboBox( true, tb );
    QFontDatabase db = new QFontDatabase();
    comboFont.insertStringList( (String[]) db.families().toArray(new String[0]) );
    connect( comboFont, SIGNAL(" activated( String  )"),
	     this, SLOT(" textFamily( String  )") );
    comboFont.lineEdit().setText( QApplication.font().family() );

    comboSize = new QComboBox( true, tb );
    int[] sizes = db.standardSizes();
    for (int i = 0; i < sizes.length; i++ )
	comboSize.insertItem( Integer.toString(sizes[i]) );
    connect( comboSize, SIGNAL(" activated(  String  )"),
	     this, SLOT(" textSize(  String  )") );
    comboSize.lineEdit().setText( Integer.toString( QApplication.font().pointSize() ) );

    actionTextBold = new QAction( tr( "Bold" ), new QIconSet(new QPixmap( "textbold.xpm" )), tr( "&Bold" ), new QKeySequence(CTRL + Key_B), this, "textBold" );
    connect( actionTextBold, SIGNAL(" activated()"), this, SLOT(" textBold()") );
    actionTextBold.addTo( tb );
    actionTextBold.addTo( menu );
    actionTextBold.setToggleAction( true );
    actionTextItalic = new QAction( tr( "Italic" ), new QIconSet(new QPixmap( "textitalic.xpm" )), tr( "&Italic" ), new QKeySequence(CTRL + Key_I), this, "textItalic" );
    connect( actionTextItalic, SIGNAL(" activated()"), this, SLOT(" textItalic()") );
    actionTextItalic.addTo( tb );
    actionTextItalic.addTo( menu );
    actionTextItalic.setToggleAction( true );
    actionTextUnderline = new QAction( tr( "Underline" ), new QIconSet(new QPixmap( "textunder.xpm" )), tr( "&Underline" ), new QKeySequence(CTRL + Key_U), this, "textUnderline" );
    connect( actionTextUnderline, SIGNAL(" activated()"), this, SLOT(" textUnderline()") );
    actionTextUnderline.addTo( tb );
    actionTextUnderline.addTo( menu );
    actionTextUnderline.setToggleAction( true );
    menu.insertSeparator();

    QActionGroup grp = new QActionGroup( this );
    grp.setExclusive( true );
    connect( grp, SIGNAL(" selected( QAction )"), this, SLOT(" textAlign( QAction )") );

    actionAlignLeft = new QAction( tr( "Left" ), new QIconSet(new QPixmap( "textleft.xpm" )), tr( "&Left" ), new QKeySequence(CTRL + Key_L), grp, "textLeft" );
    actionAlignLeft.addTo( tb );
    actionAlignLeft.addTo( menu );
    actionAlignLeft.setToggleAction( true );
    actionAlignCenter = new QAction( tr( "Center" ), new QIconSet(new QPixmap( "textcenter.xpm" )), tr( "C&enter" ), new QKeySequence(CTRL + Key_E), grp, "textCenter" );
    actionAlignCenter.addTo( tb );
    actionAlignCenter.addTo( menu );
    actionAlignCenter.setToggleAction( true );
    actionAlignRight = new QAction( tr( "Right" ), new QIconSet(new QPixmap( "textright.xpm" )), tr( "&Right" ), new QKeySequence(CTRL + Key_R), grp, "textRight" );
    actionAlignRight.addTo( tb );
    actionAlignRight.addTo( menu );
    actionAlignRight.setToggleAction( true );
    actionAlignJustify = new QAction( tr( "Justify" ), new QIconSet(new QPixmap( "textjustify.xpm" )), tr( "&Justify" ), new QKeySequence(CTRL + Key_J), grp, "textjustify" );
    actionAlignJustify.addTo( tb );
    actionAlignJustify.addTo( menu );
    actionAlignJustify.setToggleAction( true );

    menu.insertSeparator();

    QPixmap pix = new QPixmap( 16, 16 );
    pix.fill( black() );
    actionTextColor = new QAction( tr( "Color" ), new QIconSet(pix), tr( "&Color..." ),  new QKeySequence(0), this, "textColor" );
    connect( actionTextColor, SIGNAL(" activated()"), this, SLOT(" textColor()") );
    actionTextColor.addTo( tb );
    actionTextColor.addTo( menu );
}

void load(  String f )
{
    if ( !QFile.exists( f ) )
	return;
    QTextEdit edit = new QTextEdit( tabWidget );
    doConnections( edit );
    tabWidget.addTab( edit, new QFileInfo( f ).fileName() );
    QFile file = new QFile( f );
    if ( !file.open( QIODevice.IO_ReadOnly ) )
	return;
    QTextStream ts = new QTextStream( file );
    edit.setText( ts.read() );
    tabWidget.showPage( edit );
    edit.viewport().setFocus();
    filenames.put( edit, f );
}

QTextEdit currentEditor()
{
    if ( tabWidget.currentPage() != null &&
	 tabWidget.currentPage().inherits( "QTextEdit" ) )
	return (QTextEdit)tabWidget.currentPage();
    return null;
}

void doConnections( QTextEdit e )
{
    connect( e, SIGNAL(" currentFontChanged( QFont  )"),
	     this, SLOT(" fontChanged(  QFont  )") );
    connect( e, SIGNAL(" currentColorChanged(  QColor  )"),
	     this, SLOT(" colorChanged(  QColor  )") );
    connect( e, SIGNAL(" currentAlignmentChanged( int )"),
	     this, SLOT(" alignmentChanged( int )") );
}

void fileNew()
{
    QTextEdit edit = new QTextEdit( tabWidget );
    edit.setTextFormat( RichText );
    doConnections( edit );
    tabWidget.addTab( edit, tr( "noname" ) );
    tabWidget.showPage( edit );
    edit.viewport().setFocus();
}

void fileOpen()
{
    String fn = QFileDialog.getOpenFileName( "", tr( "HTML-Files (*.htm *.html);;All Files (*)" ), this );
    if ( !fn.equals("") )
	load( fn );
}

void fileSave()
{
    if ( currentEditor() == null )
	return;
    String fn;
    if ( !filenames.containsKey( currentEditor() ) ) {
	fileSaveAs();
    } else {
	QFile file = new QFile( (String) filenames.get( currentEditor() ) );
	if ( !file.open( QIODevice.IO_WriteOnly ) )
	    return;
	QTextStream ts = new QTextStream( file );
	ts.writeRawBytes(currentEditor().text(), currentEditor().text().length());
    }
}

void fileSaveAs()
{
    if ( currentEditor() == null )
	return;
    String fn = QFileDialog.getSaveFileName( "", tr( "HTML-Files (*.htm *.html);;All Files (*)" ), this );
    if ( !fn.equals("") ) {
	filenames.put( currentEditor(), fn );
	fileSave();
	tabWidget.setTabLabel( currentEditor(), new QFileInfo( fn ).fileName() );
    }
}

void filePrint()
{
    if ( currentEditor() == null )
	return;
    QPrinter printer = new QPrinter();
    printer.setFullPage(true);
    if ( printer.setup( this ) ) {
	QPainter p = new QPainter( printer );
	// Check that there is a valid device to print to.
	if ( p.device() == null ) return;
	QPaintDeviceMetrics metrics = new QPaintDeviceMetrics( p.device() );
	int dpix = metrics.logicalDpiX();
	int dpiy = metrics.logicalDpiY();
	 int margin = 72; // pt
	QRect body = new QRect( margin * dpix / 72, margin * dpiy / 72,
		    metrics.width() - margin * dpix / 72 * 2,
		    metrics.height() - margin * dpiy / 72 * 2 );
	QFont font = new QFont( "times", 10 );
	QSimpleRichText richText = new QSimpleRichText( currentEditor().text(), font, currentEditor().context(), currentEditor().styleSheet(),
				  currentEditor().mimeSourceFactory(), body.height() );
	richText.setWidth( p, body.width() );
	QRect view = new QRect( body.topLeft(), body.bottomRight() );
	int page = 1;
	do {
	    richText.draw( p, body.left(), body.top(), view, colorGroup() );
	    view.moveBy( 0, body.height() );
	    p.translate( 0 , -body.height() );
	    p.setFont( font );
	    p.drawText( view.right() - p.fontMetrics().width( Integer.toString(page) ),
			view.bottom() + p.fontMetrics().ascent() + 5, Integer.toString(page) );
	    if ( view.top()  >= richText.height() )
		break;
	    printer.newPage();
	    page++;
	} while (true);
    }
}

void fileClose()
{
//    delete currentEditor();
    if ( currentEditor() != null )
	currentEditor().viewport().setFocus();				
}

void fileExit()
{
    qApp().quit();
}

void editUndo()
{
    if ( currentEditor() == null )
	return;
    currentEditor().undo();
}

void editRedo()
{
    if ( currentEditor() == null )
	return;
    currentEditor().redo();
}

void editCut()
{
    if ( currentEditor() == null )
	return;
    currentEditor().cut();
}

void editCopy()
{
    if ( currentEditor() == null )
	return;
    currentEditor().copy();
}

void editPaste()
{
    if ( currentEditor() == null )
	return;
    currentEditor().paste();
}

void textBold()
{
    if ( currentEditor() == null )
	return;
    currentEditor().setBold( actionTextBold.isOn() );
}

void textUnderline()
{
    if ( currentEditor() == null )
	return;
    currentEditor().setUnderline( actionTextUnderline.isOn() );
}

void textItalic()
{
    if ( currentEditor() == null )
	return;
    currentEditor().setItalic( actionTextItalic.isOn() );
}

void textFamily(  String f )
{
    if ( currentEditor() == null )
	return;
    currentEditor().setFamily( f );
    currentEditor().viewport().setFocus();
}

void textSize(  String p )
{
    if ( currentEditor() == null )
	return;
    currentEditor().setPointSize( Integer.parseInt(p) );
    currentEditor().viewport().setFocus();
}

void textStyle( int i )
{
    if ( currentEditor() == null )
	return;
    if ( i == 0 )
	currentEditor().setParagType( QStyleSheetItem.DisplayBlock, QStyleSheetItem.ListDisc );
    else if ( i == 1 )
	currentEditor().setParagType( QStyleSheetItem.DisplayListItem, QStyleSheetItem.ListDisc );
    else if ( i == 2 )
	currentEditor().setParagType( QStyleSheetItem.DisplayListItem, QStyleSheetItem.ListCircle );
    else if ( i == 3 )
	currentEditor().setParagType( QStyleSheetItem.DisplayListItem, QStyleSheetItem.ListSquare );
    else if ( i == 4 )
	currentEditor().setParagType( QStyleSheetItem.DisplayListItem, QStyleSheetItem.ListDecimal );
    else if ( i == 5 )
	currentEditor().setParagType( QStyleSheetItem.DisplayListItem, QStyleSheetItem.ListLowerAlpha );
    else if ( i == 6 )
	currentEditor().setParagType( QStyleSheetItem.DisplayListItem, QStyleSheetItem.ListUpperAlpha );
    currentEditor().viewport().setFocus();
}

void textColor()
{
    if ( currentEditor() == null )
	return;
    QColor col = QColorDialog.getColor( currentEditor().color(), this );
    if ( !col.isValid() )
	return;
    currentEditor().setColor( col );
    QPixmap pix = new QPixmap( 16, 16 );
    pix.fill( black() );
    actionTextColor.setIconSet( new QIconSet(pix) );
}

void textAlign( QAction a )
{
    if ( currentEditor() == null )
	return;
    if ( a == actionAlignLeft )
	currentEditor().setAlignment( AlignLeft );
    else if ( a == actionAlignCenter )
	currentEditor().setAlignment( AlignHCenter );
    else if ( a == actionAlignRight )
	currentEditor().setAlignment( AlignRight );
    else if ( a == actionAlignJustify )
	currentEditor().setAlignment( AlignJustify );
}

void fontChanged(  QFont f )
{
    comboFont.lineEdit().setText( f.family() );
    comboSize.lineEdit().setText( Integer.toString( f.pointSize() ) );
    actionTextBold.setOn( f.bold() );
    actionTextItalic.setOn( f.italic() );
    actionTextUnderline.setOn( f.underline() );
}

void colorChanged(  QColor c )
{
    QPixmap pix = new QPixmap( 16, 16 );
    pix.fill( c );
    actionTextColor.setIconSet( new QIconSet(pix) );
}

void alignmentChanged( int a )
{
    if ( ( a == AlignAuto ) || ( a & AlignLeft ) != 0)
	actionAlignLeft.setOn( true );
    else if ( ( a & AlignHCenter ) != 0 )
	actionAlignCenter.setOn( true );
    else if ( ( a & AlignRight ) != 0 )
	actionAlignRight.setOn( true );
    else if ( ( a & AlignJustify ) != 0 )
	actionAlignJustify.setOn( true );
}

void editorChanged( QWidget w )
{
    if ( currentEditor() == null )
	return;
    fontChanged( currentEditor().font() );
    colorChanged( currentEditor().color() );
    alignmentChanged( currentEditor().alignment() );
}
}
