// KDat - a tar-based DAT archiver
// Copyright (C) 1998-2000  Sean Vyain, svyain@mail.tds.net
// Copyright (C) 2001-2002  Lawrence Widman, kdat@cardiothink.com
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <time.h>

#include <tqlabel.h>
#include <tqlayout.h>
#include <kpushbutton.h>
#include <kstdguiitem.h>

#include <kapplication.h>
#include <klocale.h>
#include <kmessagebox.h>

#include "LoggerWidget.h"
#include "Options.h"
#include "IndexDlg.h"
#include "Tape.h"
#include "TapeDrive.h"
#include "TapeManager.h"
#include "TarParser.h"
#include "Util.h"

#include "IndexDlg.moc"

IndexDlg::IndexDlg( Tape* tape, TQWidget* parent, const char* name )
        : TQDialog( parent, name, TRUE ),
          _tarParser( NULL ),
          _tape( tape ),
          _archive( NULL ),
          _totalKBytes( 0.0 ),
          _archiveCount( 0 ),
          _fileCount( 0 ),
          _totalFileCount( 0 ),
          _aborted( FALSE ),
          _numFiles( 0 ),
          _fileSize( -1 ),
          _fileMTime( -1 ),
          _fileStartRecord( -1 )
{
    setCaption( i18n( "KDat: Index" ) );
    setIconText( i18n( "KDat: Index" ) );

    resize( 500, 300 );

    const int labelWidth = 96;

    TQFrame* f1 = new TQFrame( this );
    f1->setFrameStyle( TQFrame::Panel | TQFrame::Sunken );

    TQFrame* f2 = new TQFrame( this );
    f2->setFrameStyle( TQFrame::Panel | TQFrame::Sunken );

    TQLabel* lbl1 = new TQLabel( i18n( "Elapsed time:" ), f1 );
    lbl1->setFixedSize( labelWidth, lbl1->sizeHint().height() );

    _elapsedTime = new TQLabel( i18n( "00:00:00" ), f1 );
    _elapsedTime->setFixedHeight( _elapsedTime->sizeHint().height() );

    TQLabel* lbl2 = new TQLabel( i18n( "Archives:" ), f2 );
    lbl2->setFixedSize( labelWidth, lbl2->sizeHint().height() );

    _archives = new TQLabel( i18n( "0" ), f2 );
    _archives->setFixedHeight( _archives->sizeHint().height() );

    TQLabel* lbl3 = new TQLabel( i18n( "KB read:" ), f1 );
    lbl3->setFixedSize( labelWidth, lbl3->sizeHint().height() );

    _kbytesRead = new TQLabel( i18n( "0KB" ), f1 );
    _kbytesRead->setFixedHeight( _kbytesRead->sizeHint().height() );

    TQLabel* lbl4 = new TQLabel( i18n( "Files:" ), f2 );
    lbl4->setFixedSize( labelWidth, lbl4->sizeHint().height() );

    _files = new TQLabel( i18n( "0" ), f2 );
    _files->setFixedHeight( _kbytesRead->sizeHint().height() );

    TQLabel* lbl5 = new TQLabel( i18n( "Transfer rate:" ), f1 );
    lbl5->setFixedSize( labelWidth, lbl5->sizeHint().height() );

    _transferRate = new TQLabel( i18n( "0KB/min" ), f1 );
    _transferRate->setFixedHeight( _transferRate->sizeHint().height() );

    TQLabel* lbl6 = new TQLabel( i18n( "Total files:" ), f2 );
    lbl6->setFixedSize( labelWidth, lbl6->sizeHint().height() );

    _totalFiles = new TQLabel( i18n( "0" ), f2 );
    _totalFiles->setFixedHeight( _totalFiles->sizeHint().height() );

    _log = new LoggerWidget( i18n( "Index log:" ), this );

    _ok = new KPushButton( KStdGuiItem::ok(), this );
    _ok->setFixedSize( 80, _ok->sizeHint().height() );
    connect( _ok, TQT_SIGNAL( clicked() ), this, TQT_SLOT( slotOK() ) );
    _ok->setEnabled( FALSE );

    _save = new TQPushButton( i18n( "Save Log..." ), this );
    _save->setFixedSize( 80, _save->sizeHint().height() );
    connect( _save, TQT_SIGNAL( clicked() ), _log, TQT_SLOT( save() ) );
    _save->setEnabled( FALSE );

    _abort = new TQPushButton( i18n( "Abort" ), this );
    _abort->setFixedSize( 80, _abort->sizeHint().height() );
    connect( _abort, TQT_SIGNAL( clicked() ), this, TQT_SLOT( slotAbort() ) );

    TQVBoxLayout* l1 = new TQVBoxLayout( this, 8, 4 );

    TQHBoxLayout* l1_1 = new TQHBoxLayout();
    l1->addLayout( l1_1 );
    l1_1->addStrut( 3 * lbl1->height() + 16 );
    l1_1->addWidget( f1 );
    l1_1->addWidget( f2 );

    TQVBoxLayout* l1_1_1 = new TQVBoxLayout( f1, 4, 4 );

    TQHBoxLayout* l1_1_1_1 = new TQHBoxLayout();
    l1_1_1->addLayout( l1_1_1_1 );
    l1_1_1_1->addWidget( lbl1 );
    l1_1_1_1->addWidget( _elapsedTime, 1 );

    TQHBoxLayout* l1_1_1_2 = new TQHBoxLayout();
    l1_1_1->addLayout( l1_1_1_2 );
    l1_1_1_2->addWidget( lbl3 );
    l1_1_1_2->addWidget( _kbytesRead, 1 );

    TQHBoxLayout* l1_1_1_3 = new TQHBoxLayout();
    l1_1_1->addLayout( l1_1_1_3 );
    l1_1_1_3->addWidget( lbl5 );
    l1_1_1_3->addWidget( _transferRate, 1 );

    TQVBoxLayout* l1_1_2 = new TQVBoxLayout( f2, 4, 4 );

    TQHBoxLayout* l1_1_2_1 = new TQHBoxLayout();
    l1_1_2->addLayout( l1_1_2_1 );
    l1_1_2_1->addWidget( lbl2 );
    l1_1_2_1->addWidget( _archives, 1 );

    TQHBoxLayout* l1_1_2_2 = new TQHBoxLayout();
    l1_1_2->addLayout( l1_1_2_2 );
    l1_1_2_2->addWidget( lbl4 );
    l1_1_2_2->addWidget( _files, 1 );

    TQHBoxLayout* l1_1_2_3 = new TQHBoxLayout();
    l1_1_2->addLayout( l1_1_2_3 );
    l1_1_2_3->addWidget( lbl6 );
    l1_1_2_3->addWidget( _totalFiles, 1 );

    l1->addWidget( _log, 1 );

    TQHBoxLayout* l1_2 = new TQHBoxLayout();
    l1->addLayout( l1_2 );
    l1_2->addStretch( 1 );
    l1_2->addWidget( _ok );
    l1_2->addWidget( _save );
    l1_2->addWidget( _abort );
}

IndexDlg::~IndexDlg()
{
    delete _tarParser;
}

void IndexDlg::show()
{
    startTimer( 100 );

    TQDialog::show();
}

void IndexDlg::slotEntry( const TQString & name, int size, int mtime, int record )
{
    if ( !_fileName.isEmpty() ) {
        _archive->addFile( _fileSize, _fileMTime, _fileStartRecord, record, _fileName );
    }

    _fileName        = name;
    _fileSize        = size;
    _fileMTime       = mtime;
    _fileStartRecord = record;

    /* 2002-01-28 LEW */
    // printf("IndexDlg::slotEntry called with \"%s\", %d bytes\n", name.latin1(), size);
    /* 2002-01-28 LEW */

    TQString tmp;
    tmp.setNum( ++_numFiles );
    _files->setText( tmp );
    _totalFileCount++;
    tmp.setNum( _totalFileCount );
    _totalFiles->setText( tmp );

    _log->append( name );
}

void IndexDlg::slotOK()
{
    if ( _aborted ) {
        reject();
    } else {
        accept();
    }
}

void IndexDlg::slotAbort()
{
    TQT_TQOBJECT(this)->killTimers();
    _aborted = TRUE;
}

void IndexDlg::timerEvent( TQTimerEvent* )
{
    TQT_TQOBJECT(this)->killTimers();

    // Rewind tape.
    _log->append( i18n( "Rewinding tape." ) );
    if ( !TapeDrive::instance()->rewind() ) {
        KMessageBox::error( this, i18n( "Cannot rewind tape. Indexing aborted." ));
        _ok->setEnabled( TRUE );
        _save->setEnabled( TRUE );
        _abort->setEnabled( FALSE );
        _log->append( i18n( "Cannot rewind tape." ) );
        return;
    }

    // Skip tape ID.
    if ( !TapeDrive::instance()->nextFile( 1 ) ) {
        KMessageBox::error( this, i18n( "Failed to skip tape ID. Indexing aborted." ));
        _ok->setEnabled( TRUE );
        _save->setEnabled( TRUE );
        _abort->setEnabled( FALSE );
        _log->append( i18n( "Failed to skip tape ID." ) );
        return;
    }

    _startTime = time( NULL );

    int oldElapsed = 0;
    TQString msg;
    int count;
    char *buf = new char[Options::instance()->getTapeBlockSize()];
    bool atLeastOneGoodRead = TRUE;
    while ( ( !_aborted ) && ( atLeastOneGoodRead ) ) {
        atLeastOneGoodRead = FALSE;
        _archive = NULL;

        delete _tarParser;
        _tarParser = new TarParser();
        connect( _tarParser, TQT_SIGNAL( sigEntry( const TQString &, int, int, int ) ), this, TQT_SLOT( slotEntry( const TQString &, int, int, int ) ) );

        int endBlock = 0;
        int leftover = 0;

        while ( ( count = TapeDrive::instance()->read( buf, Options::instance()->getTapeBlockSize() ) ) > 0 ) {
            // Is this the first block of the archive?
            if ( !atLeastOneGoodRead ) {
                atLeastOneGoodRead = TRUE;
                _archiveCount++;
                msg.setNum( _archiveCount );
                _archives->setText( msg );
                msg = i18n( "Indexing archive %1." ).arg( _archiveCount );
                _log->append( msg );

                msg = i18n( "Archive %1" ).arg( _archiveCount );
                _archive = new Archive( _tape, 0, msg );
                _fileName = TQString();
                _fileSize = -1;
                _fileMTime = -1;
                _fileStartRecord = -1;
                _numFiles = 0;
            }

            leftover += count;
            endBlock += leftover / Options::instance()->getTapeBlockSize();
            leftover %= Options::instance()->getTapeBlockSize();

            _tarParser->slotData( buf, count );
            _totalKBytes += (float)count / 1024.0;

            // Update stats.
            int elapsed = time( NULL ) - _startTime;
            if ( elapsed > oldElapsed )
            {
                updateStats();
                KApplication::kApplication()->processEvents();
                if ( _aborted ) {
                    break;
                }
                oldElapsed = elapsed;
            }
        }
        if ( _aborted ) {
            break;
        }
        if ( _archive ) {
            _archive->setEndBlock( endBlock );

            if ( _fileName.length() > 0 ) {
                _archive->addFile( _fileSize, _fileMTime, _fileStartRecord, _archive->getEndBlock() * ( Options::instance()->getTapeBlockSize() / 512 ), _fileName );
            }

            _tape->addChild( _archive );
        }
    }

    delete [] buf;

    updateStats();

    TapeDrive::instance()->rewind();

    if ( !_aborted ) {
        _tape->setName( i18n( "Reindexed Tape" ) );
        _tape->setSize( Options::instance()->getDefaultTapeSize() );
    }

    _ok->setEnabled( TRUE );
    _ok->setDefault( TRUE );
    _save->setEnabled( TRUE );
    _abort->setEnabled( FALSE );
}

void IndexDlg::updateStats()
{
    TQString str;
    int elapsed = time( NULL ) - _startTime;

    str = TQString::fromUtf8( TQCString().sprintf( i18n( "%02d:%02d:%02d" ).utf8(), elapsed / 3600, elapsed / 60 % 60, elapsed % 60 ) );
    _elapsedTime->setText( str );

    str = Util::kbytesToString( (int)_totalKBytes );
    _kbytesRead->setText( str );

    if ( elapsed > 0 ) {
        str = i18n( "%1/min" ).arg(Util::kbytesToString( (int)_totalKBytes * 60 / elapsed ) );
        _transferRate->setText( str );
    }
}
