/***************************************************************************
                          Interface to access Noatun
                             -------------------
    begin                : Mon Jan 15 21:09:00 CEST 2001
    copyright            : (C) 2000-2002 by Stefan Gehn
    email                : metz {AT} gehn {DOT} net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "noatunInterface.h"
#include "noatunInterface.moc"

#include <kapplication.h>
#include <kdebug.h>
#include <tqstringlist.h>
#include <tqstrlist.h>
#include <kurldrag.h>

#define TIMER_FAST  250

NoatunInterface::NoatunInterface() : PlayerInterface()
{
	mTimerValue = TIMER_FAST;
	mNoatunTimer = new TQTimer(this, "mNoatunTimer");

	connect(mNoatunTimer, TQT_SIGNAL(timeout()), TQT_SLOT(updateSlider()));

	connect(kapp->dcopClient(), TQT_SIGNAL(applicationRegistered(const TQCString&)),
		TQT_SLOT(appRegistered(const TQCString&)) );

	connect(kapp->dcopClient(), TQT_SIGNAL(applicationRemoved(const TQCString&)),
		TQT_SLOT(appRemoved(const TQCString&)));

	kapp->dcopClient()->setNotifications(true);
	TQTimer::singleShot(0, this, TQT_SLOT(myInit()));
}

NoatunInterface::~NoatunInterface()
{
	kapp->dcopClient()->setNotifications(false);
}

void NoatunInterface::myInit()
{
	// Start the timer if noatun is already running
	// Needed if user adds applet while running noatun
	if ( findRunningNoatun() )
	{
		emit playerStarted();
		mNoatunTimer->start(mTimerValue);
	}
	else
	{
		emit playerStopped();
		emit newSliderPosition(0,0);
	}
}

void NoatunInterface::appRegistered(const TQCString &appId)
{
	if (appId.contains("noatun",false))
	{
		mAppId = appId;
		emit playerStarted();
		mNoatunTimer->start(mTimerValue);
	}
}

void NoatunInterface::appRemoved(const TQCString &appId)
{
	if (appId.contains("noatun",false))
	{
	 	// is there still another noatun alive?
		if (findRunningNoatun())
			return;
		mNoatunTimer->stop();
		emit playerStopped();
		emit newSliderPosition(0,0);
	}
}

void NoatunInterface::updateSlider()
{
	// length/time in msecs, -1 means "no playobject in noatun"
	int len, time;
	TQByteArray data, replyData;
	TQCString replyType;

	if (!kapp->dcopClient()->call(mAppId, "Noatun", "length()", data, replyType, replyData, false, 200))
	{
		//kdDebug(90200) << "mediacontrol: DCOP communication Error" << endl;
		// -2 is an internal errornumber, might be used later
		len = -2;
	}
	else
	{
		TQDataStream reply(replyData, IO_ReadOnly);
		if (replyType == "int")
		{
			reply >> len;
		}
		else
		{
			//kdDebug(90200) << "mediacontrol: unexpected type of DCOP-reply" << endl;
			// -3 is an internal errornumber, might be used later
			len = -3;
		}
	}

	data = 0;
	replyData = 0;
	replyType = 0;

	if (!kapp->dcopClient()->call(mAppId, "Noatun", "position()", data,
		replyType, replyData, false, 200))
	{
		//kdDebug(90200) << "mediacontrol: DCOP communication error" << endl;
		time = -2;
	}
	else
	{
		TQDataStream reply(replyData, IO_ReadOnly);
		if (replyType == "int")
		{
			reply >> time;
		}
		else
		{
			//kdDebug(90200) << "mediacontrol: unexpected type of DCOP-reply" << endl;
			time = -3;
		}
	}

	if ((time < 0) || (len < 0)) // Noatun isn't playing and thus returns -1
	{
		len = 0;
		time = 0;
	}
	emit newSliderPosition(len/1000,time/1000);
	emit playingStatusChanged(playingStatus());
}

int NoatunInterface::playingStatus()
{
	TQByteArray data, replyData;
	TQCString replyType;

	if (!kapp->dcopClient()->call(mAppId, "Noatun", "state()", data, replyType,
		replyData, false, 200))
	{
		return Stopped;
	}
	else
	{
		int status = 0;
		TQDataStream reply(replyData, IO_ReadOnly);
		if (replyType == "int")
			reply >> status;

		if (status == 1)
			return Paused;
		else if (status == 2)
			return Playing;
		else
			return Stopped;
	}
}


// Drag-n-Drop stuff =================================================================

void NoatunInterface::dragEnterEvent(TQDragEnterEvent* event)
{
//	kdDebug(90200) << "NoatunInterface::dragEnterEvent()" << endl;
	event->accept(KURLDrag::canDecode(event));
}

void NoatunInterface::dropEvent(TQDropEvent* event)
{
//	kdDebug(90200) << "NoatunInterface::dropEvent()" << endl;
	KURL::List list;
	if (KURLDrag::decode(event, list))
	{
		TQByteArray data;
		TQDataStream arg(data, IO_WriteOnly);
		arg << list.toStringList() << false;
		kapp->dcopClient()->send(mAppId, "Noatun", "addFile(TQStringList,bool)", data);
	}
}

// ====================================================================================

void NoatunInterface::sliderStartDrag()
{
	mNoatunTimer->stop();
}

void NoatunInterface::sliderStopDrag()
{
	mNoatunTimer->start(mTimerValue);
}

void NoatunInterface::jumpToTime(int sec)
{
	TQByteArray data;
	TQDataStream arg(data, IO_WriteOnly);
	arg << (sec*1000);  // noatun wants milliseconds
	kapp->dcopClient()->send(mAppId, "Noatun", "skipTo(int)", data);
}

void NoatunInterface::playpause()
{
	if (!findRunningNoatun())
		startPlayer("noatun");
	kapp->dcopClient()->send(mAppId, "Noatun", "playpause()", TQString());
}

void NoatunInterface::stop()
{
	kapp->dcopClient()->send(mAppId, "Noatun", "stop()", TQString());
}

void NoatunInterface::next()
{
	// fastForward() is noatun from kde2
	//kapp->dcopClient()->send("noatun", "Noatun", "fastForward()", TQString());
	kapp->dcopClient()->send(mAppId, "Noatun", "forward()", TQString());
}

void NoatunInterface::prev()
{
	kapp->dcopClient()->send(mAppId, "Noatun", "back()", TQString());
}

void NoatunInterface::volumeUp()
{
	kapp->dcopClient()->send(mAppId, "Noatun", "volumeDown()", TQString());
}

void NoatunInterface::volumeDown()
{
	kapp->dcopClient()->send(mAppId, "Noatun", "volumeUp()", TQString());
}

const TQString NoatunInterface::getTrackTitle() const
{
	TQString title("");
	TQByteArray data, replyData;
	TQCString replyType;
	if (kapp->dcopClient()->call(mAppId, "Noatun", "title()", data, replyType,
		replyData, false, 200))
	{
		TQDataStream reply(replyData, IO_ReadOnly);
		if (replyType == TQSTRING_OBJECT_NAME_STRING)
			reply >> title;
	}
	return title;
}

bool NoatunInterface::findRunningNoatun()
{
	// FIXME: what if we have a dcop app named, let's say, 'noatunfrontend'?
	QCStringList allApps = kapp->dcopClient()->registeredApplications();
	TQValueList<TQCString>::const_iterator iterator;

	for (iterator = allApps.constBegin(); iterator != allApps.constEnd(); ++iterator)
	{
		if ((*iterator).contains("noatun", false))
		{
			mAppId = *iterator;
			return true;
		}
	}
	return false;
}
