/* ============================================================
 *
 * This file is a part of digiKam project
 * http://www.digikam.org
 * 
 * Date        : 2004-08-03
 * Description : setup Image Editor tab.
 *
 * Copyright (C) 2004-2008 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

// TQt includes.

#include <tqlayout.h>
#include <tqcolor.h>
#include <tqhbox.h>
#include <tqvgroupbox.h>
#include <tqlabel.h>
#include <tqwhatsthis.h>
#include <tqcheckbox.h>

// KDE includes.

#include <klocale.h>
#include <kdialog.h>
#include <kcolorbutton.h>
#include <knuminput.h>
#include <kconfig.h>
#include <kapplication.h>

// Local includes.

#include "setupeditor.h"
#include "setupeditor.moc"

namespace Digikam
{
class SetupEditorPriv
{
public:

    SetupEditorPriv()
    {
        hideToolBar          = 0;
        themebackgroundColor = 0;
        backgroundColor      = 0;
        colorBox             = 0;
        overExposureColor    = 0;
        underExposureColor   = 0;
        useRawImportTool     = 0;
    }

    TQHBox        *colorBox;

    TQCheckBox    *hideToolBar;
    TQCheckBox    *themebackgroundColor;
    TQCheckBox    *useRawImportTool;

    KColorButton *backgroundColor;
    KColorButton *underExposureColor;
    KColorButton *overExposureColor;
};

SetupEditor::SetupEditor(TQWidget* parent )
           : TQWidget(parent)
{
    d = new SetupEditorPriv;
    TQVBoxLayout *layout = new TQVBoxLayout( parent, 0, KDialog::spacingHint() );

    // --------------------------------------------------------

    TQVGroupBox *interfaceOptionsGroup = new TQVGroupBox(i18n("Interface Options"), parent);

    d->themebackgroundColor = new TQCheckBox(i18n("&Use theme background color"), interfaceOptionsGroup);

    TQWhatsThis::add(d->themebackgroundColor, i18n("<p>Enable this option to use background theme "
                                             "color in image editor area"));

    d->colorBox = new TQHBox(interfaceOptionsGroup);

    TQLabel *backgroundColorlabel = new TQLabel(i18n("&Background color:"), d->colorBox);

    d->backgroundColor = new KColorButton(d->colorBox);
    backgroundColorlabel->setBuddy(d->backgroundColor);
    TQWhatsThis::add(d->backgroundColor, i18n("<p>Customize background color to use "
                                             "in image editor area."));

    d->hideToolBar = new TQCheckBox(i18n("H&ide toolbar in fullscreen mode"), interfaceOptionsGroup);

    d->useRawImportTool = new TQCheckBox(i18n("Use Raw Import Tool to handle Raw image"), interfaceOptionsGroup);
    TQWhatsThis::add(d->useRawImportTool, i18n("<p>Set on this option to use Raw Import "
                                              "tool before to load a Raw image, "
                                              "to customize indeep decoding settings."));

    // --------------------------------------------------------

    TQVGroupBox *exposureOptionsGroup = new TQVGroupBox(i18n("Exposure Indicators"), parent);

    TQHBox *underExpoBox         = new TQHBox(exposureOptionsGroup);
    TQLabel *underExpoColorlabel = new TQLabel( i18n("&Under-exposure color:"), underExpoBox);
    d->underExposureColor       = new KColorButton(underExpoBox);
    underExpoColorlabel->setBuddy(d->underExposureColor);
    TQWhatsThis::add(d->underExposureColor, i18n("<p>Customize the color used in image editor to identify "
                                                "under-exposed pixels."));

    TQHBox *overExpoBox         = new TQHBox(exposureOptionsGroup);
    TQLabel *overExpoColorlabel = new TQLabel( i18n("&Over-exposure color:"), overExpoBox);
    d->overExposureColor       = new KColorButton(overExpoBox);
    overExpoColorlabel->setBuddy(d->overExposureColor);
    TQWhatsThis::add(d->overExposureColor, i18n("<p>Customize the color used in image editor to identify "
                                               "over-exposed pixels."));

    // --------------------------------------------------------

    layout->addWidget(interfaceOptionsGroup);
    layout->addWidget(exposureOptionsGroup);
    layout->addStretch();

    // --------------------------------------------------------

    connect(d->themebackgroundColor, TQT_SIGNAL(toggled(bool)),
            this, TQT_SLOT(slotThemeBackgroundColor(bool)));

    readSettings();
}

SetupEditor::~SetupEditor()
{
    delete d;
}

void SetupEditor::slotThemeBackgroundColor(bool e)
{
    d->colorBox->setEnabled(!e);
}

void SetupEditor::readSettings()
{
    KConfig* config = kapp->config();
    TQColor Black(TQt::black);
    TQColor White(TQt::white);
    config->setGroup("ImageViewer Settings");
    d->themebackgroundColor->setChecked(config->readBoolEntry("UseThemeBackgroundColor", true));
    d->backgroundColor->setColor(config->readColorEntry("BackgroundColor", &Black));
    d->hideToolBar->setChecked(config->readBoolEntry("FullScreen Hide ToolBar", false));
    d->underExposureColor->setColor(config->readColorEntry("UnderExposureColor", &White));
    d->overExposureColor->setColor(config->readColorEntry("OverExposureColor", &Black));
    d->useRawImportTool->setChecked(config->readBoolEntry("UseRawImportTool", false));
}

void SetupEditor::applySettings()
{
    KConfig* config = kapp->config();
    config->setGroup("ImageViewer Settings");
    config->writeEntry("UseThemeBackgroundColor", d->themebackgroundColor->isChecked());
    config->writeEntry("BackgroundColor",         d->backgroundColor->color());
    config->writeEntry("FullScreen Hide ToolBar", d->hideToolBar->isChecked());
    config->writeEntry("UnderExposureColor",      d->underExposureColor->color());
    config->writeEntry("OverExposureColor",       d->overExposureColor->color());
    config->writeEntry("UseRawImportTool",        d->useRawImportTool->isChecked());
    config->sync();
}

}  // namespace Digikam
