/** -*- C++ -*-
	@file adept/quickfilter.h
	@author Peter Rockai <me@mornfall.net>
*/

#include <klocale.h>
#include <tqlayout.h>
#include <tqtimer.h>

#include <apt-front/cache/entity/entity.h>
#include <apt-front/cache/entity/package.h>
#include <apt-front/predicate/factory.h>
#include <adept/quickfilterui.h>
#include <adept/filterlist.h>
#include <adept/lister.h>
#include <adept/utils.h>

#ifndef EPT_QUICKFILTER_H
#define EPT_QUICKFILTER_H

class KLineEdit;

namespace adept {

template< typename T >
struct QuickFilter : predicate::Implementation< T, QuickFilter< T > >,
                     InterfacingPredicate
{
    enum Type { Regex, Substring, Exact };
    enum What { Name = 0x1, Description = 0x2, Maintainer = 0x4 };

    QuickFilter()
        : m_type( Substring ), m_match( "" ), m_what( Name | Description ) {
        setupPredicate();
    }

    void setupPredicate() {
        predicate::ArgumentList l;
        l.push_back( m_match );
        predicate::Predicate< T > a = not predicate::True< T >();
        if ( m_what & Name ) a = a or predicate::Factory< T >::name( m_match );
        if ( m_what & Description )
            a = a or predicate::Factory< T >::description( m_match );
        if ( m_what & Maintainer )
            a = a or predicate::Factory< T >::maintainer( m_match );
        m_op = a;
        /* m_op = predicate::map(
            predicate::predicate( predicate::Factory< T >::description( "" )
                                  or predicate::Factory< T >::name( "" )
                                  or predicate::Factory< T
        >::maintainer( "" ) ), l ); */
    }

    std::string summary() const {
        return u8( i18n( "Search: " ) ) + "\"" + m_match + "\"";
    }

    void parseArguments( const predicate::ArgumentList & ) {}

    bool operator==( const QuickFilter &o ) const {
        return o.m_type == m_type && o.m_match == m_match;
    }

    std::string typeString() const {
        if (m_type == Regex) return "Regular Expression";
        if (m_type == Substring) return "Substring";
        if (m_type == Exact) return "Exact Match";
    }

    bool operator()( const T &p ) {
        return m_op( p );
    }

    std::string match() const {
        return m_match;
    }

    void setMatch( const std::string &s ) {
        m_match = s;
        setupPredicate();
    }

    void setWhat( int w ) {
        m_what = w;
        setupPredicate();
    }

    int what() { return m_what; }

    virtual void reset() {
        m_match = "";
        setupPredicate();
    }

protected:
    Type m_type;
    std::string m_match;
    int m_what;
    predicate::Predicate< T > m_op;
};

class QuickFilterWidget : public QuickFilterUi
{
    Q_OBJECT
  TQ_OBJECT
public:
    QuickFilterWidget( TQWidget *parent, const char *name = 0 );
    virtual Predicate predicate();
public slots:
    void predicateChanged();
protected slots:
    void textChanged( const TQString & );
protected:
    void mouseReleaseEvent( TQMouseEvent *e );
    TQTimer timer;
};

}

#endif
