/** -*- C++ -*-
    @file adept/desktoplist.cpp
    @author Peter Rockai <me@mornfall.net>
*/

#include <tqimage.h>
#include <tqlabel.h>
#include <tqcursor.h>
#include <tqcheckbox.h>
#include <kstandarddirs.h>
#include <kiconloader.h>
#include <kapplication.h>
#include <klocale.h>
#include <kdebug.h>

#include <adept/desktoplist.h>
#include <adept/utils.h>

namespace adept {

DesktopEntry::DesktopEntry( TQWidget *p, const char *n )
    : DesktopEntryUi( p, n )
{
    adjustFontSize( m_description, -1 );
}

void DesktopEntry::resize( int w, int h )
{
    // kdDebug() << "DesktopEntry::resize( " << w << ", " << h << " )" << endl;
    int margin = layout()->margin() + layout()->spacing() * 2
                 + 6 /* spacer */ + m_icon->width() + m_check->width();
    if ( m_check->isVisible() ) margin += m_check->width() + layout()->spacing();
    int hFW1 = m_name->heightForWidth( w - margin );
    int hFW2 = m_description->heightForWidth( w - margin );
    /* kdDebug() << "margin = " << margin << ", hFW1 = " << hFW1 << ", hFW2 = "
       << hFW2 << endl; */
    int height = 2 * layout()->margin() + layout()->spacing() + hFW1 + hFW2;
    if ( height < 32 + 2*layout()->margin() /* icon size + margin */ )
        height = 32 + 2*layout()->margin();
    // m_description->resize( w - margin, m_description->heightForWidth( w - margin ) );
    TQWidget::resize( w, height );
}

void DesktopEntry::setItem( ExtendableItem *i ) {
    m_polished = false;
    ItemExtender::setItem( i );
    m_name->setText( u8( "<b>" ) + u8( entity().name() ) + u8( "</b>" ) );
    m_description->setText( u8( entity().description() ) );
    if ( !item()->list()->displayCheckboxes() ) {
        m_check->hide();
    }
    if ( !package().valid() || !package().hasVersion() ) {
        setEnabled( false );
    } else {
        connect( m_check, TQT_SIGNAL( toggled( bool ) ),
                 this, TQT_SLOT( toggled() ) );
        notifyPostChange();
    }
}

void DesktopEntry::mousePressEvent( TQMouseEvent *e ) {
    kdDebug() << "DesktopEntry::mousePressEvent" << endl;
    e->ignore();
}

void DesktopEntry::mouseReleaseEvent( TQMouseEvent *e ) {
    kdDebug() << "DesktopEntry::mouseReleaseEvent" << endl;
    e->ignore();
}

void DesktopEntry::showEvent( TQShowEvent *e ) {
    DesktopEntryUi::showEvent( e );
    // DesktopEntryUi::polish();
    if ( m_polished ) return;
    item()->list()->polishing();
    kdDebug() << "polish for " << entity().name() << endl;
    TQImage icon( KGlobal::iconLoader()->iconPath( entity().icon(), -32, true ) );
    if ( icon.isNull() ) {
        icon = TQImage( KGlobal::dirs()->findResource( "desktopicon", entity().icon() ) );
    }
    m_icon->setPixmap( icon.isNull() ? item()->list()->emptyIcon() :
                       TQPixmap( icon.smoothScale( 32, 32, TQ_ScaleMin ) ) );
    item()->list()->polished();
    m_polished = true;
}

void DesktopEntry::toggled() {
    /* if ( package().isInstalled() ) {
        if ( package().markedRemove() )
            package().markKeep();
        else
            package().markRemove();
    } else if ( package().markedInstall() )
        package().markKeep();
    else
    package().markInstall(); */

    if ( package().isInstalled() ) {
        if ( package().markedRemove() )
            item()->list()->fireRequest( package(), component::State::AKeep );
        else
            item()->list()->fireRequest( package(), component::State::ARemove );
    } else if ( package().markedInstall() )
        item()->list()->fireRequest( package(), component::State::AKeep );
    else
        item()->list()->fireRequest( package(), component::State::AInstall );
    notifyPostChange();
}

void DesktopList::polishing() {
    if ( m_polishing == 0 )
        TQApplication::setOverrideCursor( TQCursor( TQt::BusyCursor ) );
    m_polishing++;
    if ( m_polishing % 10 == 0 )
        kapp->processEvents();
}

void DesktopList::polished() {
    if ( m_polishing == childCount() )
        TQApplication::restoreOverrideCursor();
}

void DesktopList::notifyPostChange( component::Base * ) {
    utils::Range< ExtendableItem * > r = extenders();
    for ( ; r != r.end(); ++r ) {
        dynamic_cast< DesktopEntry * >( (*r)->extender() )->notifyPostChange();
    }
}

void DesktopEntry::notifyPostChange() {
    if ( !package().valid() || !package().hasVersion() )
        return;

    m_check->blockSignals( true );
    m_check->setChecked( ( package().isInstalled()
                         || entity().package().markedInstall() )
                         && !entity().package().markedRemove() );
    m_check->blockSignals( false );
}

DesktopList::DesktopList( TQWidget *parent, const char *name )
    : ExtendableList( parent, name ), m_polishing( 0 ), m_displayCheckboxes( true )
{
    observeComponent< component::State >();
    addColumn( i18n( "Application" ) );
    setResizeMode( LastColumn );
    setToggleColumn( -1 ); // no toggling, thanks
    setExtenderHighlight( true );
    m_emptyIcon = TQPixmap(
        TQImage(
            KGlobal::iconLoader()->iconPath( u8( "exec" ), -32, false )
            ).smoothScale( 32, 32, TQ_ScaleMin ) );
    connect( this, TQT_SIGNAL( clicked( TQListViewItem *, const TQPoint &, int ) ),
             this, TQT_SLOT( processClick( TQListViewItem *, const TQPoint &, int ) ) );
}

void DesktopList::processClick( TQListViewItem *it, const TQPoint &, int ) {
    kdDebug() << "DesktopList::processClick..." << endl;
    DesktopItem *i = dynamic_cast< DesktopItem * >( it );
    if ( !i ) return;
    emit showDescription( i->entity() );
}

void DesktopList::insertRange( Range r ) {
    m_range = r;
    DesktopItem *last = 0;
    int i = 0;
    TQApplication::setOverrideCursor( TQCursor( TQt::BusyCursor ) );
    for( ; r != r.end(); r.advance() ) {
        if ( i % 20 == 0 )
            kapp->processEvents();
        if ( !r->package( entity::Package() ).valid() )
            continue;
        DesktopItem *i = last ? new DesktopItem( this, last ) : new DesktopItem( this );
        last = i;
        i->setEntity( *r );
        ++ i;
    }
    TQApplication::restoreOverrideCursor();
}

ItemExtender *DesktopItem::createExtender() {
    return new DesktopEntry();
}

DesktopItem *DesktopEntry::item() const {
    return dynamic_cast< DesktopItem * >( m_item );
}

entity::Desktop DesktopEntry::entity() const {
    return item()->entity();
}

}
