/***************************************************************************
    copyright            : (C) 2003-2006 by Robby Stephenson
    email                : robby@periapsis.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of version 2 of the GNU General Public License as  *
 *   published by the Free Software Foundation;                            *
 *                                                                         *
 ***************************************************************************/

#include "xsltimporter.h"
#include "xslthandler.h"
#include "tellicoimporter.h"
#include "../filehandler.h"
#include "../collection.h"

#include <klocale.h>
#include <kurlrequester.h>

#include <tqhbox.h>
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqgroupbox.h>

#include <memory>

using Tellico::Import::XSLTImporter;

namespace {

static bool isUTF8(const KURL& url_) {
  // read first line to check encoding
  std::auto_ptr<Tellico::FileHandler::FileRef> ref(Tellico::FileHandler::fileRef(url_));
  if(!ref->isValid()) {
    return false;
  }

  ref->open();
  TQTextStream stream(ref->file());
  TQString line = stream.readLine().lower();
  return line.find(TQString::fromLatin1("utf-8")) > 0;
}

}

// always use utf8 for xslt
XSLTImporter::XSLTImporter(const KURL& url_) : Tellico::Import::TextImporter(url_, isUTF8(url_)),
    m_coll(0),
    m_widget(0),
    m_URLRequester(0) {
}

Tellico::Data::CollPtr XSLTImporter::collection() {
  if(m_coll) {
    return m_coll;
  }

  if(m_xsltURL.isEmpty()) {
    // if there's also no widget, then something went wrong
    if(!m_widget) {
      setStatusMessage(i18n("A valid XSLT file is needed to import the file."));
      return 0;
    }
    m_xsltURL = m_URLRequester->url();
  }
  if(m_xsltURL.isEmpty() || !m_xsltURL.isValid()) {
    setStatusMessage(i18n("A valid XSLT file is needed to import the file."));
    return 0;
  }

  XSLTHandler handler(m_xsltURL);
  if(!handler.isValid()) {
    setStatusMessage(i18n("Tellico encountered an error in XSLT processing."));
    return 0;
  }
//  kdDebug() << text() << endl;
  TQString str = handler.applyStylesheet(text());
//  kdDebug() << str << endl;

  Import::TellicoImporter imp(str);
  m_coll = imp.collection();
  setStatusMessage(imp.statusMessage());
  return m_coll;
}

TQWidget* XSLTImporter::widget(TQWidget* parent_, const char* name_) {
  // if the url has already been set, then there's no widget
  if(!m_xsltURL.isEmpty()) {
    return 0;
  }

  m_widget = new TQWidget(parent_, name_);
  TQVBoxLayout* l = new TQVBoxLayout(m_widget);

  TQGroupBox* box = new TQGroupBox(1, Qt::Vertical, i18n("XSLT Options"), m_widget);
  l->addWidget(box);

  (void) new TQLabel(i18n("XSLT file:"), box);
  m_URLRequester = new KURLRequester(box);

  TQString filter = i18n("*.xsl|XSL Files (*.xsl)") + TQChar('\n');
  filter += i18n("*|All Files");
  m_URLRequester->setFilter(filter);

  l->addStretch(1);
  return m_widget;
}

#include "xsltimporter.moc"
