/***************************************************************************
    copyright            : (C) 2007 by Robby Stephenson
    email                : robby@periapsis.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of version 2 of the GNU General Public License as  *
 *   published by the Free Software Foundation;                            *
 *                                                                         *
 ***************************************************************************/

#include "griffithimporter.h"
#include "../collections/videocollection.h"
#include "tellicoimporter.h"
#include "../tellico_debug.h"

#include <kglobal.h>
#include <kstandarddirs.h>
#include <kprocess.h>

#include <tqdir.h>
#include <tqfile.h>

using Tellico::Import::GriffithImporter;

GriffithImporter::~GriffithImporter() {
  if(m_process) {
    m_process->kill();
    delete m_process;
    m_process = 0;
  }
}

Tellico::Data::CollPtr GriffithImporter::collection() {
  TQString filename = TQDir::homeDirPath() + TQString::fromLatin1("/.griffith/griffith.db");
  if(!TQFile::exists(filename)) {
    myWarning() << "GriffithImporter::collection() - database not found: " << filename << endl;
    return 0;
  }

  TQString python = KStandardDirs::findExe(TQString::fromLatin1("python"));
  if(python.isEmpty()) {
    myWarning() << "GriffithImporter::collection() - python not found!" << endl;
    return 0;
  }

  TQString griffith = KGlobal::dirs()->findResource("appdata", TQString::fromLatin1("griffith2tellico.py"));
  if(griffith.isEmpty()) {
    myWarning() << "GriffithImporter::collection() - griffith2tellico.py not found!" << endl;
    return 0;
  }

  m_process = new KProcess();
  connect(m_process, TQT_SIGNAL(receivedStdout(KProcess*, char*, int)), TQT_SLOT(slotData(KProcess*, char*, int)));
  connect(m_process, TQT_SIGNAL(receivedStderr(KProcess*, char*, int)), TQT_SLOT(slotError(KProcess*, char*, int)));
  connect(m_process, TQT_SIGNAL(processExited(KProcess*)), TQT_SLOT(slotProcessExited(KProcess*)));
  *m_process << python << griffith;
  if(!m_process->start(KProcess::Block, KProcess::AllOutput)) {
    myDebug() << "ExecExternalFetcher::startSearch() - process failed to start" << endl;
    return 0;
  }

  return m_coll;
}

void GriffithImporter::slotData(KProcess*, char* buffer_, int len_) {
  TQDataStream stream(m_data, IO_WriteOnly | IO_Append);
  stream.writeRawBytes(buffer_, len_);
}

void GriffithImporter::slotError(KProcess*, char* buffer_, int len_) {
  TQString msg = TQString::fromLocal8Bit(buffer_, len_);
  myDebug() << "GriffithImporter::slotError() - " << msg << endl;
  setStatusMessage(msg);
}


void GriffithImporter::slotProcessExited(KProcess*) {
//  myDebug() << "GriffithImporter::slotProcessExited()" << endl;
  if(!m_process->normalExit() || m_process->exitStatus()) {
    myDebug() << "GriffithImporter::slotProcessExited() - process did not exit successfully" << endl;
    return;
  }

  if(m_data.isEmpty()) {
    myDebug() << "GriffithImporter::slotProcessExited() - no data" << endl;
    return;
  }

  TQString text = TQString::fromUtf8(m_data, m_data.size());
  TellicoImporter imp(text);

  m_coll = imp.collection();
  if(!m_coll) {
    myDebug() << "GriffithImporter::slotProcessExited() - no collection pointer" << endl;
  } else {
    myLog() << "GriffithImporter::slotProcessExited() - results found: " << m_coll->entryCount() << endl;
  }
}

bool GriffithImporter::canImport(int type) const {
  return type == Data::Collection::Video;
}

#include "griffithimporter.moc"
