/***************************************************************************
    copyright            : (C) 2007 by Robby Stephenson
    email                : robby@periapsis.org
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of version 2 of the GNU General Public License as  *
 *   published by the Free Software Foundation;                            *
 *                                                                         *
 ***************************************************************************/

#include "drophandler.h"
#include "../mainwindow.h"
#include "../tellico_kernel.h"
#include "../tellico_debug.h"

#include <kurldrag.h>
#include <kmimetype.h>

using Tellico::DropHandler;

DropHandler::DropHandler(TQObject* parent_) : TQObject(parent_) {
}

DropHandler::~DropHandler() {
}

// assume the object is always the main window, that's the
// only object with this event filter
bool DropHandler::eventFilter(TQObject* obj_, TQEvent* ev_) {
  Q_UNUSED(obj_);
  if(ev_->type() == TQEvent::DragEnter) {
    return dragEnter(static_cast<TQDragEnterEvent*>(ev_));
  } else if(ev_->type() == TQEvent::Drop) {
    return drop(static_cast<TQDropEvent*>(ev_));
  }
  return false;
}

bool DropHandler::dragEnter(TQDragEnterEvent* event_) {
  bool accept = KURLDrag::canDecode(event_) || TQTextDrag::canDecode(event_);
  if(accept) {
    event_->accept(accept);
  }
  return accept;
}

bool DropHandler::drop(TQDropEvent* event_) {
  KURL::List urls;
  TQString text;

  if(KURLDrag::decode(event_, urls)) {
  } else if(TQTextDrag::decode(event_, text) && !text.isEmpty()) {
    urls << KURL(text);
  }
  return !urls.isEmpty() && handleURL(urls);
}

bool DropHandler::handleURL(const KURL::List& urls_) {
  bool hasUnknown = false;
  KURL::List tc, pdf, bib, ris;
  for(KURL::List::ConstIterator it = urls_.begin(); it != urls_.end(); ++it) {
    KMimeType::Ptr ptr = KMimeType::findByURL(*it);
    if(ptr->is(TQString::fromLatin1("application/x-tellico"))) {
      tc << *it;
    } else if(ptr->is(TQString::fromLatin1("application/pdf"))) {
      pdf << *it;
    } else if(ptr->is(TQString::fromLatin1("text/x-bibtex")) ||
              ptr->is(TQString::fromLatin1("application/x-bibtex"))) {
      bib << *it;
    } else if(ptr->is(TQString::fromLatin1("application/x-research-info-systems"))) {
      ris << *it;
    } else {
      myDebug() << "DropHandler::handleURL() - unrecognized type: " << ptr->name() << " (" << *it << ")" << endl;
      hasUnknown = true;
    }
  }
  MainWindow* mainWindow = ::tqqt_cast<MainWindow*>(Kernel::self()->widget());
  if(!mainWindow) {
    myDebug() << "DropHandler::handleURL() - no main window!" << endl;
    return !hasUnknown;
  }
  if(!tc.isEmpty()) {
    mainWindow->importFile(Import::TellicoXML, tc);
  }
  if(!pdf.isEmpty()) {
    mainWindow->importFile(Import::PDF, pdf);
  }
  if(!bib.isEmpty()) {
    mainWindow->importFile(Import::Bibtex, bib);
  }
  if(!ris.isEmpty()) {
    mainWindow->importFile(Import::RIS, ris);
  }
  // any unknown urls get passed
  return !hasUnknown;
}

#include "drophandler.moc"
