#ifndef _KVI_WINDOW_H_
#define _KVI_WINDOW_H_
//=============================================================================
//
//   File : kvi_window.h
//   Creation date : Tue Jul 6 1999 14:52:20 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2004 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
//
//=============================================================================

#include "kvi_string.h"
#include "kvi_qstring.h"
#include "kvi_qcstring.h"
#include "kvi_settings.h"

#ifdef COMPILE_ON_WINDOWS
	// The brain-damaged MSVC compiler can't instantiate TQList templates without a destructor definition
	#include "kvi_mdichild.h"
#else
	class KviMdiChild;
#endif

#include <tqframe.h>

class KviFrame;
class KviTaskBarItem;
class KviConfig;
class KviIrcView;
class KviInput;
class KviConsole;
class KviIrcContext;
class KviIrcConnection;
class KviWindowToolPageButton;

class KviTalPopupMenu;
class TQPixmap;
class TQSplitter;
class KviTalHBox;
class TQToolButton;
class TQTextCodec;
class KviTalWidgetStack;

#include "kvi_tal_hbox.h"
#include "kvi_tal_widgetstack.h"
#include "kvi_accel.h"

#ifdef COMPILE_CRYPT_SUPPORT
	class KviCryptController;
	class KviCryptSessionInfo;
#endif

#include <tqwidget.h>

//
// Window types
//
// There are KVI_WINDOW_NUM_TYPES predefined
// window types: these are used by the KVIrc core and distributed modules
// If you add a new def, increase KVI_WINDOW_NUM_TYPES
//

#define KVI_WINDOW_TYPE_CONSOLE 0
#define KVI_WINDOW_TYPE_CHANNEL 1
#define KVI_WINDOW_TYPE_QUERY 2
#define KVI_WINDOW_TYPE_HELP 3
#define KVI_WINDOW_TYPE_TERM 4
#define KVI_WINDOW_TYPE_EDITOR 5
#define KVI_WINDOW_TYPE_DCCCHAT 6
#define KVI_WINDOW_TYPE_DCCSEND 7
#define KVI_WINDOW_TYPE_SOCKETSPY 8
#define KVI_WINDOW_TYPE_LINKS 9
#define KVI_WINDOW_TYPE_TOOL 10
#define KVI_WINDOW_TYPE_GNUTELLA 11
#define KVI_WINDOW_TYPE_DIRBROWSER 12
#define KVI_WINDOW_TYPE_DCCCANVAS 13
#define KVI_WINDOW_TYPE_DCCVOICE 14
#define KVI_WINDOW_TYPE_LIST 15
#define KVI_WINDOW_TYPE_OFFER 16
#define KVI_WINDOW_TYPE_LOGVIEW 17
#define KVI_WINDOW_TYPE_DEADCHANNEL 18
#define KVI_WINDOW_TYPE_DEADQUERY 19
#define KVI_WINDOW_TYPE_SCRIPTEDITOR 20
#define KVI_WINDOW_TYPE_SCRIPTOBJECT 21
#define KVI_WINDOW_TYPE_USERWINDOW 22
#define KVI_WINDOW_TYPE_DEBUG 23


#define KVI_WINDOW_NUM_TYPES 24


#define KVI_WINDOW_TYPE_USER 10000




#define KVI_ACTIVITY_NONE 0
#define KVI_ACTIVITY_VERYLOW 1
#define KVI_ACTIVITY_LOW 2
#define KVI_ACTIVITY_MEDIUM 3
#define KVI_ACTIVITY_HIGH 4
#define KVI_ACTIVITY_VERYHIGH 5

#define KVI_ACTIVITY_ICE 0
#define KVI_ACTIVITY_VERYCOLD 1
#define KVI_ACTIVITY_COLD 2
#define KVI_ACTIVITY_UNDEFINED 3
#define KVI_ACTIVITY_HOT 4
#define KVI_ACTIVITY_VERYHOT 5
#define KVI_ACTIVITY_FIRE 6

#ifdef COMPILE_USE_QT4
    class TQPushButton;
	#define BUTTON_CLASS TQPushButton
#else
	#define BUTTON_CLASS TQToolButton
#endif


class KVIRC_API KviWindow : public TQWidget
{
	friend class KviInput;
	friend class KviFrame;
	friend class KviTaskBarItem;
	friend class KviTaskBarButton;
	friend class KviTreeTaskBarItem;
	friend class KviTreeTaskBar;
	TQ_PROPERTY(int KviProperty_ChildFocusOwner READ type)
	Q_OBJECT
  TQ_OBJECT
public:
	KviWindow(int type,KviFrame * lpFrm,const TQString &name,KviConsole * pConsole = 0);
	virtual ~KviWindow();
protected: // almost private: don't touch :D
	TQString                               m_szName;                  // the current window name (usually also the target)
	KviFrame                            * m_pFrm;
	KviConsole                          * m_pConsole;
	KviIrcContext                       * m_pContext;

	int                                   m_iType;
	
	KviTaskBarItem                      * m_pTaskBarItem;
	TQWidget                             * m_pFocusHandler;
	TQString                               m_szPlainTextCaption;
	TQString                               m_szHtmlActiveCaption;
	TQString                               m_szHtmlInactiveCaption;
	KviIrcView                          * m_pIrcView;
	KviInput                            * m_pInput;
	TQSplitter                           * m_pSplitter;
	KviTalHBox                               * m_pButtonBox;
	unsigned long int                     m_uId;
	TQString                               m_szTextEncoding;
#ifdef COMPILE_CRYPT_SUPPORT
	KviWindowToolPageButton             * m_pCryptControllerButton;
	KviCryptController                  * m_pCryptController;
	KviCryptSessionInfo                 * m_pCryptSessionInfo;
#endif
	BUTTON_CLASS                        * m_pTextEncodingButton;
	TQToolButton                         * m_pHideToolsButton;
	TQWidget                             * m_pLastFocusedChild;
	KviAccel                            * m_pAccel;
	static const char                   * m_typeTable[KVI_WINDOW_NUM_TYPES + 1];
	// text encoding and decoding
	//unsigned int                          m_uTextEncoding;
	TQTextCodec                          * m_pTextCodec;
//	KviToolWindowsContainer             * m_pEditorsContainer;
public:
	// The global ID of this window: unique in the application
	TQString id(){ return TQString("%1").arg(m_uId); };
	unsigned long int numericId(){ return m_uId; };
public:
	// THIS is the function that should be used
	const TQString & windowName(){ return m_szName; };
	void setWindowName(const TQString &szName);
	// force QT to set our UNICODE name too... FIXME: this should be removed later...
	virtual void setName(const char * szName);

	// Window type management
	int type() const { return m_iType; };
	// This returns a descriptive name of the window type
	// if the window is an user window, the typeString returned
	// by THIS implementation is "unknown"
	virtual const char * typeString();

	TQTextCodec * textCodec(){ return m_pTextCodec ? m_pTextCodec : defaultTextCodec(); };
	void forceTextCodec(TQTextCodec * c);

	// The frame that this window belongs to
	// It is always non-null and never changes
	KviFrame * frame() const { return m_pFrm; };
	// The KviIrcView of this window: may be NULL if the window has no KviIrcView (and thus supports no direct output)
	KviIrcView * view() const { return m_pIrcView; };
	// The mdiParent widget: may be nulll if the window is undocked
	KviMdiChild * mdiParent(){ return (KviMdiChild *)parent(); };
	// The console that this window belongs to: may be null for windows that aren't bound to irc contexts
	KviConsole * console(){ return m_pConsole; };
	// same as above
	KviIrcContext * context(){ return m_pContext; };
	// the current IRC connection (if any)
	KviIrcConnection * connection();
	// The splitter of this window: it *shouldn't* be null... but ... well.. who knows ? :D ...better check it
	TQSplitter * splitter(){ return m_pSplitter; };
	// The window has ALWAYS a taskbar item
	KviTaskBarItem * taskBarItem(){ return m_pTaskBarItem; };
	// The window *might* have a button container
	virtual TQFrame * buttonContainer(){ return (TQFrame*)m_pButtonBox; };
	virtual void toggleButtonContainer();
	// The window *might* have an output proxy: if it has no view() for example
	virtual KviWindow * outputProxy();
	// The window input widget
	KviInput * input(){ return m_pInput; };

	// The target of this window: empty when it makes no sense :D
	virtual const TQString & target(){ return KviTQString::empty; };
	// The local nickname bound to this window: might be empty when a local nickname makes no sense
	virtual const TQString & localNick(){ return KviTQString::empty; };

#ifdef COMPILE_CRYPT_SUPPORT
	KviCryptSessionInfo * cryptSessionInfo(){ return m_pCryptSessionInfo; };
	void setCryptSessionInfo(KviCryptSessionInfo * i);
#endif

	virtual bool activityMeter(unsigned int * puActivityValue,unsigned int * puActivityTemperature);

	void unhighlight();

	virtual void getTaskBarTipText(TQString &buffer);

	// This is meaningful only if view() is non NULL
	const TQString & lastLineOfText();
	const TQString & lastMessageText();

	const TQString &textEncoding(){ return m_szTextEncoding; };
	// returns true if the encoding could be succesfully set
	bool setTextEncoding(const TQString &szTextEncoding);
	// this must return a default text codec suitable for this window
	virtual TQTextCodec * defaultTextCodec();
	// encode the text from szSource by using m_uTextEncoding
	KviTQCString encodeText(const TQString &szText);
	TQString decodeText(const char * szText);

	void contextPopup();
	// Raises the window (after a light delay to prevent focus pingpongs)
	void delayedAutoRaise();
	// Window state: might work :D
	bool isMinimized();
	bool isMaximized();
	// Retrieves the default log file name: this is pre-build
	void getDefaultLogFileName(TQString &buffer);
	// Well...the external geometry :)
	TQRect externalGeometry();

	void delayedClose(); // close that jumps out of the current event loop

	// Interesting overridables:
	virtual void fillContextPopup(KviTalPopupMenu * p);
	virtual void getConfigGroupName(TQString &buf);
//	virtual void getBaseLogFileName(KviStr &buffer);
	virtual void getBaseLogFileName(TQString &buffer);
	virtual void updateCaption();
	virtual void applyOptions();
	virtual void updateIcon();
	virtual void ownMessage(const TQString &text){};
	virtual void ownAction(const TQString &text){};
	virtual const TQString & plainTextCaption(){ return m_szPlainTextCaption; };
	virtual const TQString & htmlActiveCaption(){ return m_szHtmlActiveCaption; };
	virtual const TQString & htmlInactiveCaption(){ return m_szHtmlInactiveCaption; };
	virtual void setFocus();

	void internalOutput(KviIrcView * pView,int msg_type,const kvi_wchar_t * text,int iFlags=0);
	// You *might* want to override these too.. but better don't touch them :D
	virtual void output(int msg_type,const char * format,...);
	virtual void outputNoFmt(int msg_type,const char * text,int iFlags=0);
	virtual void output(int msg_type,const kvi_wchar_t * format,...);
	virtual void outputNoFmt(int msg_type,const kvi_wchar_t * text,int iFlags=0){ internalOutput(m_pIrcView,msg_type,text,iFlags); };
	virtual void output(int msg_type,const TQString &szFmt,...);
	virtual void outputNoFmt(int msg_type,const TQString &szText,int iFlags=0); // <-- these are KviIrcView::AppendTextFlags
	// Just helpers.. FIXME: might be redesigned in some other way
	void updateBackgrounds(TQObject * obj = 0);

	void demandAttention();
	bool hasAttention();

	// This should die, probably
	void listWindowTypes();

public slots:
	void dock();
	void undock();
	void autoRaise();
	void maximize();
	void minimize();
	void restore();
	void reloadImages();
protected slots:
	void savePropertiesAsDefault();
	void toggleCryptController();     // This has to be here even if the crypt support is enabled...moc does not support conditionals
	void cryptControllerFinished();   // same as above
	void cryptSessionInfoDestroyed(); //  same as above
	void textEncodingButtonClicked();
	void systemPopupRequest(const TQPoint &pnt);
	void systemTextEncodingPopupSmartActivated(int id);
	void systemTextEncodingPopupSmartUtf8Activated(int id);
	void systemTextEncodingPopupStandardActivated(int id);
	void systemTextEncodingPopupDefault();
	void childDestroyed();
signals:
	void windowNameChanged();
protected:
	// Loading and saving of properties
	// Protected: only KviFrame can call these
	virtual void saveProperties(KviConfig *cfg);
	virtual void loadProperties(KviConfig *cfg);
	// Creation and destruction events: overridden in windows that have script events bound to creation and destruction
	virtual void triggerCreationEvents(){};
	virtual void triggerDestructionEvents(){};
	// Internal: do not touch :D (KviFrame)
	virtual void createTaskBarItem();
	virtual void destroyTaskBarItem();
	// called by KviFrame
	// either lost the active window status or the frame is no longer active (but we're still the active kvirc's subwindow)
	virtual void lostUserFocus();
	// Sets the progress for the taskbar item: if "progress" makes sense in your window , well , use this
	void setProgress(int progress);
	// call this in the constructor if your caption is fixed:
	// it will set m_szPlainTextCaption to szCaption and it will
	// automatically use it without the need of overriding fillCaptionBuffers
	void setFixedCaption(const TQString &szCaption);
	// this by default calls fillSingleColorCaptionBuffer(plainTextCaption());
	virtual void fillCaptionBuffers();
	// protected helper
	void fillSingleColorCaptionBuffers(const TQString &szName);
	// Virtual events that signal dock state change
	virtual void youAreDocked();
	virtual void youAreUndocked();
	// Reimplement to show a special icon in the taskbar items and captions
	virtual TQPixmap * myIconPtr();
	// Sets the type of this window: be careful with this
	void setType(int iType);

	bool eventFilter(TQObject *o,TQEvent *e);

	// Virtuals overridden to manage the internal layouts...
	virtual void moveEvent(TQMoveEvent *e);
	virtual void closeEvent(TQCloseEvent *e);
	virtual void wheelEvent(TQWheelEvent *e);
	virtual void childEvent(TQChildEvent *e);
	virtual void focusInEvent(TQFocusEvent *e);

	void childInserted(TQWidget * o);
	void childRemoved(TQWidget * o);

	void activateSelf();

	// Internal helpers
	void createCryptControllerButton(TQWidget * par);
	void createTextEncodingButton(TQWidget * par);
	void createSystemTextEncodingPopup();

	BUTTON_CLASS * createToolButton(TQWidget * par,const char * nam,int pixon,int pixoff,const TQString & tooltip,bool bOn);
	// This is called by KviInput: actually it links the widgetAdded
	virtual void childrenTreeChanged(TQWidget * widgetAdded);
	
	virtual bool focusNextPrevChild(bool bNext);
	
	virtual void preprocessMessage(TQString & szMessage);
};

#ifndef _KVI_WINDOW_CPP_
	// The active window:
	//   This is almost always non null
	//   The exception is the startup (when there are no windows at all)
	//   and the last phase of the destructor.
	//   You usually shouldn't care of checking this pointer for NULL unless
	//   you're running very early at startup or very late at shutdown
	extern KVIRC_API KviWindow * g_pActiveWindow;
#endif

#endif //_KVI_WINDOW_H_
