/***************************************************************************
 *   Copyright (C) 2005 by David Saxton                                    *
 *   david@bluehaze.org                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 ***************************************************************************/

#include "libraryitem.h"
#include "node.h"
#include "resistance.h"
#include "resistordip.h"

#include <kiconloader.h>
#include <klocale.h>
#include <tqpainter.h>

Item* ResistorDIP::construct( ItemDocument *itemDocument, bool newItem, const char *id )
{
	return new ResistorDIP( (ICNDocument*)itemDocument, newItem, id );
}

LibraryItem* ResistorDIP::libraryItem()
{
	return new LibraryItem(
		"ec/resistordip",
		i18n("Resistor DIP"),
		i18n("Discrete"),
		"resistordip.png",
		LibraryItem::lit_component,
		ResistorDIP::construct
			);
}

ResistorDIP::ResistorDIP( ICNDocument *icnDocument, bool newItem, const char *id )
	: Component( icnDocument, newItem, id ? id : "multiplexer" )
{
	m_name = i18n("Resistor DIP");
	m_desc = i18n("Set of resistors with identical values in a Dual Inline Package.");
	
	m_resistorCount = 0;
	for ( int i=0; i<maxCount; ++i )
		m_resistance[i] = 0l;
	
	createProperty( "resistance", Variant::Type::Double );
	property("resistance")->setCaption( i18n("Resistance") );
	property("resistance")->setUnit( TQChar(0x3a9) );
	property("resistance")->setValue(1e4);
	property("resistance")->setMinValue(1e-6);
	
	createProperty( "count", Variant::Type::Int );
	property("count")->setCaption( i18n("Count") );
	property("count")->setMinValue(2);
	property("count")->setMaxValue(maxCount);
	property("count")->setValue(8);
}

ResistorDIP::~ResistorDIP()
{
}


void ResistorDIP::dataChanged()
{
	initPins();
	const double resistance = dataDouble("resistance");
	for ( int i=0; i<m_resistorCount; ++i )
		m_resistance[i]->setResistance(resistance);
	
	const TQString display = TQString::number( resistance / getMultiplier(resistance), 'g', 3 ) + getNumberMag(resistance) + TQChar(0x3a9);
	addDisplayText( "res", TQRect( offsetX(), offsetY()-16, 32, 12 ), display );
}


void ResistorDIP::initPins()
{
	const int count = dataInt("count");
	const double resistance = dataDouble("resistance");
	
	if ( count == m_resistorCount )
		return;
	
	if ( count < m_resistorCount )
	{
		for ( int i=count; i<m_resistorCount; ++i )
		{
			removeElement( m_resistance[i], false );
			m_resistance[i] = 0l;
			removeNode( "n"+TQString::number(i) );
			removeNode( "p"+TQString::number(i) );
		}
	}
	else
	{
		for ( int i=m_resistorCount; i<count; ++i )
		{
			const TQString nid = "n"+TQString::number(i);
			const TQString pid = "p"+TQString::number(i);
			m_resistance[i] = createResistance( createPin( -24, 0, 0, nid ), createPin( 24, 0, 180, pid ), resistance );
		}
	}
	m_resistorCount = count;
	
	setSize( -16, -count*8, 32, count*16, true );
	updateDIPNodePositions();
}


void ResistorDIP::updateDIPNodePositions()
{
	for ( int i=0; i<m_resistorCount; ++i )
	{
		m_nodeMap["n"+TQString::number(i)].y = offsetY() + 8 + 16*i;
		m_nodeMap["p"+TQString::number(i)].y = offsetY() + 8 + 16*i;
	}
	updateAttachedPositioning();
}


void ResistorDIP::drawShape( TQPainter &p )
{
	int _x = int(x()+offsetX());
	int _y = int(y()+offsetY());
	
	initPainter(p);
	for ( int i=0; i<m_resistorCount; ++i )
		p.drawRect( _x, _y+16*i+2, 32, 12 );
	deinitPainter(p);
}

