/*
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
*/

/*
  This class gives function that interact with kaddressbook.
  begin:     Fri 2004-07-23
  copyright: (C) 2004 by John Tapsell
  email:     john@geola.co.uk
*/

#include "addressbook.h"
#include "../viewcontainer.h"
#include "../konversationmainwindow.h"
#include "../server.h"
#include "../channel.h"
#include "../konversationapplication.h"

#include <tqstringlist.h>
#include "tqwidget.h"

#include <klocale.h>
#include <kmessagebox.h>
#include <kapplication.h>
#include <dcopclient.h>
#include <kwin.h>


namespace Konversation
{

    Addressbook *Addressbook::m_instance=0L;

    Addressbook::Addressbook() : DCOPObject( "KIMIface")
    {
        addressBook = KABC::StdAddressBook::self(true);
        m_ticket=NULL;
    }
    Addressbook::~Addressbook()
    {
        if (m_instance == this)
            sd.setObject(m_instance, 0, false);
    }

    Addressbook *Addressbook::self()
    {
        if (!m_instance) { sd.setObject(m_instance, new Addressbook()); }
        return m_instance;
    }

    TQStringList Addressbook::allContacts()
    {
        TQStringList contactUIDS;
        for( KABC::AddressBook::Iterator it = addressBook->begin(); it != addressBook->end(); ++it )
            if(hasAnyNicks(*it)) contactUIDS.append((*it).uid());
        return contactUIDS;
    }
    //Produces a string list of all the irc nicks that are known.
    TQStringList Addressbook::allContactsNicks()
    {
        TQStringList contacts;
        for( KABC::AddressBook::Iterator it = addressBook->begin(); it != addressBook->end(); ++it )
            contacts += TQStringList::split( TQChar( 0xE000 ), (*it).custom("messaging/irc", "All") );
        return contacts;
    }

    TQStringList Addressbook::onlineContacts()
    {
        TQStringList contactUIDS;
        for( KABC::AddressBook::Iterator it = addressBook->begin(); it != addressBook->end(); ++it )
            if(isOnline(*it)) contactUIDS.append((*it).uid());

        return contactUIDS;
    }
    TQStringList Addressbook::reachableContacts()
    {
        return onlineContacts();
    }
    TQStringList Addressbook::fileTransferContacts()
    {
        return onlineContacts();
    }
    bool Addressbook::isPresent(const TQString &uid)
    {
        return hasAnyNicks(addressBook->findByUid(uid));
    }
    TQString Addressbook::displayName(const TQString &uid)
    {
        return getBestNick(addressBook->findByUid(uid));
    }
    TQString Addressbook::presenceString(const TQString &uid)
    {
        if(uid.isEmpty())
        {
            kdDebug() << "Addressbook::presenceString() called with an empty uid" << endl;
            return TQString("Error");
        }
        switch( presenceStatus(uid))
        {
            case 0:
                return "";
            case 1:
                return i18n("Offline");
            case 2:
                return i18n("Connecting");        //Shouldn't happen - not supported.
            case 3:
                return i18n("Away");
            case 4:
                return i18n("Online");
        }
        return TQString("Error");
    }
    int Addressbook::presenceStatus(const TQString &uid)
    {
        return presenceStatusByAddressee(addressBook->findByUid(uid));
    }

    bool Addressbook::canReceiveFiles(const TQString &uid)
    {
        if(uid.isEmpty())
        {
            kdDebug() << "Addressbook::canReceiveFiles() called with empty uid" << endl;
            return false;
        }
        int presence = presenceStatus(uid);

        return (presence == 4) || (presence == 3);
    }
    bool Addressbook::canRespond(const TQString &uid)
    {
        if(uid.isEmpty())
        {
            kdDebug() << "Addressbook::canRespond called with empty uid" << endl;
            return false;
        }
        //this should return false if they are offline.
        int result = presenceStatus(uid);
        if(result == 3 || result == 4) return true;
        return false;
    }
    TQString Addressbook::locate(const TQString &contactId, const TQString &protocol)
    {
        if(contactId.isEmpty())
        {
            kdDebug() << "Addressbook::locate called with empty contactId" << endl;
            return TQString();
        }
        if(protocol != "messaging/irc")
            return TQString();

        return getKABCAddresseeFromNick(contactId).uid();
    }
    TQPixmap Addressbook::icon(const TQString &uid)
    {

        Images* icons = KonversationApplication::instance()->images();
        TQIconSet currentIcon;
        if(!isPresent(uid))
            return TQPixmap();

        switch(presenceStatus(uid))
        {
            case 0:                               //Unknown
            case 1:                               //Offline
            case 2:                               //connecting - invalid for us?
                currentIcon = icons->getKimproxyOffline();
                break;
            case 3:                               //Away
                currentIcon = icons->getKimproxyAway();
                break;
            case 4:                               //Online
                currentIcon = icons->getKimproxyOnline();
                break;
            default:
                //error
                kdDebug() << "Unknown status " << uid << endl;
                return TQPixmap();
        }

        TQPixmap joinedIcon = currentIcon.pixmap(TQIconSet::Automatic, TQIconSet::Active, TQIconSet::On);
        return joinedIcon;
    }
    TQString Addressbook::context(const TQString &uid)
    {
        if(uid.isEmpty())
        {
            kdDebug() << "Addressbook::contact called with empty uid" << endl;
            return TQString();
        }
        TQString context;
        return context;
    }
    TQStringList Addressbook::protocols()
    {
        TQStringList protocols;
        protocols.append("messaging/irc");
        return protocols;
    }

    // ACTORS
    /**
     * Send a single message to the specified addressee
     * Any response will be handled by the IM client as a normal
     * conversation.
     * @param uid the KABC uid you want to chat with.
     * @param message the message to send them.
     */
    void Addressbook::messageContact( const TQString &uid, const TQString& message )
    {
        if(uid.isEmpty())
        {
            focusAndShowErrorMessage(i18n("Another KDE application tried to use Konversation for instant messaging, but did not specify any contact to send the message to.  This is probably a bug in the other application."));
            return;
        }
        KABC::Addressee addressee = addressBook->findByUid(uid);
        if(addressee.isEmpty())
        {
            focusAndShowErrorMessage(i18n("Another KDE application tried to use Konversation for instant messaging, but Konversation could not find the specified contact in the KDE address book."));
            return;
        }
        NickInfoPtr nickInfo = getNickInfo(addressee);
        if(!nickInfo)
        {
            TQString user = addressee.fullEmail();
            if(!user.isEmpty()) user = " (" + user + ')';
            focusAndShowErrorMessage(i18n("Another KDE application tried to use Konversation for instant messaging, but the requested user%1 is not online.").arg(user));
            return;
        }

        nickInfo->getServer()->dcopSay(nickInfo->getNickname(), message);
    }

    /**
     * Open a chat to a contact, and optionally set some initial text
     */
    void Addressbook::messageNewContact( const TQString &contactId, const TQString &/*protocol*/ ) {
    if(contactId.isEmpty() )
    {
        kdDebug() << "Addressbook::messageNewContact called with empty contactid" << endl;
        focusAndShowErrorMessage(i18n("Another KDE application tried to use Konversation for instant messaging, but did not specify any contact to send the message to.  This is probably a bug in the other application."));
        return;
    }
    messageContact(contactId, TQString());
}

/**
 * Start a chat session with the specified addressee
 * @param uid the KABC uid you want to chat with.
 */
void Addressbook::chatWithContact( const TQString &uid )
{
    if(uid.isEmpty())
    {
        kdDebug() << "Addressbook::chatWithContact called with empty uid" << endl;
        focusAndShowErrorMessage(i18n("Another KDE application tried to use Konversation for instant messaging, but did not specify any contact to send the message to.  This is probably a bug in the other application."));
        return;
    }
    messageContact(uid, TQString());
}

/**
 * Send the file to the contact
 * @param uid the KABC uid you are sending to.
 * @param sourceURL a KURL to send.
 * @param altFileName an alternate filename describing the file
 * @param fileSize file size in bytes
 */
void Addressbook::sendFile(const TQString &uid, const KURL &sourceURL, const TQString &altFileName, uint fileSize)
{
    if(uid.isEmpty())
    {
        focusAndShowErrorMessage(i18n("Another KDE application tried to use Konversation to send a file to a contact, but did not specify any contact to send the file to.  This is probably a bug in the other application."));
        return;
    }
    KABC::Addressee addressee = addressBook->findByUid(uid);
    if(addressee.isEmpty())
    {
        focusAndShowErrorMessage(i18n("Another KDE application tried to use Konversation to send a file to a contact, but Konversation could not find the specified contact in the KDE address book."));
        return;
    }
    NickInfoPtr nickInfo = getNickInfo(addressee);
    if(!nickInfo)
    {
        TQString user = addressee.fullEmail();
        if(!user.isEmpty()) user = " (" + user + ')';
        focusAndShowErrorMessage(i18n("Another KDE application tried to use Konversation to send a file to a contact, but the requested user%1 is not currently online.").arg(user));
        return;
    }
    nickInfo->getServer()->addDccSend(nickInfo->getNickname(), sourceURL, altFileName, fileSize);
    TQWidget *widget = nickInfo->getServer()->getViewContainer()->getWindow();
    KWin::demandAttention(widget->winId());       //If activeWindow request is denied, at least demand attention!
    KWin::activateWindow(widget->winId());        //May or may not work, depending on focus stealing prevention.

}

// MUTATORS
// Contact list
/**
 * Add a contact to the contact list
 * @param contactId the protocol specific identifier for the contact, eg UIN for ICQ, screenname for AIM, nick for IRC.
 * @param protocolId the protocol, eg one of "AIMProtocol", "MSNProtocol", "ICTQProtocol", ...
 * @return whether the add succeeded.  False may signal already present, protocol not supported, or add operation not supported.
 */
bool Addressbook::addContact( const TQString &/*contactId*/, const TQString &/*protocolId*/ ) {
focusAndShowErrorMessage(i18n("Another KDE application tried to use Konversation to add a contact.  Konversation does support this."));
return false;
//Nicks are auto added if they are put in the addressbook - I don' think there is anything useful I can do.
}

void Addressbook::emitContactPresenceChanged(const TQString &uid, int presence)
{
    if(uid.isEmpty())
    {
        //This warning below is annoying.  FIXME - disabled because it's too verbose
        //		kdDebug() << "Addressbook::emitContactPresenceChanged was called with empty uid" << endl;
        return;
    }
    Q_ASSERT(kapp->dcopClient());
    emit contactPresenceChanged(uid, kapp->dcopClient()->appId(), presence);
    //	kdDebug() << "Presence changed for uid " << uid << " to " << presence << endl;
}

void Addressbook::emitContactPresenceChanged(const TQString &uid)
{
    if(uid.isEmpty())
    {
        kdDebug() << "Addressbook::emitContactPresenceChanged was called with empty uid" << endl;
        return;
    };

    emitContactPresenceChanged(uid, presenceStatus(uid));
}

}                                                 //NAMESPACE

#include "addressbook.moc"
