/* This file is part of the KDE project
   Copyright (C)  2001 Montel Laurent <lmontel@mandrakesoft.com>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include <kapplication.h>
#include <klocale.h>

#include <tqlayout.h>
#include <tqvbox.h>
#include <kdebug.h>
#include <tqlabel.h>
#include <tqcombobox.h>

#include <klineedit.h>
#include <kurlrequester.h>
#include <kseparator.h>
#include <kiconloader.h>
#include "KoInsertLink.h"
#include <kdesktopfile.h>
#include <krecentdocument.h>

using namespace KOfficePrivate;

KoInsertLinkDia::KoInsertLinkDia( TQWidget *parent, const char *name, bool displayBookmarkLink )
    : KDialogBase( KDialogBase::IconList, i18n("Insert Link"),
		   KDialogBase::Ok | KDialogBase::Cancel,
		   KDialogBase::Ok, parent, name )
{
  bookmarkLink = 0L;
  TQVBox *page=addVBoxPage(i18n("Internet"), TQString(),BarIcon("html",KIcon::SizeMedium));
  internetLink = new  internetLinkPage(page );
  connect(internetLink,TQT_SIGNAL(textChanged()),this,TQT_SLOT(slotTextChanged (  )));

  page=addVBoxPage(i18n("Mail & News"), TQString(),BarIcon("mail_generic",KIcon::SizeMedium));
  mailLink = new  mailLinkPage(page );
  connect(mailLink,TQT_SIGNAL(textChanged()),this,TQT_SLOT(slotTextChanged ()));

  page=addVBoxPage(i18n("File"), TQString(),BarIcon("filenew",KIcon::SizeMedium));
  fileLink = new  fileLinkPage(page );
  connect(fileLink,TQT_SIGNAL(textChanged()),this,TQT_SLOT(slotTextChanged ()));

  if ( displayBookmarkLink)
  {
      page=addVBoxPage(i18n("Bookmark"), TQString(),BarIcon("bookmark",KIcon::SizeMedium));
      bookmarkLink = new  bookmarkLinkPage(page );
      connect(bookmarkLink,TQT_SIGNAL(textChanged()),this,TQT_SLOT(slotTextChanged ()));
  }

  connect( this, TQT_SIGNAL( aboutToShowPage(TQWidget *) ), this, TQT_SLOT( tabChanged(TQWidget *) ) );

  slotTextChanged ( );
  resize(400,300);
}

void KoInsertLinkDia::tabChanged(TQWidget *)
{
    switch( activePageIndex() )
    {
    case 0:
      internetLink->setLinkName( currentText );
      break;
    case 1:
      mailLink->setLinkName( currentText );
      break;
    case 2:
      fileLink->setLinkName( currentText );
      break;
    case 3:
    {
        if ( bookmarkLink)
            bookmarkLink->setLinkName( currentText );
    }
    break;
    default:
      kdDebug()<<"Error in linkName\n";
    }
    enableButtonOK( !(linkName().isEmpty()  || hrefName().isEmpty()) );
}

void KoInsertLinkDia::slotTextChanged ( )
{
    enableButtonOK( !(linkName().isEmpty()  || hrefName().isEmpty()));
    currentText = linkName();
}

bool KoInsertLinkDia::createLinkDia(TQString & _linkName, TQString & _hrefName, const TQStringList& bkmlist, bool displayBookmarkLink, TQWidget* parent, const char* name)
{
    bool res = false;

    KoInsertLinkDia *dlg = new KoInsertLinkDia( parent, name, displayBookmarkLink );
    dlg->setHrefLinkName(_hrefName,_linkName, bkmlist);
    if ( dlg->exec() == Accepted )
    {
        _linkName = dlg->linkName();
        _hrefName = dlg->hrefName();
        res = true;
    }
    delete dlg;

    return res;
}

void KoInsertLinkDia::setHrefLinkName(const TQString &_href, const TQString &_link, const TQStringList & bkmlist)
{
    if ( bookmarkLink)
        bookmarkLink->setBookmarkList(bkmlist);
    if ( _href.isEmpty())
    {
        if ( !_link.isEmpty() )
        {
            internetLink->setLinkName(_link);
            showPage(0);
            slotTextChanged ( );
        }
        return;
    }
    if(_href.find("http://")!=-1 || _href.find("https://")!=-1 ||_href.find("ftp://")!=-1 )
    {
        internetLink->setHrefName(_href);
        internetLink->setLinkName(_link);
        showPage(0);
    }
    else if(_href.find("file:/")!=-1)
    {
        fileLink->setHrefName(_href);
        fileLink->setLinkName(_link);
        showPage(2);
    }
    else if(_href.find("mailto:")!=-1 || _href.find("news:")!=-1)
    {
        mailLink->setHrefName(_href);
        mailLink->setLinkName(_link);
        showPage(1);
    }
    else if(_href.find("bkm://")!=-1)
    {
        if ( bookmarkLink )
        {
            bookmarkLink->setHrefName(_href.mid(6));
            bookmarkLink->setLinkName(_link);
            showPage(3);
        }
    }
    slotTextChanged ( );
}

TQString KoInsertLinkDia::linkName() const
{
    TQString result;
    switch(activePageIndex())
    {
    case 0:
      result=internetLink->linkName();
      break;
    case 1:
      result=mailLink->linkName();
      break;
    case 2:
      result=fileLink->linkName();
      break;
    case 3:
    {
        if ( bookmarkLink)
            result=bookmarkLink->linkName();
    }
    break;
    default:
      kdDebug()<<"Error in linkName\n";
    }
  return result;
}

TQString KoInsertLinkDia::hrefName() const
{
    TQString result;
    switch(activePageIndex())
    {
    case 0:
      result=internetLink->hrefName();
      break;
    case 1:
      result=mailLink->hrefName();
      break;
    case 2:
      result=fileLink->hrefName();
      break;
    case 3:
    {
        if ( bookmarkLink )
            result=bookmarkLink->hrefName();
    }
    break;
    default:
      kdDebug()<<"Error in hrefName\n";
    }
  return result;
}

void KoInsertLinkDia::slotOk()
{
    KDialogBase::slotOk();
}


internetLinkPage::internetLinkPage( TQWidget *parent , char *name  )
  : TQWidget(parent,name)
{
  TQVBoxLayout *lay1 = new TQVBoxLayout( this );
  lay1->setSpacing( KDialog::spacingHint() );
  TQVBoxLayout *lay2 = new TQVBoxLayout( lay1);
  lay2->setSpacing( KDialog::spacingHint() );

  TQLabel* tmpTQLabel = new TQLabel( this);

  lay2->addWidget(tmpTQLabel);
  tmpTQLabel->setText(i18n("Text to display:"));

  m_linkName = new TQLineEdit( this );
  lay2->addWidget(m_linkName);

  tmpTQLabel = new TQLabel( this);
  lay2->addWidget(tmpTQLabel);

  tmpTQLabel->setText(i18n("Internet address:"));
  m_hrefName = new TQLineEdit( this );

  lay2->addWidget(m_hrefName);

  lay2->addStretch( 1 );
  
  m_linkName->setFocus();

  connect(m_linkName,TQT_SIGNAL(textChanged ( const TQString & )),this,TQT_SLOT(textChanged ( const TQString & )));
  connect(m_hrefName,TQT_SIGNAL(textChanged ( const TQString & )),this,TQT_SLOT(textChanged ( const TQString & )));
  KSeparator* bar1 = new KSeparator( KSeparator::HLine, this);
  bar1->setFixedHeight( 10 );
  lay2->addWidget( bar1 );
}

TQString internetLinkPage::createInternetLink()
{
    TQString result=m_hrefName->text();

    if(result.isEmpty())
        return result;

    if(result.find("http://")==-1 && result.find("https://")==-1 && result.find("ftp://")==-1)
        result = "http://"+result;
    return result;
}


void internetLinkPage::setLinkName(const TQString & _name)
{
    m_linkName->setText(_name);
}

void internetLinkPage::setHrefName(const TQString &_name)
{
    m_hrefName->setText(_name);
}

TQString internetLinkPage::linkName()const
{
  return m_linkName->text();
}

TQString internetLinkPage::hrefName()
{
  return createInternetLink();
}

void internetLinkPage::textChanged ( const TQString & )
{
    emit textChanged();
}

bookmarkLinkPage::bookmarkLinkPage( TQWidget *parent , char *name  )
  : TQWidget(parent,name)
{
  TQVBoxLayout *lay1 = new TQVBoxLayout( this );
  lay1->setSpacing( KDialog::spacingHint() );
  TQVBoxLayout *lay2 = new TQVBoxLayout( lay1);
  lay2->setSpacing( KDialog::spacingHint() );

  TQLabel* tmpTQLabel = new TQLabel( this);

  lay2->addWidget(tmpTQLabel);
  tmpTQLabel->setText(i18n("Text to display:"));

  m_linkName = new TQLineEdit( this );
  lay2->addWidget(m_linkName);

  tmpTQLabel = new TQLabel( this);
  lay2->addWidget(tmpTQLabel);

  tmpTQLabel->setText(i18n("Bookmark name:"));
  m_hrefName = new TQComboBox( this );

  lay2->addWidget(m_hrefName);

  lay2->addStretch( 1 );
  
  m_linkName->setFocus();

  connect(m_linkName,TQT_SIGNAL(textChanged ( const TQString & )),this,TQT_SLOT(textChanged ( const TQString & )));
  connect(m_hrefName,TQT_SIGNAL(textChanged ( const TQString & )),this,TQT_SLOT(textChanged ( const TQString & )));
  KSeparator* bar1 = new KSeparator( KSeparator::HLine, this);
  bar1->setFixedHeight( 10 );
  lay2->addWidget( bar1 );
}

TQString bookmarkLinkPage::createBookmarkLink()
{
    TQString result=m_hrefName->currentText();

    if(result.isEmpty())
        return result;

    if(result.find("bkm://")==-1)
        result = "bkm://"+result;
    return result;
}


void bookmarkLinkPage::setLinkName(const TQString & _name)
{
    m_linkName->setText(_name);
}

void bookmarkLinkPage::setHrefName(const TQString &_name)
{
    m_hrefName->setCurrentText(_name);
}

void bookmarkLinkPage::setBookmarkList(const TQStringList & bkmlist)
{
    m_hrefName->clear();
    m_hrefName->insertStringList(bkmlist, 0);
    if ( bkmlist.isEmpty())
        m_linkName->setEnabled( false);
    //m_hrefName->setEditable(true);
}

TQString bookmarkLinkPage::linkName()const
{
  return m_linkName->text();
}

TQString bookmarkLinkPage::hrefName()
{
  return createBookmarkLink();
}

void bookmarkLinkPage::textChanged ( const TQString & )
{
    emit textChanged();
}

mailLinkPage::mailLinkPage( TQWidget *parent , char *name  )
  : TQWidget(parent,name)
{
  TQVBoxLayout *lay1 = new TQVBoxLayout( this );
  lay1->setSpacing( KDialog::spacingHint() );
  TQVBoxLayout *lay2 = new TQVBoxLayout( lay1);
  lay2->setSpacing( KDialog::spacingHint() );

  TQLabel* tmpTQLabel = new TQLabel( this);

  lay2->addWidget(tmpTQLabel);
  tmpTQLabel->setText(i18n("Text to display:"));

  m_linkName = new TQLineEdit( this );
  lay2->addWidget(m_linkName);

  tmpTQLabel = new TQLabel( this);
  lay2->addWidget(tmpTQLabel);

  tmpTQLabel->setText(i18n("Target:"));
  m_hrefName = new TQLineEdit( this );

  lay2->addWidget(m_hrefName);
  lay2->addStretch( 1 );
  
  connect(m_linkName,TQT_SIGNAL(textChanged ( const TQString & )),this,TQT_SLOT(textChanged ( const TQString & )));
  connect(m_hrefName,TQT_SIGNAL(textChanged ( const TQString & )),this,TQT_SLOT(textChanged ( const TQString & )));
  KSeparator* bar1 = new KSeparator( KSeparator::HLine, this);
  bar1->setFixedHeight( 10 );
  lay2->addWidget( bar1 );
}

TQString mailLinkPage::createMailLink()
{
    TQString result=m_hrefName->text();

    if(result.isEmpty())
        return result;

    if(result.find("mailto:")==-1 && result.find("news:")==-1)
        result = "mailto:"+result;
    return result;
}


void mailLinkPage::setLinkName(const TQString & _name)
{
    m_linkName->setText(_name);
}

void mailLinkPage::setHrefName(const TQString &_name)
{
    m_hrefName->setText(_name);
}

TQString mailLinkPage::linkName()const
{
  return m_linkName->text();
}

TQString mailLinkPage::hrefName()
{
  return createMailLink();
}

void mailLinkPage::textChanged ( const TQString & )
{
    emit textChanged();
}

fileLinkPage::fileLinkPage( TQWidget *parent , char *name  )
  : TQWidget(parent,name)
{
  TQVBoxLayout *lay1 = new TQVBoxLayout( this );
  lay1->setSpacing( KDialog::spacingHint() );
  TQVBoxLayout *lay2 = new TQVBoxLayout( lay1);
  lay2->setSpacing( KDialog::spacingHint() );

  TQLabel* tmpTQLabel = new TQLabel( this);

  lay2->addWidget(tmpTQLabel);
  tmpTQLabel->setText(i18n("Text to display:"));

  m_linkName = new TQLineEdit( this );
  lay2->addWidget(m_linkName);

  tmpTQLabel = new TQLabel( this);
  lay2->addWidget(tmpTQLabel);
  tmpTQLabel->setText(i18n("Recent file:"));

  TQComboBox * recentFile = new TQComboBox( this );
  recentFile->setMaximumWidth( kapp->desktop()->width()*3/4 );
  lay2->addWidget(recentFile);

  TQStringList fileList = KRecentDocument::recentDocuments();
  TQStringList lst;
  lst <<"";
  for (TQStringList::ConstIterator it = fileList.begin();it != fileList.end(); ++it)
  {
      KDesktopFile f(*it, true /* read only */);
      if ( !f.readURL().isEmpty())
          lst.append( f.readURL());
  }
  if ( lst.count()<= 1 )
  {
      recentFile->clear();
      recentFile->insertItem( i18n("No Entries") );
      recentFile->setEnabled( false );
  }
  else
      recentFile->insertStringList( lst);
  
  recentFile->setSizePolicy( TQSizePolicy::Expanding, TQSizePolicy::Fixed );
  
  connect( recentFile , TQT_SIGNAL(highlighted ( const TQString &)), this,  TQT_SLOT( slotSelectRecentFile( const TQString & )));

  tmpTQLabel = new TQLabel( this);
  lay2->addWidget(tmpTQLabel);

  tmpTQLabel->setText(i18n("File location:"));
  m_hrefName = new KURLRequester( this );

  lay2->addWidget(m_hrefName);
  lay2->addStretch( 1 );

  connect(m_linkName,TQT_SIGNAL(textChanged ( const TQString & )),this,TQT_SLOT(textChanged ( const TQString & )));
  connect(m_hrefName,TQT_SIGNAL(textChanged ( const TQString & )),this,TQT_SLOT(textChanged ( const TQString & )));

  KSeparator* bar1 = new KSeparator( KSeparator::HLine, this);
  bar1->setFixedHeight( 10 );
  lay2->addWidget( bar1 );
}

void fileLinkPage::slotSelectRecentFile( const TQString &_file )
{
    m_hrefName->lineEdit()->setText(_file );
}

TQString fileLinkPage::createFileLink()
{
    TQString result=m_hrefName->lineEdit()->text();
    if(result.isEmpty())
        return result;

    if(result.find("file:/")==-1)
        result = "file://"+result;
    return result;
}

void fileLinkPage::setLinkName(const TQString & _name)
{
    m_linkName->setText(_name);
}

void fileLinkPage::setHrefName(const TQString &_name)
{
    m_hrefName->lineEdit()->setText(_name);
}

TQString fileLinkPage::linkName()const
{
  return m_linkName->text();
}

TQString fileLinkPage::hrefName()
{
  return createFileLink();
}

void fileLinkPage::textChanged ( const TQString & )
{
    emit textChanged();
}

#include "KoInsertLink.moc"
