/* This file is part of the KDE project
   Copyright (C) 1998, 1999, 2000 Torben Weis <weis@kde.org>
   Copyright (C) 2004 David Faure <faure@kde.org>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#ifndef KO_DOCUMENT_INFO_H
#define KO_DOCUMENT_INFO_H

#include <tqobject.h>
#include <tqmap.h>
#include <kconfig.h>
#include <koffice_export.h>

class TQString;
class TQStringList;
class TQDomDocument;
class TQDomElement;
class TQDomNode;
class TQDateTime;
class KoStore;
class KoDocumentInfoPage;
class KoXmlWriter;

class KOFFICECORE_EXPORT KoDocumentInfo : public TQObject
{
    Q_OBJECT
  TQ_OBJECT
public:
    KoDocumentInfo( TQObject* parent = 0, const char* name = 0 );
    virtual ~KoDocumentInfo();

    bool load( const TQDomDocument& doc );
    bool loadOasis( const TQDomDocument& metaDoc );

    TQDomDocument save();
    bool saveOasis( KoStore* store );

    /**
     * This info has an accessor because it's the most commonly used.
     * Equivalent to page("about")->title() (but checking that the page exists)
     */
    TQString title() const;
    TQString creator() const;

    KoDocumentInfoPage* page( const TQString& name ) const;
    TQStringList pages() const;
    void documentInfochanged() { emit sigDocumentInfoModifed();}
 signals:
    void sigDocumentInfoModifed();
};

class KOFFICECORE_EXPORT KoDocumentInfoPage : public TQObject
{
public:
    KoDocumentInfoPage( TQObject *parent, const char* name );

    virtual bool load( const TQDomElement& e ) = 0;
    virtual TQDomElement save( TQDomDocument& doc ) = 0;
    virtual bool loadOasis( const TQDomNode& metaDoc ) = 0;
    virtual bool saveOasis( KoXmlWriter &xmlWriter ) = 0;
};

class KOFFICECORE_EXPORT KoDocumentInfoAuthor : public KoDocumentInfoPage
{
    Q_OBJECT
  TQ_OBJECT
public:
    KoDocumentInfoAuthor( KoDocumentInfo* info );
    ~KoDocumentInfoAuthor();

    virtual bool load( const TQDomElement& e );
    virtual TQDomElement save( TQDomDocument& doc );
    virtual bool loadOasis( const TQDomNode& metaDoc );
    virtual bool saveOasis( KoXmlWriter &xmlWriter );

    TQString fullName() const;
    TQString initial() const;
    TQString title() const;
    TQString company() const;
    TQString email() const;
    TQString telephoneHome() const;
    TQString telephoneWork() const;
    TQString fax() const;
    TQString country() const;
    TQString postalCode() const;
    TQString city() const;
    TQString street() const;
    TQString position() const;

    void setFullName( const TQString& n );
    void setTitle( const TQString& n );
    void setCompany( const TQString& n );
    void setEmail( const TQString& n );
    void setTelephoneHome( const TQString& n );
    void setTelephoneWork( const TQString& n );
    void setFax( const TQString& n );
    void setCountry( const TQString& n );
    void setPostalCode( const TQString& n );
    void setCity( const TQString& n );
    void setStreet( const TQString& n );
    void setInitial( const TQString& n );
    void setPosition( const TQString& n );
    void initParameters();

private:
    TQString m_fullName;
    TQString m_title;
    TQString m_company;
    TQString m_email;
    TQString m_telephoneHome;
    TQString m_telephoneWork;
    TQString m_fax;
    TQString m_country;
    TQString m_postalCode;
    TQString m_city;
    TQString m_street;
    TQString m_initial;
    TQString m_position;
    KConfig *m_emailCfg;
};

class KOFFICECORE_EXPORT KoDocumentInfoAbout : public KoDocumentInfoPage
{
    Q_OBJECT
  TQ_OBJECT
public:
    KoDocumentInfoAbout( KoDocumentInfo* info );

    virtual bool load( const TQDomElement& e );
    virtual TQDomElement save( TQDomDocument& doc );
    virtual bool loadOasis( const TQDomNode& metaDoc );
    virtual bool saveOasis( KoXmlWriter &xmlWriter );

    TQString title() const;
    TQString abstract() const;
    TQString subject() const;
    TQString keywords() const;
    TQString initialCreator() const;
    TQString editingCycles() const;
    TQString creationDate() const;
    TQString modificationDate() const;

    void setKeywords( const TQString &n );
    void setSubject( const TQString& n );

    void setTitle( const TQString& n );
    void setAbstract( const TQString& n );
    void saveParameters();

public slots:
    void resetMetaData();

private:
    KoDocumentInfo* m_docInfo;
    TQString m_title;
    TQString m_abstract;
    TQString m_keywords;
    TQString m_subject;
    TQString m_initialCreator;
    TQDateTime m_creationDate;
    TQDateTime m_modificationDate;
    int m_editingCycles;
    bool m_firstSave;
};

class KOFFICECORE_EXPORT KoDocumentInfoUserMetadata : public KoDocumentInfoPage
{
    Q_OBJECT
  TQ_OBJECT
public:
    KoDocumentInfoUserMetadata( KoDocumentInfo* info );

    virtual bool load( const TQDomElement& e );
    virtual TQDomElement save( TQDomDocument& doc );
    virtual bool loadOasis( const TQDomNode& metaDoc );
    virtual bool saveOasis( KoXmlWriter &xmlWriter );

    TQMap<TQString, TQString>* metadataList() { return &m_metaList; }

private:
    TQMap<TQString, TQString>  m_metaList;
    TQStringList             m_reserved;
};

#endif
